/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef  VEC_FROM_LOOK_ALIGN_H
#define  VEC_FROM_LOOK_ALIGN_H

/** Linearize info from an alignment into a vec in sync with the first vector.

\file VecFromLookAlign.h

*/

#include "CoreTools.h"
#include "Basevector.h"
#include "AlignErr.h"
#include "lookup/LookAlign.h"

/// Fill readIncorr with appropriate values: correct, ins, del, etc...
/// The vector readIncorr is the same length and orientation as bases.
/// If the alignment does not cover the base, we fill in NO_BASE
/// If there is a deletion, we assign it randomly to the base before or
/// after, and output its position to incorStreamPtr if it is not null.
/// The vector delCount contains the total number of deletions after 
/// each base, where "after" is counted in the direction of the alignment.

void VecFromLookAlign(vec<unsigned char> & readIncorr, 
			 vec<int> & delCount,
			 const basevector & bases, 
			 const basevector & ref, 
			 const look_align & la,
			 ostream * incorStreamPtr = 0);

/// Fill readIncorr with appropriate values: correct, ins, del, etc...
/// If the alignment does not cover the base, we fill in NO_BASE
/// If there is a deletion, we assign it randomly to the base before or
/// after, and output its position to incorStreamPtr if it is not null.
/// Delegates to the const basevector & ref version.
inline void VecFromLookAlign(vec<unsigned char> & readIncorr,
			 vec<int> & delCount,
			 const basevector & bases, 
			 const vecbasevector & ref, 
			 const look_align & la,
			 ostream * incorStreamPtr = 0) {
  VecFromLookAlign(readIncorr, delCount, bases, ref[la.target_id], la, 
		      incorStreamPtr);
}

/// Compress the deletion information into readIncorr, setting the deletion
/// randomly onto the base before or after.
void SetDeletionsRandomly(vec<unsigned char> & readIncorr, 
			  const vec<int> & delCount);

/// Compress the deletion information into readIncorr, setting the deletion
/// onto the base before or after with the worst quality score.
void SetDeletionsWithQuals(vec<unsigned char> & readIncorr, 
			   const vec<int> & delCount,
			   const qualvector & q);

/// Remove errors within EXCLUDE_ENDS of the end of the alignment.
/// We replace the values from the end to the error with NO_BASE.
/// To be precise, we require that there be EXCLUDE_ENDS good
/// bases in a row before we start accepting the alignment. All
/// bases in the EXCLUDE_ENDS length are considered good and accepted.
/// This is meant to be used after one of the SetDeletions methods.
void RemoveErrorsAtEnds(vec<unsigned char> & readIncorr, int EXCLUDE_ENDS);

/// Return true if alignment is perfect for read bases [0,last).
inline bool PerfectToBase(const look_align & la, int last, 
			  const basevector & read, 
			  const basevector & ref) {
  vec<unsigned char> incorr;
  vec<int> del;
  VecFromLookAlign(incorr, del, read, ref, la);
  for (int i=0; i != last; ++i) {
    if (incorr[i] != AlignErr::MATCH || del[i]) return false;
  }
  return true;
}

#endif // VEC_FROM_LOOK_ALIGN_H
