/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "lookup/VecFromLookAlign.h"
#include "assembly/BoundAlignment.h"
#include "assembly/BasicSequence.h"
#include "PrintAlignment.h"

void VecFromLookAlign(vec<unsigned char> & readIncorr, 
			 vec<int> & delCount,
			 const basevector & bases, 
			 const basevector & ref, 
			 const look_align & la,
			 ostream * incorStreamPtr) {
  const int BSIZE = bases.size();
  readIncorr.assign(BSIZE, AlignErr::NO_BASE);  
  delCount.assign(BSIZE, 0);  

  BasicBoundAlignment ba = BasicBoundAlignment
    (BasicSequence(&bases, la.query_id),
     (la.rc1 ? orient_RC : orient_FW), 
     BasicSequence(&ref, la.target_id), 
     orient_FW,
     la.a);
  for ( BasicBoundAlignment::Iterator i(&ba); !i.IsAtEnd(); ++i) {
    int readOffset = (la.rc1
		      ? BSIZE - i.GetOffsetOnFirst() - 1
		      : i.GetOffsetOnFirst());
    AssertLt(readOffset, BSIZE);

    if (i.IsMatch())
      readIncorr[readOffset] = AlignErr::MATCH;
    else if (i.IsGapOnSecond())    
      readIncorr[readOffset] = AlignErr::INSERTION;
    else if (i.IsGapOnFirst()) {
      ++delCount[readOffset];
    }
    else readIncorr[readOffset] = AlignErr::MISMATCH;
  }

  if (incorStreamPtr) {
    ostream & incorStream = *incorStreamPtr;
    basevector btemp = bases;
    if (la.rc1) btemp.ReverseComplement();

    PrintVisualAlignment(False, incorStream, btemp, ref,la.a);
    readIncorr.Println(incorStream,"");
    delCount.Println(incorStream,"");
  }
}


void SetDeletionsRandomly(vec<unsigned char> & readIncorr, 
			  const vec<int> & delCount) {
  for (int i=0; i != readIncorr.isize(); ++i) {
    if (delCount[i]) {
      if (readIncorr.isize() -1 == i) readIncorr[i] = AlignErr::DELETION;
      else {
	int pos = drand48() < 0.5 ? i : i+1;
	readIncorr[pos] = AlignErr::DELETION;
      }
    }
  }
}

void SetDeletionsWithQuals(vec<unsigned char> & readIncorr, 
			   const vec<int> & delCount,
			   const qualvector & q) {
  for (int i=0; i != readIncorr.isize(); ++i) {
    if (delCount[i]) {
      if (readIncorr.isize() -1 == i) readIncorr[i] = AlignErr::DELETION;
      else {
	int pos = q[i] < q[i+1] ? i : i+1;
	readIncorr[pos] = AlignErr::DELETION;
      }
    }
  }
}
  
void RemoveErrorsAtEnds(vec<unsigned char> & readIncorr, int EXCLUDE_ENDS) {
  int alignStart = 0;
  while (readIncorr[alignStart] == AlignErr::NO_BASE) ++alignStart;
  int alignEnd = readIncorr.size() -1;
  while (readIncorr[alignEnd] == AlignErr::NO_BASE) --alignEnd;
  //PRINT2(alignStart, alignEnd);
  int lastBad = alignStart-1;
  for (int i=alignStart; i != readIncorr.isize(); ++i) {
    if (AlignErr::MATCH != readIncorr[i]) lastBad = i;
    if (i - lastBad >= EXCLUDE_ENDS) break;
  }
  //PRINT(lastBad);
  for (int i=alignStart; i <= lastBad; ++i) readIncorr[i] = AlignErr::NO_BASE;

  lastBad = alignEnd+1;
  for (int i=alignEnd; i >= 0; --i) {
    if (AlignErr::MATCH != readIncorr[i]) lastBad = i;
    if (lastBad - i >= EXCLUDE_ENDS) break;
  }
  //PRINT(lastBad);
  for (int i=alignEnd; i >= lastBad; --i) readIncorr[i] = AlignErr::NO_BASE;
}

