#ifndef LOOKUP_HIT_H
#define LOOKUP_HIT_H
/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////
#include "CoreTools.h"
#include "lookup/KmerIndex.h"

/// \file Hit.h

/// A Query encodes a query position and its associated kmer.  It fits
/// in 64 bits and works for any reasonable query (length up to 2^31).
struct Query {
  unsigned int k; 
  int query_pos; ///< negative query_pos values are used to encode rc queries
  Query(int in_query_pos = 0, unsigned int k = 0, bool rc = false) :
    k(k),
    query_pos(rc ? -in_query_pos -1 : in_query_pos)
  { AssertGe(in_query_pos, 0); }
  bool IsRc() const { return query_pos<0; }
  unsigned int Kmer() const { return k; }
  int QueryPos() const { return (query_pos>=0) ? query_pos : -(1+query_pos); }
};

inline ostream & operator<<(ostream &out, const Query &h)
{
  return out << h.QueryPos() << (h.IsRc() ? "/" : "\\") << h.Kmer();
}

#define RETURN_IF_UNEQUAL(expr1, expr2) \
{ if ((expr1)<(expr2)) return true; if ((expr2)<(expr1)) return false; }

struct CompareQueriesByPos {
  bool operator()(const Query &a, const Query &b) {
    return (a.QueryPos() < b.QueryPos());
  }
};

struct CompareQueriesByKmer {
  bool operator()(const Query &a, const Query &b) {
    return (a.Kmer() < b.Kmer());
  }
};

/// Transform a single basevector in queries.  
void BasesToQueries(const basevector &bases, vec<Query> &queries, unsigned int K);

/// Transform a vecbasevector into queries.  The query positions are
/// derived from the concatenation of all the basevectors, which
/// should therefore be less than 2^31 bases in total length.
void BasesToQueries(const vecbasevector &bases, vec<Query> &queries, unsigned int K);

/// A RawHit encodes a position at which a query kmer matches a target
/// kmer.  It fits in 64 bits and works for any reasonable query:
/// query length is up to 2^31.  The offset is relative to a
/// particular LookupTable.
struct RawHit {
  unsigned int offset; 
  int query_pos; ///< negative query_pos values are used to encode rc hits
  RawHit(unsigned int offset = 0, int in_query_pos = 0, bool rc = false) :
    offset(offset),
    query_pos(rc ? -in_query_pos -1 : in_query_pos)
  { AssertGe(in_query_pos, 0); }
  bool IsRc() const { return query_pos<0; }
  unsigned int Offset() const { return offset; }
  int QueryPos() const { return (query_pos>=0) ? query_pos : -(1+query_pos); }
  longlong QueryStartOnTarget() const { return longlong(Offset()) - longlong(QueryPos()); }
};

inline ostream & operator<<(ostream &out, const RawHit &h)
{
  return out << h.QueryPos() << (h.IsRc() ? ":" : "=") << h.Offset();
}

struct RawHitIsFw {
  bool operator()(const RawHit &a) {
    return !a.IsRc();
  }
};

struct CompareRawHitsByOffset {
  bool operator()(const RawHit &a, const RawHit &b) {
    return (a.Offset() < b.Offset());
  }
};

struct RawHitOffsetIsBefore {
  RawHit b;
  RawHitOffsetIsBefore(RawHit b) : b(b) { } 
  bool operator()(const RawHit &a) {
    return (a.Offset() < b.Offset());
  }
};

struct CompareRawHitsByQueryStart {
  bool operator()(const RawHit &a, const RawHit &b) {
    return (a.QueryStartOnTarget() < b.QueryStartOnTarget());
  }
};

struct CompareRawHitsByQueryStartOffset {
  bool operator()(const RawHit &a, const RawHit &b) {
    RETURN_IF_UNEQUAL(a.QueryStartOnTarget(), b.QueryStartOnTarget());
    return a.Offset() < b.Offset();
  }
};


/// A ProcessedHit adds to a RawHit some additional information: how
/// many times did this offset occur?  What contig of target does it
/// correspond to?  What is the range of QueryStartOnTarget values
/// being encoded by this object? Fits in 128 bits.

struct ProcessedHit : public RawHit {
  unsigned short nhits;
  unsigned short bw;
  unsigned int contig;
  ProcessedHit(unsigned int offset = 0, unsigned int contig = 0,
	       int in_query_pos = 0, bool rc = false, unsigned int in_nhits = 0) :
    RawHit(offset, in_query_pos, rc),
    nhits(min(in_nhits, static_cast<unsigned int>(numeric_limits<unsigned short>::max()))),
    bw(0),
    contig(contig)
  { }
  unsigned int TargetContig() const { return contig; }
  unsigned short NHits() const { return nhits; }
  unsigned short Bandwidth() const { return bw; }
  void SetNHits(unsigned int n) { nhits = min(n, static_cast<unsigned int>(numeric_limits<unsigned short>::max())); }
  void SetBandwidth(unsigned int b) { bw = min(b, static_cast<unsigned int>(numeric_limits<unsigned short>::max())); }
};

inline ostream & operator<<(ostream &out, const ProcessedHit &h)
{
  return out << static_cast<RawHit>(h) << "(" << h.TargetContig() << ")["
	     << h.NHits() << "," << h.Bandwidth() << "]";
}

struct CompareProcessedHitsByRcContigQueryStart {
  bool operator()(const ProcessedHit &a, const ProcessedHit &b) {
    RETURN_IF_UNEQUAL(a.IsRc(), b.IsRc());
    RETURN_IF_UNEQUAL(a.TargetContig(), b.TargetContig());
    return (a.QueryStartOnTarget() < b.QueryStartOnTarget());
  }
};

// Note that a standard ascending sort puts better hits at the
// beginning
struct CompareProcessedHitsByQuality {
  bool operator()(const ProcessedHit &a, const ProcessedHit &b) {
    return a.NHits() > b.NHits();
  }
};

template <typename It>
unsigned int totalHits(It first, It last)
{
  unsigned int h = 0;
  for ( ; first!=last; ++first) {
    h += first->NHits();
  }
  return h;
}

// Prepare hits for alignment with gaps by grouping nearby hits and
// forming clusters when the hits are close enough together.  We also
// sort the vector to put the most promising hits first, to improve
// efficiency of rejecting bad alignments.
class ClusterHits {
  int bw;
public:
  ClusterHits(int bw) : bw(bw) { }
  // Use this version to increase bandwidth of ProcessedHits clusters
  void operator()(serfvec<ProcessedHit> &hits);
  int bandwidth() const { return bw; }
};



#undef RETURN_IF_UNEQUAL

#endif
