// Copyright (c) 2003 Whitehead Institute for Biomedical Research
// 

#ifndef FORCE_DEBUG
#define NDEBUG
#endif

#include "assembly/VecDataManager.h"

template <class T>
void
VecDataManager<T>::LoadAllData()
{
    if ( ! mbDataLoaded )
    {
        Assert( IsAsciiVec( mStrVecFile ) );

        READX( mStrVecFile, mVecData );
        mbDataLoaded = true;
    }
}

template <class T>
void
VecDataManager<T>::ResizeToFit( const int id )
{
    unsigned int targetSize = id + 1;
    
    if ( mVecData.size() < targetSize )
    {
        unsigned int targetCapacity = mVecData.capacity();
        
        while ( targetCapacity < targetSize )
        {
            targetCapacity = (targetCapacity+1) * 2;
        }
        
        mVecData.reserve( targetCapacity );
        mVecData.resize( targetSize );
    }
}

template <class T>
void
VecDataManager<T>::Write( const bool bOverwrite,
                          const String &strOutFile )
{
    if ( ! mbModified && 
         IsRegularFile(strOutFile) && 
         IsRegularFile(mStrVecFile) &&
         RealPath(strOutFile) == RealPath(mStrVecFile) )
        return;

    ForceAssert( bOverwrite || ! mbModified || ! IsRegularFile( strOutFile ) );

    this->LoadAllData();

    WRITE( strOutFile, mVecData );
}


// specialize LoadAllData for trims.
template <>
void
VecDataManager< pair<int,int> >::LoadAllData( )
{
    if ( ! mbDataLoaded )
    {
        int nLines = LineCount( mStrVecFile );
        mVecData.reserve( nLines );
        
        Ifstream( trimsStream, mStrVecFile );
        
        for (int lineNum=0; lineNum<nLines; ++lineNum) 
        {
            int leftAmount;
            int rightAmount;
            
            trimsStream >> leftAmount >> rightAmount;
            mVecData.push_back( make_pair( leftAmount, rightAmount ) );
        }
    
        trimsStream.close( );
        mbDataLoaded = true;
    }
}    

// specialize Write for trims.
template <>
void
VecDataManager< pair<int,int> >::Write( const bool bOverwrite,
                                        const String &strOutFile )
{
    if ( ! mbModified && 
         IsRegularFile(strOutFile) && 
         IsRegularFile(mStrVecFile) &&
         RealPath(strOutFile) == RealPath(mStrVecFile) )
        return;

    ForceAssert( bOverwrite || ! IsRegularFile( strOutFile ) );

    this->LoadAllData();

    Ofstream( trimsStream, strOutFile );

    int nLines = mVecData.size();
        
    for (int lineNum=0; lineNum<nLines; ++lineNum) 
    {
        pair<int,int> theTrims = mVecData[lineNum];
        
        trimsStream << theTrims.first << ' ' << theTrims.second << '\n';
    }

    trimsStream.close( );
}

template class VecDataManager<int>;
template class VecDataManager<String>;
template class VecDataManager< pair<int,int> >;
