/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/// Wraps an STL vector, adds asserts for debugging and useful methods.
/// \class Vec
/// Vec.h defines class vec, which wraps the STL class vector, in such a way
/// that if compiled with NDEBUG on, it is (or should be) the same as a vector
/// (with some added functionality -- see below), but otherwise does run-time 
/// checking of each vector reference to make sure it is in range.
///
/// In debug mode, resize and reserve will fail if you (in effect) ask for more
/// than 100GB (or 1.5 GB on 32-bit systems).

#ifndef VEC_H
#define VEC_H

#include <unistd.h>

#include <algorithm>
#include <fstream>
#include <iomanip>
#include <iterator>
#include <memory>
#include <strstream>
#include <vector>
#include <functional>

#include "system/Assert.h"
#include "String.h"
#include "system/System.h"
#include "system/Types.h"

// Size checker: identity operation unless it FatalErr's and never returns.
template <class T>
inline size_type ValidatedSize(ulonglong i)
{
  //  PRINT(i);
#ifndef NDEBUG
  static const ulonglong TOO_BIG = (4 == sizeof(char *)) 
    ? (ulonglong)(INT_MAX) // 2 GB
    : ULLCONST(100 * 1000) * ULLCONST(1000 * 1000); // ~100GB

  if ( ulonglong(i) * ulonglong(sizeof(T)) > TOO_BIG ) {
    FatalErr( "Attempt to resize vec<T> object to " << i 
	      << ", where sizeof(T) = " << sizeof(T) << "."
	      << "\nTOO_BIG=" << TOO_BIG 
	      << ", request=" << i << " T's" );
  }
#endif
  return i;
}

template <class T> class vec : public vector<T> {

 public:

// ===========================================================================
//
// CONSTRUCTORS
//
// ===========================================================================

  vec()            : vector< T >() {}
  
  explicit vec(ulonglong n) : vector< T >(ValidatedSize<T>(n)) {}
  
  vec(ulonglong n, const T& value) : vector< T >( ValidatedSize<T>(n), value ) {}

  enum ConstructorBehavior { IDENTITY };

  vec( ulonglong n, const ConstructorBehavior constructor_type  ) : 
    vector<T>(ValidatedSize<T>(n))
  {    ForceAssert( constructor_type == IDENTITY );
       for ( size_type i = 0; i < n; i++ )
            (*this)[i] = i;    }

  vec(const vector<T>& v) : vector<T>( v ) {}

  // vec<double> can be constructed e.g. from "{1.3,5.2,9}".

  explicit vec( const String& s );

  template<class ForwardIterator>
  vec(ForwardIterator first, ForwardIterator last) : 
    vector<T>( first, last ) {}

  ///Asserts index within bounds.
  typename vector<T>::reference operator[]( size_type i ) {
    AssertLt( i,  vector<T>::size() ); // Asserts index within bounds
    return vector<T>::operator[](i);   // ... and returns the element
  }
  
  ///Asserts index within bounds.
  typename vector<T>::const_reference operator[](size_type i) const {
    AssertLt( i, vector<T>::size() );  // Asserts index within bounds
    return vector<T>::operator[](i);   // ... and returns the element
  }

  void resize( ulonglong i, T c = T( ) ) { 
    vector<T>::resize( ValidatedSize<T>(i), c ); 
  }

  void reserve( ulonglong i ) { vector<T>::reserve(ValidatedSize<T>(i)); }

  typename vector<T>::reference front( ) {
    AssertGt( vector<T>::size( ), 0 ); // Asserts index within bounds
    return vector<T>::front( );   // ... and returns the element
  }

  typename vector<T>::const_reference front( ) const {
    AssertGt( vector<T>::size( ), 0 ); // Asserts index within bounds
    return vector<T>::front( );   // ... and returns the element
  }

  typename vector<T>::reference back( ) {
    AssertGt( vector<T>::size( ), 0 ); // Asserts index within bounds
    return vector<T>::back( );   // ... and returns the element
  }

  typename vector<T>::const_reference back( ) const {
    AssertGt( vector<T>::size( ), 0 ); // Asserts index within bounds
    return vector<T>::back( );   // ... and returns the element
  }

// ===========================================================================
//
// FUNCTIONS TO PUSH ELEMENTS ONTO VECTORS
//
// ===========================================================================

  /// push_front (insert item before first element of vector)

  void push_front( const T& t1 )
  {    vector<T>::insert( vector<T>::begin( ), t1 );    }

  /// Generalized push_back, allowing up to 12 items pushed back at a time.

  void push_back( const T& t1 )
  {    vector<T>::push_back(t1);    }
  void push_back( const T& t1, const T& t2 )
  {    vector<T>::push_back(t1); vector<T>::push_back(t2);    }
  void push_back( const T& t1, const T& t2, const T& t3 )
  {    vector<T>::push_back(t1); vector<T>::push_back(t2);
       vector<T>::push_back(t3);    }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4 )
  {    vector<T>::push_back(t1); vector<T>::push_back(t2);
       vector<T>::push_back(t3); vector<T>::push_back(t4);    }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5 )
  {    vector<T>::push_back(t1); vector<T>::push_back(t2);
       vector<T>::push_back(t3); vector<T>::push_back(t4);
       vector<T>::push_back(t5);    }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7, const T& t8 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7, t8 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7, const T& t8, const T& t9 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7, t8, t9 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7, const T& t8, const T& t9, const T& t10 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7, t8, t9, t10 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7, const T& t8, const T& t9, const T& t10,
       const T& t11 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7, t8, t9, t10, t11 );     }
  void push_back( const T& t1, const T& t2, const T& t3, const T& t4, const T& t5,
       const T& t6, const T& t7, const T& t8, const T& t9, const T& t10,
       const T& t11, const T& t12 )
  {    push_back( t1, t2, t3, t4, t5 );
       push_back( t6, t7, t8, t9, t10, t11, t12 );     }

  // push: construct object from up to ten arbitrary arguments, then push it back.

  template<class X1>
  void push( const X1& x1 )
  {    vector<T>::push_back( T(x1) );    }
  template<class X1, class X2>
  void push( const X1& x1, const X2& x2 )
  {    vector<T>::push_back( T(x1, x2) );    }
  template<class X1, class X2, class X3>
  void push( const X1& x1, const X2& x2, const X3& x3 )
  {    vector<T>::push_back( T(x1, x2, x3) );    }
  template<class X1, class X2, class X3, class X4>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4 )
  {    vector<T>::push_back( T(x1, x2, x3, x4) );    }
  template<class X1, class X2, class X3, class X4, class X5>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6, x7) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7,
       class X8>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7, const X8& x8 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6, x7, x8) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7,
       class X8, class X9>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7, const X8& x8, const X9& x9 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6, x7, x8, x9) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7,
       class X8, class X9, class X10>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7, const X8& x8, const X9& x9, const X10& x10 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6, x7, x8, x9, x10) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7,
       class X8, class X9, class X10, class X11>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7, const X8& x8, const X9& x9, const X10& x10,
       const X11& x11 )
  {    vector<T>::push_back( T(x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11) );    }
  template<class X1, class X2, class X3, class X4, class X5, class X6, class X7,
       class X8, class X9, class X10, class X11, class X12>
  void push( const X1& x1, const X2& x2, const X3& x3, const X4& x4, const X5& x5,
       const X6& x6, const X7& x7, const X8& x8, const X9& x9, const X10& x10,
       const X11& x11, const X12& x12 )
  {    vector<T>::push_back( 
            T(x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12) );    }

  void push_back_copies( const T& t1, const int n )
  {    for ( int i = 0; i < n; i++ )
            vector<T>::push_back(t1);    }
  
  void append( const vec<T>& y ) 
  {    insert( this->end( ), y.begin( ), y.end( ) );    }
  
  void append( const vec<T>& y, unsigned int i, unsigned int j ) {
    if ( j == y.size( ) ) insert( this->end( ), y.begin( ) + i, y.end( ) );
    else insert( this->end( ), y.begin( ) + i, y.begin( ) + j );   
  }

// ===========================================================================
//
// FUNCTIONS TO TEST IF VECTOR HAS AN ATTRIBUTE
//
// ===========================================================================

  Bool nonempty( ) const { return ! this->empty( ); }
  Bool solo( ) const { return this->size( ) == 1; }

  Bool Ordered( ) const
  {    for ( unsigned int i = 1; i < this->size( ); i++ )
            if ( (*this)[i] < (*this)[i-1] ) return False;
       return True;    }

  Bool UniqueOrdered( ) const
  {    for ( unsigned int i = 1; i < this->size( ); i++ )
            if ( (*this)[i] <= (*this)[i-1] ) return False;
       return True;    }

// ===========================================================================
//
// FUNCTIONS TO EXTRACT PART OF A VECTOR
//
// ===========================================================================

  // SetToSubOf: Set *this to the n entries of x, starting at start.

  void SetToSubOf( const vec<T>& x, int start, int n )
  {    resize(n);
       for ( int i = 0; i < n; i++ )
            (*this)[i] = x[ start + i ];    }

  inline friend vec<T> SubOf( const vec<T>& x, int start, int n )
  {    vec<T> s(n);
       for ( int i = 0; i < n; i++ )
            s[i] = x[ start + i ];
       return s;    }

  void SetToRangeOf( const vec<T>& v, size_type i, size_type j )
  {    AssertLe( i, j );
       resize(0);
       reserve( j - i );
       for ( size_type x = i; x < j; x++ )
            push_back( v[x] );    }

  inline friend vec<T> RangeOf( const vec<T>& v, size_type i, size_type j )
  {    AssertLe( i, j );
       vec<T> r;
       r.reserve( j - i );
       for ( size_type x = i; x < j; x++ )
            r.push_back( v[x] );
       return r;    }

  int isize( ) const { return this->size( ); }

  void SetCat( const vec<T>& v1, const vec<T>& v2 )
  {    *this = v1;
       append(v2);    }

  void clear_and_resize( size_type n )
  {    this->clear( );
       resize(n);    }

  void resize_and_set( size_type n, const T& x )
  {    resize(n);
       for ( size_type i = 0; i < n; i++ )
            (*this)[i] = x;    }

  void SetToReverseOf( const vec<T>& v )
  {    resize( v.size( ) );
       for ( unsigned int i = 0; i < v.size( ); i++ )
            (*this)[i] = v[ v.size( ) - i - 1 ];    }

  void ReverseMe( )
  {    for ( unsigned int i = 0; i < this->size( )/2; i++ )
       {    T temp = (*this)[i];
            (*this)[i] = (*this)[ this->size( ) - i - 1 ];
            (*this)[ this->size( ) - i - 1 ] = temp;    }    }

  Bool Contains( const vec<T>& v ) const
  {    for ( int i = 0; i <= isize( ) - v.isize( ); i++ )
       {    int j;
            for ( j = 0; j < v.isize( ); j++ )
                 if ( (*this)[i+j] != v[j] ) break;
            if ( j == v.isize( ) ) return True;    }
       return False;    }

  Bool Contains( const vec<T>& v, const int pos ) const
  {    if ( pos >= 0 )
       {    if ( v.isize( ) + pos > isize( ) ) return False;    }
       else if ( v.isize( ) > isize( ) ) return False;
       int j, start = ( pos >= 0 ? pos : isize( ) - v.isize( ) );
       for ( j = 0; j < v.isize( ); j++ )
            if ( (*this)[start+j] != v[j] ) break;
       return j == v.isize( );    }

// ===========================================================================
//
// FUNCTIONS TO ERASE ELEMENTS FROM VECTORS - PART 1 (MEMBER FUNCTIONS)
//
// ===========================================================================

  /// Erase: erase range of elements, where range is given by half-open interval.

  void Erase( int start, int stop )
  {    erase( this->begin( ) + start, this->begin( ) + stop );    }

  /// EraseValue: erase all entries having the given value.

  void EraseValue( const T& x )
  {    vec<T>& v = *this;
       int count = 0;
       for ( int i = 0; i < v.isize( ); i++ )
       {    if ( v[i] != x )
            {    if ( i != count ) v[count] = v[i];
                 ++count;    }    }
       v.resize(count);    }


  /// print values to ostream, separated by sep.
  void Print(ostream & os, const char * sep = " ") const {
    copy(this->begin(), this->end(), ostream_iterator<T>(os, sep));
  }

  /// print values to ostream, separated by sep, with newline at end.
  void Println(ostream & os, const char * sep = " ") const {
    Print(os, sep); os << endl;
  }

  ///Set myself from text stream containing list of values of unknown length.
  void ReadFromTextStream(istream & is) {
    this->clear();
    T t;
    while (true) {
      is >> t;
      if (!is) break;
      push_back(t);
    }
  }

  /// CountValue: count all entries having the given value.

  int CountValue( const T& x ) const
  {    const vec<T>& v = *this;
       int count = 0;
       for ( int i = 0; i < v.isize( ); i++ )
            if ( v[i] == x ) ++count;
       return count;    }

// ===========================================================================
//
// MEMBER FUNCTIONS TO DO ARITHMETIC ON VECTORS.
//
// ===========================================================================

  /// Multiply a vector by a constant of type X.

  template<class X>
  vec & operator*=(const X & x) {
    const int S = this->size();
    for (int i = 0; i != S; ++i) {
      (*this)[i] = static_cast<T>( x * (*this)[i]);
    }
    return *this;
  }

  /// Divide a vector by a constant of type X.

  template<class X>
  vec & operator/=(const X & x) {
    return this->operator*=(1.0/x);
  }

  /// Add two vectors together.
  /// If vx is longer, add up to this vector's size.
  /// if vx is shorter, add up to vx's size only.

  template<class X> vec & operator+=(const vec<X> & vx) {
    const int S = min(this->size(), vx.size());
    for (int i = 0; i != S; ++i) {
      (*this)[i] += static_cast<T>(vx[i]);
    }
    return *this;
  }

  //stand-alone operators are implemented in terms of op=
  //See meyers, more effective C++, item 22 for reasons.

  /// Multiply a vector by a constant of type X.

  template<class X> friend vec operator*(const vec & v, const X & x) {
    return vec(v) *= x;
  }

  /// Multiply a vector by a constant of type X.

  template<class X> friend vec operator*(const X & x, const vec & v) {
    return vec(v) *= x;
  }
  /// Divide a vector by a constant of type X.

  template<class X> friend vec operator/(const vec & v, const X & x) {
    return vec(v) /= x;
  }

  /// Add two vectors together.
  /// If vx is longer, add up to this vector's size.
  /// if vx is shorter, add up to vx's size only.

  template<class X> friend vec operator+(const vec & v, const vec<X> & vx) {
    return vec(v) += vx;
  }

  /// NextDiff(i): return index of next element after i that is different from
  /// the ith element.

  inline int NextDiff( int i )
  {    int j;
       for ( j = i + 1; j < isize( ); j++ )
            if ( (*this)[j] != (*this)[i] ) break;
       return j;    }

};

template<class T> int SizeSum( const vec< vec<T> >& v )
{    int sum = 0;
     for ( int i = 0; i < v.isize( ); i++ )
          sum += v[i].size( );
     return sum;    }

template<class T> longlong SizeSumLong( const vec< vec<T> >& v )
{    longlong sum = 0;
     for ( int i = 0; i < v.isize( ); i++ )
          sum += v[i].size( );
     return sum;    }

/// mappedvec is a subclass of vec that has its contents mmap()ed in.
/// The file given must have been produced by BinaryWrite2().
///
/// These vecs must all be const, so they cannot be constructed
/// directly, but must be constructed via the BinaryMmap2() function,
/// provided below, which (more or less) returns a const pointer to a
/// mappedvec object.

// This implementation is fantastically delicate, as it depends
// heavily upon the precise implementation of std::vector, which can
// change at any time.

template <typename T>
class mappedvec;

template <typename T>
void BinaryMmap3( const String& filename, auto_ptr< const mappedvec<T> >& vp );

template <typename T>
class mappedvec : public vec<T> 
{
 private:
  mappedvec() : vec<T>() {}
  mappedvec( const mappedvec& ); // intentionally undefined

 public:
  friend
  void BinaryMmap3<>( const String& filename, auto_ptr< const mappedvec<T> >& vp );

  ~mappedvec();
};


template<class T> inline bool Member( const vector<T>& v, const T& x )
{    for ( unsigned int i = 0; i < v.size( ); i++ )
          if ( v[i] == x ) return true;
     return false;    }

/// Destroy returns the memory used by a vector, more or less following the 
/// instructions given in Stroustrup's book, The C++ Programming Language, Special 
/// Edition (2000), p. 457.

template<class T> inline void Destroy( vec<T>& v )
{    v.clear( );
     vec<T> tmp = v;
     v.swap(tmp);    }
template<class T1, class T2> inline void Destroy( vec<T1>& v1, vec<T2>& v2 )
{    Destroy(v1), Destroy(v2);    }
template<class T1, class T2, class T3> 
inline void Destroy( vec<T1>& v1, vec<T2>& v2, vec<T3>& v3 )
{    Destroy(v1), Destroy(v2), Destroy(v3);    }
template<class T1, class T2, class T3, class T4> 
inline void Destroy( vec<T1>& v1, vec<T2>& v2, vec<T3>& v3, vec<T4>& v4 )
{    Destroy(v1), Destroy(v2), Destroy(v3), Destroy(v4);    }
template<class T1, class T2, class T3, class T4, class T5> 
inline void Destroy( vec<T1>& v1, vec<T2>& v2, vec<T3>& v3, vec<T4>& v4, 
     vec<T5>& v5 )
{    Destroy(v1), Destroy(v2), Destroy(v3), Destroy(v4), Destroy(v5);    }

/// Return the position of an element in a vector, else -1.
template<class T> inline int Position( const vector<T>& v, const T& x )
{    for ( unsigned int i = 0; i < v.size( ); i++ )
          if ( v[i] == x ) return i;
     return -1;    }

/// BinPosition.  Return the position of an element in a sorted vector, else -1.
/// If the element appears more than once, the position of one of its instances
/// is returned.

template<class T> inline int BinPosition( const vector<T>& v, const T& x )
{    if ( v.size( ) == 0 ) return -1;
     int first = 0, last = (int) v.size( ) - 1, next;
     while (1)
     {    if (first == last) return ( !(x < v[last]) && !(v[last] < x) ) ? last : -1;
          next = first + (last - first) / 2;
          if ( x < v[next] ) last = next;
          else if ( v[next] < x ) first = next + 1;
          else return next;    }    }

template<class T> inline Bool BinMember( const vector<T>& v, const T& x )
{    return BinPosition( v, x ) >= 0;    }

/// BinSubset: determine if v is a subset of w; assumes w is sorted and that there
/// is no repetition.

template<class T> inline Bool BinSubset( const vector<T>& v, const vector<T>& w )
{    for ( int i = 0; i < (int) v.size( ); i++ )
          if ( !BinMember( w, v[i] ) ) return False;
     return True;    }

/// Determine if v is a subset of w; assumes that there is no repetition.
template<class T> inline Bool Subset( const vec<T>& v, const vec<T>& w )
{    for ( int i = 0; i < v.isize( ); i++ )
          if ( !Member( w, v[i] ) ) return False;
     return True;    }

inline bool Nonnegative( const vec<int>& v )
{    for ( unsigned int i = 0; i < v.size( ); i++ )
          if ( v[i] < 0 ) return false;
     return true;    }

template<class T> vector<T> Reverse( const vector<T>& v )
{    return vector<T>( v.rbegin( ), v.rend( ) );    }

template<class T> vec<T> Reverse( const vec<T>& v )
{    return vec<T>( v.rbegin( ), v.rend( ) );    }

template<class T> void ReverseThis( vec<T>& v )
{    for ( unsigned int i = 0; i < v.size( )/2; i++ )
          swap( v[i], v[ v.size( ) - i - 1 ] );    }

// ===========================================================================
//
// SORT FUNCTIONS AND THE LIKE
//
// ===========================================================================

template<class T> void Sort( vec<T>& v )
{    sort( v.begin( ), v.end( ) );    }

template<class T> void ReverseSort( vec<T>& v )
{    sort( v.rbegin( ), v.rend( ) );    }

/* 
// Old version. Two new versions below are faster.
template<class T> void UniqueSort( vec<T>& v )
{    sort( v.begin( ), v.end( ) );
     v.erase( unique( v.begin(), v.end() ), v.end() );    }

template <class T> void UniqueSort( vec<T> & v) 
{
  if ( elems.empty() ) return;
  sort(elems.begin(),elems.end());
  
  typename vec<T>::iterator iter = elems.begin();
  typename vec<T>::iterator iter_end = elems.end();
  typename vec<T>::iterator move_to = elems.begin();
  T last_unique_elem = *iter;

  for ( iter++, move_to++ ; iter != iter_end ; iter++ ) {
    if ( *iter != last_unique_elem ) {
      *(move_to++) = last_unique_elem = *iter;
    }
  }
  elems.erase(move_to,iter_end);
}
*/

/// Leaves only unique elements in a vector \c v; these elements
/// will be also sorted.
template <class T> void UniqueSort(vec<T> & v) {
  if ( v.empty() ) return;
  sort(v.begin(),v.end());
  
  typename vec<T>::iterator iter = v.begin();
  typename vec<T>::iterator iter_end = v.end();
  typename vec<T>::iterator last_unique_elem_iter = iter; 
  typename vec<T>::iterator move_to = v.begin();

  // we know that v is non-empty, so it's ok to increase iter
  // right away in the next line:
  for ( iter++, move_to++ ; iter != iter_end ; iter++ ) {
    // we follow the standard assumptions of STL here.
    // the latters assume that operator== is defined
    // (but not necessarily !=):
    if ( *iter == *last_unique_elem_iter ) continue;
    *(move_to++) = *(last_unique_elem_iter = iter);
  }
  v.erase(move_to,iter_end);
}


template<class T> void RandomShuffle( vec<T>& v )
{    random_shuffle( v.begin( ), v.end( ) );    }

// SortSync( vec& v, vec& w ): sort v, moving the elements of w synchronously.

//forward declare for use in SortSync
template<class V, typename C> 
void WhatPermutation( const V & v, vec<int>& perm, C comparator, bool inv = true );

template<class S, class T, typename F > 
void SortSync( vec<S>& v, vec<T>& w, F comparator ) {
  ForceAssertEq( v.size( ), w.size( ) );
  vec<int> perm;
  WhatPermutation<vec<S>,F>(v, perm, comparator);
  PermuteVec(v, perm);
  PermuteVec(w, perm);
}

template<class S, class T> 
void SortSync( vec<S>& v, vec<T>& w ) {
  SortSync( v, w, less<S>() );
}

template<class S, class T> 
void ReverseSortSync( vec<S>& v, vec<T>& w ) {
  SortSync( v, w, greater<S>() );
}

template<class S, class T> 
void UniqueSortSync( vec<S>& v, vec<T>& w ) {
  SortSync( v, w, less<S>() );
  int count = 0;
  for ( int i = 0; i < v.isize( ); i++ )
  {    if ( i > 0 && v[i] == v[i-1] ) continue;
       if ( count != i )
       {    v[count] = v[i];
            w[count] = w[i];    }
       ++count;    }
  v.resize(count), w.resize(count);
}

// SortSync( v, w, x )-type functions.

template<class S, class T, class U, typename F > 
void SortSync( vec<S>& v, vec<T>& w, vec<U>& x, F comparator ) 
{    ForceAssertEq( v.size( ), w.size( ) );
     ForceAssertEq( v.size( ), x.size( ) );
     vec<int> perm;
     WhatPermutation<vec<S>,F>(v, perm, comparator);
     PermuteVec(v, perm), PermuteVec(w, perm), PermuteVec(x, perm);    }
template<class S, class T, class U> 
void SortSync( vec<S>& v, vec<T>& w, vec<U>& x ) 
{    SortSync( v, w, x, less<S>() );    }
template<class S, class T, class U> 
void ReverseSortSync( vec<S>& v, vec<T>& w, vec<U>& x ) 
{    SortSync( v, w, x, greater<S>() );    }

// SortSync( v, w, x, y )-type functions.

template<class S, class T, class U, class V, typename F > 
void SortSync( vec<S>& v, vec<T>& w, vec<U>& x, vec<V>& y, F comparator ) 
{    ForceAssertEq( v.size( ), w.size( ) );
     ForceAssertEq( v.size( ), x.size( ) );
     ForceAssertEq( v.size( ), y.size( ) );
     vec<int> perm;
     WhatPermutation<vec<S>,F>(v, perm, comparator);
     PermuteVec(v, perm), PermuteVec(w, perm), PermuteVec(x, perm);
     PermuteVec(y, perm);    }
template<class S, class T, class U, class V> 
void SortSync( vec<S>& v, vec<T>& w, vec<U>& x, vec<V>& y ) 
{    SortSync( v, w, x, y, less<S>() );    }
template<class S, class T, class U, class V> 
void ReverseSortSync( vec<S>& v, vec<T>& w, vec<U>& x, vec<V>& y ) 
{    SortSync( v, w, x, y, greater<S>() );    }

// ===========================================================================
//
// FUNCTIONS TO FIND SHARED ELEMENTS
//
// ===========================================================================

/// Intersection: make pairs of all elements in the first vector that also
/// occur in the second (which can result in duplicates).  Put the pairs
/// in the result. Both vectors must be SORTED.  This version is meant
/// for situations where two objects may compare equal but contain different
/// information, and we want to preserve both copies of that information.
template<class T> 
void
Intersection( const vec<T>& v1, const vec<T>& v2,
              vec<pair<T,T> > & result )
{    
  result.clear();
  typename vec<T>::const_iterator v1iter = v1.begin();
  typename vec<T>::const_iterator v2iter = v2.begin();
  while ( v1iter != v1.end() && v2iter != v2.end() )
  {
    if ( *v1iter < *v2iter )
      ++v1iter;
    else if ( *v2iter < *v1iter )
      ++v2iter;
    else
      result.push_back( make_pair(*v1iter++,*v2iter) );
  }
}

/// Intersection: copy all the elements in the first vector that also
/// occur in the second (which can result in duplicates).  Both
/// vectors must be SORTED.  There are two versions: the first accepts
/// as an argument a container for the result, the second simply
/// returns the result.  The elements in the result are taken from the
/// first vector.
template<class T> 
void
Intersection( const vec<T>& v1, const vec<T>& v2,
              vec<T>& result )
{    
  result.clear();
  typename vec<T>::const_iterator v1iter = v1.begin();
  typename vec<T>::const_iterator v2iter = v2.begin();
  while ( v1iter != v1.end() && v2iter != v2.end() )
  {
    if ( *v1iter < *v2iter )
      ++v1iter;
    else if ( *v2iter < *v1iter )
      ++v2iter;
    else
      result.push_back( *v1iter++ );
  }
}

/// Intersection: copy all the elements in the first vector that also
/// occur in the second (which can result in duplicates).  Both
/// vectors must be SORTED.  Returns the result.  
/// The elements in the result are taken from the
/// first vector.
template<class T>
vec<T> 
Intersection( const vec<T>& v1, const vec<T>& v2 )
{    
  vec<T> w;
  Intersection( v1, v2, w ); 
  return w;    
}

/// Meet: determine if two SORTED vectors have an element in common.  

template<class T> Bool Meet( const vec<T>& v1, const vec<T>& v2 )
{
  typename vec<T>::const_iterator v1iter = v1.begin();
  typename vec<T>::const_iterator v2iter = v2.begin();
  while ( v1iter != v1.end() && v2iter != v2.end() )
  {
    if ( *v1iter < *v2iter )
      ++v1iter;
    else if ( *v2iter < *v1iter )
      ++v2iter;
    else
      return true;
  }
  return false;
}

// ===========================================================================
//
// FUNCTIONS TO ERASE ELEMENTS FROM VECTORS - PART 2 (NON-MEMBER FUNCTIONS)
//
// ===========================================================================

/// EraseIf: wrapper around erase-remove_if idiom.

template<class T> void EraseIf( vec<T>& v, bool (T::*f)( ) const )
{    v.erase( remove_if( v.begin( ), v.end( ), mem_fun_ref(f) ), v.end( ) );    }

/// Another version of EraseIf: erase v[x] if erase[x] = True.

template<class T> void EraseIf( vec<T>& v, const vec<Bool>& erase )
{    int count = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    if ( !erase[i] )
          {    if ( count != i ) v[count] = v[i];
               ++count;    }    }
     v.resize(count);    }

/// EraseTheseIndices: Erase some elements of a vector, as determined by a sorted
/// list of indices.  Not efficiently implemented.

template<class T> void EraseTheseIndices( vec<T>& v, const vec<int>& these )
{    int count = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    if ( !BinMember( these, i ) )
          {    if ( count != i ) v[count] = v[i];
               ++count;    }    }
     v.resize(count);    }

// ===========================================================================
//
// INPUT AND OUTPUT
//
// ===========================================================================

/// IsAsciiVec() returns whether or not filename is a saved vec.  It
/// does this by checking whether the first line of the file is an
/// ASCII representation of a number.  If it encounters a non-digit,
/// non-whitespace character before it finds a newline, it returns
/// false.  Otherwise, it returns true.

bool IsAsciiVec( const String &filename );

/// ***** The following is outmoded: please use BinaryRead2, *****
/// ***** etc. (below) except for backward compatibility.    *****
///
/// BinaryRead and BinaryWrite allow one to efficiently read and write a vector
/// of simple objects.  
/// WARNING: This will not work if members of the objects are discontiguous in 
/// memory, as can happen if members are forcibly aligned to word boundaries.
///

template<class T> void BinaryWrite( int fd, const vec<T>& v ) {    
  int n = v.size( );
  WriteBytes( fd, &n, sizeof(int) );
  if ( n > 0 ) WriteBytes( fd, &v[0], (longlong) sizeof(T) * (longlong) n );    
}

/// BinaryWriteComplex is a different version of BinaryWrite.  The intention 
/// is that it not be called directly from a .cc file, but instead that the 
/// relevant .h file define BinaryWrite for a given class to be 
/// BinaryWriteComplex.  Ditto for BinaryReadComplex.
template<class T> void BinaryWriteComplex( int fd, const vec<T>& v )
{    int n = v.size( );
     WriteBytes( fd, &n, sizeof(int) );
     for ( int i = 0; i < n; i++ )
          BinaryWrite( fd, v[i] );    }

template<class T> void BinaryRead( int fd, vec<T>& v ) {    
  int n;
  ReadBytes( fd, &n, sizeof(int) );
  v.resize(n);
  if ( n > 0 ) ReadBytes( fd, &v[0], (longlong) sizeof(T) * (longlong) n );    
}

template<class T> void BinaryReadComplex( int fd, vec<T>& v )
{    int n;
     ReadBytes( fd, &n, sizeof(int) );
     v.resize(n);
     for ( int i = 0; i < n; i++ )
          BinaryRead( fd, v[i] );    }

/// BinaryReadSubset: read selected entries from file written with BinaryWrite.
template<class T> void BinaryReadSubset( const String& filename, 
     const vec<int>& ids, vec<T>& v )
{    int fd = OpenForRead(filename);
     v.resize( ids.size( ) );
     for ( int i = 0; i < ids.isize( ); i++ )
     {    lseek( fd, sizeof(int) + ids[i] * sizeof(T), SEEK_SET );
          read( fd, &v[i], sizeof(T) );    }
     close(fd);    }

/// ***** The following is outmoded: please use BinaryRead2, *****
/// ***** etc. (below) except for backward compatibility.    *****

template<class T> void BinaryWrite0( const String& filename, const vec<T>& v )
{    int fd = OpenForWrite(filename);
     longlong n = v.size( );
     String length = ToString(n) + "\n";
     int k = length.size( );
     WriteBytes( fd, length.c_str( ), k );
     if ( n > 0 ) WriteBytes( fd, &v[0], (longlong) sizeof(T) * (longlong) n );
     close(fd);    }

template<class T> void BinaryRead0( const String& filename, vec<T>& v )
{    String ns;
     {    Ifstream( in, filename );
          in >> ns;    }
     ForceAssert( ns.IsInt( ) );
     longlong n = ns.Int( );
     int fd = OpenForRead(filename);
     lseek( fd, ns.size( ) + 1, SEEK_SET );
     v.resize(n);
     if ( n > 0 ) ReadBytes( fd, &v[0], (longlong) sizeof(T) * (longlong) n );
     close(fd);    }

template<class T> void BinaryReadSubset0( const String& filename, 
     const vec<int>& ids, vec<T>& v )
{    String ns;
     {    Ifstream( in, filename );
          in >> ns;    }
     ForceAssert( ns.IsInt( ) );
     longlong n = ns.Int( );
     int k = ns.size( );
     int fd = OpenForRead(filename);
     v.resize( ids.size( ) );
     for ( int i = 0; i < ids.isize( ); i++ )
     {    ForceAssertGe( ids[i], 0 );
          ForceAssertLt( ids[i], n );
          lseek( fd, k + 1 + ids[i] * sizeof(T), SEEK_SET );
          read( fd, &v[i], sizeof(T) );    }
     close(fd);    }

//=============================================================================
//=============================================================================

/// BinaryRead2, BinaryWrite2, BinaryReadSubset2: same as above, but smarter:
///
/// - first 34 bytes = "binary format 2, header = 3 lines\n";
/// - next  13 bytes = number of entries, as ascii, left justified and blank-padded;
/// - next  15 bytes = "\nlittle endian\n" or "\nbig endian   \n";
/// - the objects.
///
/// This format should NOT be modified.  If we ever have need to modify it, we 
/// should create BinaryRead3, etc.
///
/// BREAD2 and BREADX2 mirror READ and READX in System.h.

template<class T> void BinaryWrite2( const String& filename, const vec<T>& v );
template<class T> void BinaryRead2( const String& filename, vec<T>& v, bool strict = false );
template<class T> void BinaryReadSubset2( const String& filename,
     const vec<int>& ids, vec<T>& v, Bool append = False, bool strict = false );
template<class T> void BinaryReadRange2( const String& filename,
     longlong from, longlong to, vec<T>& v, bool strict = false );
template<class T> longlong BinarySize2( const String& filename, bool strict = false );


///Append source to target.
void BinaryCat2(const String & target, const String & source);

/** Get the size of the data in a Binary2 file, return -1 if fail.
  That is, it will return 1 for a vec<char> file and 4 for a vec<int> file.
 Will return -1 but will not fail if the file is not in Binary2 format,
 so it can be used to test the format of a file.
*/
int GetBinary2ElementSize(const String & filename, bool strict = false);

#define BREAD2( FILE, TYPE, DATA )   \
     TYPE DATA;                      \
     BinaryRead2( FILE, DATA );

#define BREADX2( FILE, DATA )    \
     BinaryRead2( FILE, DATA );

//=============================================================================
//=============================================================================

/// BinaryRead3, BinaryWrite3, BinaryReadSubset3: same as above, but even smarter:
///
/// - first 34 bytes = "binary format 3, header = 4 lines\n";
/// - next  13 bytes = number of entries, as ascii, left justified and blank-padded;
/// - next  15 bytes = "\nlittle endian\n" or "\nbig endian   \n";
/// - next  34 bytes = "padding to make long word aligned\n";
/// - the objects.
///
/// This format should NOT be modified.  If we ever have need to modify it, we 
/// should create BinaryRead4, etc.
///
/// BREAD3 and BREADX3 mirror READ and READX in System.h.

template<class T> void BinaryWrite3( const String& filename, const vec<T>& v );
template<class T> void BinaryRead3( const String& filename, vec<T>& v, 
     bool strict = false, const Bool append = False );
template<class T> void BinaryReadSubset3( const String& filename,
     const vec<int>& ids, vec<T>& v, Bool append = False, bool strict = false );
template<class T> void BinaryReadRange3( const String& filename,
     longlong from, longlong to, vec<T>& v, bool strict = false );
template<class T> longlong BinarySize3( const String& filename, bool strict = false );
int GetBinary3ElementSize(const String & filename, bool strict = false );

template<class T> void BinaryMmap3( const String& filename, 
                                    auto_ptr< const mappedvec<T> >& pv );

///Append source to target.
void BinaryCat3(const String & target, const String & source);

#define BREAD3( FILE, TYPE, DATA )   \
     TYPE DATA;                      \
     BinaryRead3( FILE, DATA );

#define BREADX3( FILE, DATA )    \
     BinaryRead3( FILE, DATA );

// Macro to make it easy to use mappedvecs.  The auto_ptr is given a
// weird name to reduce the likelihood of a collision with another
// variable.
#define MakeMappedVec( FILE, T, DATA )   \
     auto_ptr< const mappedvec<T> > DATA ## _asdf_ptr; \
     BinaryMmap3( FILE, DATA ## _asdf_ptr ); \
     const mappedvec<T>& DATA = *DATA ## _asdf_ptr;


/// This class allows the progressive writing of a binary format 3
/// file.  Implemented in VecTemplate.h.
///
/// If you want a lot of these at the same time, set the optional arg
/// keep_open to false, so that you don't run out of file descriptors.

template <typename T> 
class Binary3Writer {
 public:
  Binary3Writer( ) : m_fd (-1) {}
  Binary3Writer( const String& filename, bool keep_open = true )
  { this->Open( filename, keep_open ); }
  ~Binary3Writer();

  void Write( const T& object );
  void WriteMultiple( const vec<T>& objects );

  void Open( const String& filename, bool keep_open = true );
  void Close();

  const String& Filename() { return m_filename; }
  // This will return the filename, even if the file has been Close()d.

 private:
  // Disallow copy constructor and assignment operator.  Not implemented.
  Binary3Writer( const Binary3Writer<T>& );
  Binary3Writer<T>& operator=( const Binary3Writer<T>& );

  int m_fd;           // used if m_keep_open
  String m_filename;  // used if ! m_keep_open
  longlong m_objectCount;
  bool m_keep_open;
};


/// \class Binary3Iter
/// This class lets you iterate through a binary format 3 vector file
/// stored on disk.  Use this if you just want to handle each element,
/// in order, in one pass through a loop.  This is more efficient in
/// terms of both time and memory (!) than loading the whole vector.
///
/// Usage for, eg, a vec<int>:
///
///   int s;
///   for( Binary3Iter<int> iter(filename, &s); iter.More(); iter.Next(&s) )
///
/// The constructor and each call to Next fill v with the next item.
/// Also, iter.N() = size of vector, and iter.I() = current index.

template <typename T>
class Binary3Iter {
public:
  Binary3Iter( const String& filename, T* p_to_fill, 
	       longlong max_memory = 128 * 1024 /* empirically good */ );
  ~Binary3Iter();

  void Next( T* p_to_fill );
  bool More() const { return m_globalIndex < m_globalSize; }
  longlong N() const { return m_globalSize; }
  longlong I() const { return m_globalIndex; }

private:
  void FillBuffer();

  int m_fd;
  longlong m_globalSize, m_globalIndex;
  unsigned int m_localIndex, m_maxsize;
  vec<T> m_data;
};



// ============================================================================
// ========================================================================

/// Helper for functions that load Binary format 2 or 3, whichever the file is.
int WhichBinaryFormat( const String& filename );

/// Return the number of elements of a binary format 2 or 3 file. 
/// Assert if file is not binary 2 or 3 format.
/// Note that this is not templatized, unlike the BinarySize*** functions.
longlong BinaryNumElements( const String & filename);




// ============================================================================
// ========================================================================

void PrettyPrint( ostream& o, const vec<int>& v, int max_items = 0,
     String terminator = "\n" );

void PrettyPrint( ostream& o, const vec<longlong>& v, int max_items = 0,
     String terminator = "\n" );

void PrettyPrint( ostream& o, const vec<double>& v, int max_items = 0,
     String terminator = "\n" );

// CompactPrint: print a vector of objects, separated by blanks.

template<class T> void CompactPrint( ostream& out, const vec<T>& v )
{    for ( int i = 0; i < v.isize( ); i++ )
     {    if ( i > 0 ) out << " ";
          out << v[i];    }    }

/**Permute input vector v in place according to permutation.
   Preconditions:
   - v.size() == permutation.size()
   If the permutation contains a -1, the position corresponding to that
   is essentially ignored and ends up in one of the available empty spaces

   This works for all std::vectors and vecs.
*/
template<class V>
void PermuteVec(V & v, const vec<int> & permutation) {
  AssertEq(v.size(), permutation.size());
  vec<int> o = permutation;
  for (int i = 0; i != (longlong) v.size(); ++i) {
    while (o[i] != i && o[i] != -1) {
      std::swap(v[i], v[o[i]]);
      std::swap(o[i], o[o[i]]);
    }
  }
}

// Instantiate with a vector v, then use in place of operator<.
// It will tell you i<j if v[i]<v[j].  Helper for WhatPermutation.
template<class S, typename C> 
struct indirect_compare : public binary_function<int,int,bool> {
  const vec<S>& v;
  C comp;
  indirect_compare(const vec<S>& v ) : v(v) {}
  indirect_compare(const vec<S>& v, C comparator) : v(v), comp(comparator){}
  bool operator() ( int i, int j ) { return comp( v[i], v[j] ); }
};

// What permutation would we apply to V to get it sorted?
// NOTE: If v is a vec<int>, this inverts it.
template<class V, typename C >
void WhatPermutation( const V & v, vec<int>& permutation, C comparator, 
		      bool inv ) {
  int n = v.size();
  vec<int> perm(n);
  for(int i=0; i<n; i++) perm[i]=i;
  sort( perm.begin(), perm.end(), 
	indirect_compare<typename V::value_type,C>(v,comparator) );
  if( inv ) {
    // That's the inverse of the permutation PermuteVec takes.
    permutation.resize(n);
    for(int i=0; i<n; i++) permutation[perm[i]]=i;
  }
  else {
    swap(perm, permutation);
  }
}

template<class V>
void WhatPermutation( const V & v, vec<int>& permutation, bool inv = true ) {
  WhatPermutation< V, less<typename V::value_type> >
    (v,permutation, less<typename V::value_type>(), inv );
}


/// WriteAppend has an implementation for T = alignment_plus in Alignment.{h,cc}.

template<class T> void WriteAppend( const String& f, const vec<T>& v )
{    ForceAssert( !IsRegularFile( f + ".gz" ) );
     static longlong max_size_bound = longlong(10000000) * longlong(100000000);
     if ( !IsRegularFile(f) )
     {    ForceAssertLt( (longlong) v.size( ), max_size_bound );
          Ofstream( out, f );
          out << setfill( '0' ) << setw(15) << v.size( ) << "\n";
          for ( longlong i = 0; i < (longlong) v.size( ); i++ )
               out << v[i];    }
     else
     {    longlong n;
          {    Ifstream( in, f );
               in >> n;    }
          ForceAssertLt( n + (longlong) v.size( ), max_size_bound );
          ostrstream osize;
          osize << setfill( '0' ) << setw(15) << n + v.size( ) << "\n";
          int fd = Open( f, O_WRONLY );
          WriteBytes( fd, osize.str( ), 16 );
          close(fd);
          ofstream out( f.c_str( ), ios::app );
          for ( longlong i = 0; i < (longlong) v.size( ); i++ )
               out << v[i];    }    }

/// a specialized version of WriteAppend for String

template <>
void WriteAppend( const String& f, const vec<String>& v );

template<class T> ostream& operator<<(ostream& s, const vec<T>& v)
{    s << v.size( ) << "\n";
     for ( unsigned int i = 0; i < v.size( ); i++ )
          s << v[i];
     return s;    }

#ifdef __DECCXX_VER
#pragma do_not_instantiate ostream& operator<<(ostream&, const vec<int>&)
#pragma do_not_instantiate ostream& operator<<(ostream&, const vec<longlong>&)
#pragma do_not_instantiate ostream& operator<<(ostream&, const vec<float>&)
#pragma do_not_instantiate ostream& operator<<(ostream&, const vec<String>&)
#endif

ostream& operator<<(ostream& s, const vec<unsigned short>& v);
ostream& operator<<(ostream& s, const vec<int>& v);
ostream& operator<<(ostream& s, const vec<longlong>& v);
ostream& operator<<(ostream& s, const vec<float>& v);
ostream& operator<<(ostream& s, const vec<double>& v);
ostream& operator<<(ostream& s, const vec<String>& v);

template<class T> istream& operator>>(istream& s, vec<T>& v)
{    int n;
     s >> n;
     v.resize(n);
     char c;
     s.get(c);
     for ( unsigned int i = 0; i < v.size( ); i++ )
       s >> v[i];   // Breaks cxx
     return s;    }


#ifdef __DECCXX_VER
#pragma do_not_instantiate istream& operator>>(istream&, vec<String>&)
#endif

istream& operator>>(istream& s, vec<String>& v);

/// Print out a matrix, with left-justified entries, and given separation between
/// columns.  (Justification may be changed by supplying an optional argument
/// consisting of a string of l's and r's.)

void PrintTabular( ostream& out, const vec< vec<String> >& rows, int sep,
     String justify = String( ) );


#endif
