/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif
#define STRING_FAST_EXECUTE

#include <ctype.h>
#include <stdio.h>
#include <float.h>

#include <iostream>
#include <fstream>
#include <strstream>
#include <vector>

#include "system/Assert.h"
#include "String.h"
#include "system/Types.h"

#ifndef FatalErr
     #define FatalErr(message) { cout << message << endl; ForceAssert(0 == 1); }
#endif

char* String::data_not_inlined( )
{ return ( ( SelfOwned() && ( length_ < cutoff_length ) ) ? start_of_buffer_ : data_ptr_ ); }
const char* String::data_not_inlined( ) const
{ return ( ( SelfOwned() && ( length_ < cutoff_length ) ) ? start_of_buffer_ : data_ptr_ ); }

String operator_plus( const String& first, const String& second )
{    String answer( first );
     answer += second;
     return answer;    }

bool operator_eq( const String& left, const String& right )
{    return ( left.size() == right.size() &&
              strcmp( left.data(), right.data() ) == 0 );    }

bool operator_ne( const String& left, const String& right )
{    return !( left == right );    }

bool operator_lt ( const String& left, const String& right )
{    return ( strcmp(left.data(), right.data()) < 0 );    }

// realloc_() prepares the String to store data of up to new_size
// chars.  Any existing data is discarded.  The data array is
// null-terminated at the new length.

void String::realloc_( const unsigned int new_size )
{
  if ( size() == new_size )
    return;

  // Case 1: the String is currently stored externally or is short
  // enough to be stored locally, and the new size is small enough to
  // store it locally.
  if ( ! SelfOwned() || size() < cutoff_length )
  {
    if ( new_size < cutoff_length )
    {
      // we don't need to reallocate anything
    }
    // Case 2: the String is currently stored externally or is short
    // enough to be stored locally, but the new size is too big for
    // that.
    else
    {
      // allocate a new pointer to a large-enough chunk of memory
      data_ptr_ = new char[new_size+1];
      //      clog << new_size+1 << "\n" ;
    }
  }
  // Case 3: the String is currently dynamically allocated, but the
  // new size is small enough to store locally.
  else 
  {
    if ( new_size < cutoff_length )
    {
      // delete the data pointer
      //      clog << (int) -length_-1 << endl;
      delete [] data_ptr_;
    }
    // Case 4: the String is currently dynamically allocated, but needs
    // to be resized
    else 
    {
      // delete the data pointer
      //      clog << (int) -length_-1 << endl;
      delete [] data_ptr_;
      // allocate a new pointer to a large-enough chunk of memory
      data_ptr_ = new char[new_size+1];
      //      clog << new_size+1 << "\n" ;
    }
  }

  // adjust length_ to the new size, clearing the "vassal" bit
  length_ = new_size;
  // null-terminate it for safety
  data()[new_size] = 0;
}


// realloc_and_copy_() prepares the String to store data of up to
// new_size chars, preserving existing data up to the new size.  The
// data array is null-terminated at the new length.

void String::realloc_and_copy_( const unsigned int new_size )
{
  if ( size() == new_size )
    return;

  // First case: the String is currently stored externally or is short
  // enough to be stored locally and the new size is short enough to
  // store locally.
  if ( ! SelfOwned() || size() < cutoff_length )
  {
    if ( new_size < cutoff_length )
    {
      // no reallocation or copying necessary
    }
    // Case 2: the String is currently stored externally or is short
    // enough to be stored locally, but the new size is too big for
    // local storage.
    else
    {
      // allocate a new pointer to a large-enough chunk of memory
      char* new_data = new char[new_size+1];
      //      clog << new_size+1 << "\n" ;
      // copy the existing data in the local array to the new chunk
      memcpy( new_data, data(), size() );
      // set the pointer to the address of the chunk
      data_ptr_ = new_data;
    }
  }
  // Case 3: the String is currently dynamically allocated and is
  // being erased (new_size == 0).
  else
  {
    if ( new_size == 0 )
    {
      // delete the data pointer
      //      clog << (int) -length_-1 << endl;
      delete [] data_ptr_;
    }
    // Case 4: the String is currently dynamically allocated and is
    // being shortened enough to be stored locally.
    else if ( new_size < cutoff_length )
    {
      // copy the location of data
      char* old_data = data_ptr_;
      // copy the data to the local array
      memcpy( start_of_buffer_, old_data, new_size );
      // delete the old data
      //      clog << (int) -length_-1 << endl;
      delete [] old_data;
    } 
    // Case 5: the String is currently dynamically allocated and is
    // being resized.
    else 
    {
      // get a new chunk of memory that's large enough
      char* new_data = new char[new_size+1];
      //      clog << new_size+1 << "\n" ;
      // copy the old data to the new chunk
      memcpy( new_data, data_ptr_, min<unsigned int>(size(), new_size) );
      // delete the old data
      //      clog << (int) -length_-1 << endl;
      delete [] data_ptr_;
      // point to the new chunk
      data_ptr_ = new_data;
    }
  }

  // adjust length_, clearing the "vassal" bit
  length_ = new_size;
  // null-terminate it at the new length
  data()[new_size] = 0; 
}
	    
void String::resize(unsigned int n)
{
  realloc_and_copy_(n);
}

void String::Set( const char* x, int l )
{
  realloc_(l);
  memcpy(data(),x,l);
  Assert( strncmp( data(), x, l ) == 0 );
}

// All the constructors initialize length_ and data_ to valid
// values before calling realloc_().

String::String( ) 
  : length_(0)
{
  data()[0] = 0;
}

String::String(int n) 
  : length_(0)
{
  data()[0] = 0;
  realloc_(n);
}

String::String(char c) : length_(0)
{    data()[0] = 0;
     realloc_(1);
     data()[0] = c;    }

String::String(const String& s)
  : length_(0)
{
  data()[0] = 0;
  realloc_( s.size() );
  memcpy(data(), s.data(), length_);
  Assert( strcmp( data(), s.data() ) == 0 );
}

String::String( const String& s, const int n ) : length_(0)
{
  AssertGe( n, 0 );
  data()[0] = 0;
  realloc_( s.size() * n );
  for ( int i = 0; i < n; i++ )
       memcpy( data() + s.size( ) * i, s.data(), s.size( ) );
}

String::String(const string& s)
  : length_(0)
{
  data()[0] = 0;
  realloc_( s.size() );
  memcpy(data(), s.c_str(), length_); 
  //This assert is not complete security: it stops at the first '\0'.
  Assert( strcmp( data(), s.c_str() ) == 0 );
}

String::String(const char* x)
  : length_(0)
{ if ( x == 0 ) FatalErr( "The String constructor has been passed a null pointer." );
  data()[0] = 0;
  realloc_( strlen(x) );
  memcpy(data(), x, length_); 
  Assert( strcmp( data(), x ) == 0 );
}

String::String( const vector<char>& v )
  : length_(0)
{ data()[0] = 0;
  realloc_( v.size( ) );
  memcpy( data(), &v[0], v.size( ) ); 
}

String& String::operator= (const String& s)
{
  if ( data() == s.data() ) // avoid self-assignment
    return *this;
  realloc_( s.size() );
  memcpy(data(), s.data(), size());
  Assert( strcmp( data(), s.data() ) == 0 );
  return *this;    
}

String& String::operator= (const char* x)
{
  this->Set(x, strlen(x));
  return *this; 
} 

String& String::operator= (const char c)
{
  this->Set(&c, 1);
  return *this; 
} 

String& String::operator+= ( const String& s )
{
  if ( s.size() == 0 )
    return *this;
  unsigned int original_length = size();
  realloc_and_copy_( original_length + s.size() );
  memcpy( data() + original_length, s.data(), s.size() );
  return *this;    
}

String& String::operator+= ( const char* x )
{
  int len = strlen( x );
  if ( len == 0 )
    return *this;
  unsigned int original_length = size();
  realloc_and_copy_( original_length + len );
  memcpy( data() + original_length, x, len );
  return *this;
}

String& String::operator+= ( const char c )
{
  realloc_and_copy_( size() + 1 );
  data()[ length_-1 ] = c;
  return *this;
}

String::~String( )
{ 
  if ( SelfOwned() && length_ >= cutoff_length )
    delete [] data();
}

ostream& operator<< ( ostream &out, const String& s )
{    
  return out << s.data(); 
}

istream& operator>> ( istream& in, String& s)
{
  char chars[50001];
  int pos = 0;
  char c;
  while(in)
  {
    in.get(c);
    if ( !isspace(c) )
    {
      in.putback(c);
      break; 
    }
  }
  while(in)
  {
    in.get(c);
    if ( !in || isspace(c) ) 
      break;
    chars[pos++] = c;
    if ( pos == 50000 )
    { 
      cout << "String operator istream: attempt to read String "
	   << "of length > 50000.  Abort.\n";
      exit(-1); 
    }
  }
  chars[pos] = 0;
  s.Set( chars, pos );  
  return in;   
}

String String::substr( unsigned int start, int length ) const
{    if ( length == -1 ) length = isize( ) - start;
     if ( start + length > size() ) length = size() - start;
     String answer;
     answer.realloc_(length);
     memcpy( answer.data(), this->data() + start, length );
     return answer;    }

void String::erase( int start, int length )
{
  replace( start, length, "" );    
}

void String::erase()
{
  this->resize( 0 );
}

///Line ends at either Unix-style LF or DOS-style CR-LF or Mac-style CR.
void getline( istream& in_strm, String& out_str )
{
  static char buffer[16384];

  out_str = "";
  int pos = 0;
  char c=0;
  //bool lineEnded = false;
  while(in_strm)
  {
    in_strm.get(c);
    if ( c == 0x0D && in_strm.peek() == 0x0A) {
      in_strm.get(c);
      break;
    }
    else if ( c == '\n'  
	      || c == 0x0D //end line at DOS-style CR-LF or Mac-style CR.
	      || ! in_strm ) 
      break;

    buffer[pos++] = c;

    if ( pos == 16383 )
      {
	buffer[pos] = 0;
	out_str += buffer;
	pos = 0;
      }
  }

  buffer[pos] = 0;
  out_str += buffer;   
}


String::operator string( ) const {
  string s;
  s.resize(size());
  memcpy(&s[0], data(), size()); 
  //This assert is not complete security: it stops at the first '\0'.
  Assert( strcmp( data(), s.c_str() ) == 0 );
  return s;
}

void String::replace( unsigned int start, unsigned int length, 
		      const char* replacement )
{ 
  if ( start + length > size() ) 
    length = size() - start;

  char* head_start = this->data();
  char* tail_start = head_start + start + length;

  const unsigned int head_len = start;
  const unsigned int tail_len = size() - start - length;
  const unsigned int replacement_len = strlen( replacement );

  String answer;
  answer.realloc_( size() - length + replacement_len );
  if ( head_len )
    memcpy( answer.data(), 
	    head_start, head_len );
  if ( replacement_len )
    memcpy( answer.data() + start, 
	    replacement, replacement_len );
  if ( tail_len )
    memcpy( answer.data() + start + replacement_len, 
	    tail_start, tail_len );
  
  *this = answer;
}

void String::append( const String& s, int start, int length )
{
  *this += s.substr( start, length );   
}

void String::Slurp(istream & is) {
  vector<char> buf;
  buf.reserve(10000);
  char c;
  while (is.get(c)) buf.push_back(c);
  *this = String(buf);
}

Bool String::IsInt( ) const
{    unsigned int n = size( );
     if ( n >= 1 && 
          ( (*this)[n-1] == 'K' || (*this)[n-1] == 'M' || (*this)[n-1] == 'G' ) )
     {    --n;    }
     for ( unsigned int i = 0; i < n; i++ )
     {    if ( i == 0 && (*this)[i] == '-' );
          else if ( i == 0 && (*this)[i] == '+' );
          else 
          {    int digit = (*this)[i] - '0';
               if ( digit < 0 || digit > 9 ) return False;    }    }
     return True;    }

// Int does not check for integers which are too large to fit in a longlong.

longlong String::Int( ) const
{    unsigned int n = size( );
     longlong multiplier = 1;
     if ( n >= 1 && (*this)[n-1] == 'K' )
     {    multiplier = 1000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'M' )
     {    multiplier = 1000000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'G' )
     {    multiplier = 1000000000;
          --n;    }
     bool negated = false;
     longlong answer = 0;
     for ( unsigned int i = 0; i < n; i++ )
     {    if ( i == 0 && (*this)[i] == '-' ) negated = true;
          else if ( i == 0 && (*this)[i] == '+' );
          else 
          {    int digit = (*this)[i] - '0';
               if ( digit < 0 || digit > 9 )
                    FatalErr( "Attempt to convert string \"" << *this <<
                         "\" to an int failed." );
               answer = 10*answer + digit;    }    }
     if (negated) answer = -answer;
     return answer * multiplier;    }

Bool String::IsDouble() const
{    unsigned int n = size( );
     double multiplier = 1;
     if ( n >= 1 && (*this)[n-1] == 'K' )
     {    multiplier = 1000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'M' )
     {    multiplier = 1000000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'G' )
     {    multiplier = 1000000000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == '%' )
     {    multiplier = 0.01;
          --n;    }
     double ret = 0.;
     int i = sscanf(c_str(),"%lg",&ret );
     return i != 0;    }

double String::Double( ) const 
{    unsigned int n = size( );
     double multiplier = 1;
     if ( n >= 1 && (*this)[n-1] == 'K' )
     {    multiplier = 1000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'M' )
     {    multiplier = 1000000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == 'G' )
     {    multiplier = 1000000000;
          --n;    }
     else if ( n >= 1 && (*this)[n-1] == '%' )
     {    multiplier = 0.01;
          --n;    }
     double ret = 0.;
     int i = sscanf(c_str(),"%lg",&ret );
     if (i == 0) 
     {    FatalErr("Attempt to convert string \"" << *this 
               << "\" to a double failed." );    }
     ret *= multiplier;
     return ret;    }

Bool String::IsBool() const {
  return empty() || "True" == *this || "False" == *this;
}

Bool String::ToBool() const {
  if (empty() || "False" == *this) return False;
  else if ("True" == *this) return True;
  else {
    FatalErr("Attempt to convert string \"" << *this
	<< "\" to a Bool failed.");
    return False;//should not happen, here only to avoid compiler warning
  }
}

// int String::Position( const String& s ) const
// { 
//   return Position( s.c_str() );
// }

int String::Position( const char* x ) const
{ 
  int x_size = strlen(x);
  int max_pos = (int) this->size() - x_size;
  for ( int i = 0; i <= max_pos; i++ )
  {
    int j;
    for ( j = 0; j < x_size; ++j )
      if ( (*this)[i + j] != x[j] ) 
	break;
    if ( j == x_size ) 
      return i; 
  }
  return -1;    
}

int String::Position( const String& x ) const
{    
  int max_pos = (int) this->size() - (int) x.size();
  for ( int i = 0; i <= max_pos; i++ )
  {
    int j;
    for ( j = 0; j < (int) x.size( ); ++j )
      if ( (*this)[i + j] != x[j] ) 
	break;
    if ( j == (int) x.size( ) ) 
      return i;    
  }
  return -1;    
}

int String::Position( const String& x, const int &endSearchAt ) const
{    
  int endPt = endSearchAt < (int) this->size() ? endSearchAt : this->size();
  int max_pos = endPt - (int) x.size();
  for ( int i = 0; i <= max_pos; i++ )
  {
    int j;
    for ( j = 0; j < (int) x.size( ); ++j )
      if ( (*this)[i + j] != x[j] ) 
	break;
    if ( j == (int) x.size( ) ) 
      return i;    
  }
  return -1;    
}

int String::PositionAfter( const String& x, const int &startSearchAt ) const
{    
  int max_pos = this->size() - (int) x.size();
  for ( int i = startSearchAt; i <= max_pos; i++ )
  {
    int j;
    for ( j = 0; j < (int) x.size( ); ++j )
      if ( (*this)[i + j] != x[j] ) 
	break;
    if ( j == (int) x.size( ) ) 
      return i;    
  }
  return -1;    
}

int String::PosRev( const String& s ) const
{    for ( int i = this->isize( ) - s.isize( ); i >= 0; i-- )
          if ( this->Contains( s, i ) ) return i;
     return -1;    }

bool String::Contains( const char* x ) const
{
  return Position(x) != -1;    
}

bool String::Contains( const String& s ) const
{
  return Position(s) != -1;   
}

String String::Before( const char* x ) const
{ 
  int n = Position( x );
  if ( n == -1 ) 
    FatalErr( "Before(): Couldn't find \"" + String(x) + "\" in \"" + *this + "\".\n" );
  return substr( 0, n );    
}

String String::Before( const String& s ) const
{
  int n = Position(s);
  if ( n == -1 ) 
    FatalErr( "Before(): Couldn't find \"" + s + "\" in \"" + *this + "\".\n" );
  return substr( 0, n );    
}

String String::SafeBefore( const String & s) const {
  int n = Position(s);
  if (n != -1) return substr(0,n);
  else return *this;
}

String String::SafeAfter( const String & s) const {
  int n = Position(s);
  if (n != -1) return substr(n + s.size(), size());
  else return *this;
}

String String::After( const char* x ) const
{
  int n = Position(x);
  if ( n == -1 ) 
    FatalErr( "After() couldn't find \"" + String(x) + "\" in \"" + *this + "\"." );
  return substr( n + strlen(x), size( ) );    
}

String String::After( const String& s ) const
{
  int n = Position(s);
  if ( n == -1 ) 
    FatalErr( "After(): Couldn't find \"" + s + "\" in \"" + *this + "\".\n" );
  return substr( n + s.size( ), size( ) );    
}

bool String::Contains( const char* x, int pos ) const
{
  int n = strlen(x);
  if ( pos == -1 ) 
    pos = (int) this->size() - n;
  if ( pos < 0 ) 
    return false;
  int j;
  for ( j = 0; j < n; ++j )
  {
    if (pos + j >= this->isize())  { break; }
    if ((*this)[pos + j] != x[j]) { break; }
  }
  if ( j == n ) 
    return true;
  return false;    
}


void String::ToLower() {
  const int S = size();
  char c;
  for (int i = 0; i != S; ++i) {
    (*this)[i] = tolower((*this)[i]);
  }
}

void String::ToUpper() {
  const int S = size();
  char c;
  for (int i = 0; i != S; ++i) {
      (*this)[i] = toupper((*this)[i]);
  }
}

bool String::Contains( const String& s, int pos ) const
{
  if ( pos == -1 ) 
    pos = this->size( ) - s.size();
  if ( pos < 0 ) 
    return false;
  unsigned int j;
  for ( j = 0; j < s.size(); ++j )
    if ( (*this)[pos + j] != s[j] ) 
      break;
  if ( j == s.size() ) 
    return true;
  return false;
}

String String::Evaluate( const bool verbose ) const
{
  if ( verbose )
    cout << " Evaluating '" << *this << "'." << endl;
  
  if ( this->empty() )
    return "";
  String expression(*this);
  
  for ( unsigned int check = 0; check < expression.size(); check++ ) {
    char check_char = expression[check];
    if ( ! isdigit( check_char ) &&
	 check_char != '(' &&
	 check_char != ')' &&
	 check_char != '*' &&
	 check_char != '/' &&
	 check_char != '+' &&
	 check_char != '-' )
      return "" ;
  }

  int left_paren_pos;
  while ( ( left_paren_pos = expression.Position("(") ) != -1 ) {
    if ( verbose )
      cout << "  Found left paren at " << left_paren_pos << "." << endl;
    int right_paren_pos = expression.Position(")");
    if ( right_paren_pos == -1 )
      return "";
    if ( verbose )
      cout << "  Found right paren at " << right_paren_pos << "." << endl;
    int paren_length = right_paren_pos - left_paren_pos + 1;
    String subexpression = expression.substr(left_paren_pos + 1, paren_length - 2);
    expression.replace( left_paren_pos, paren_length, subexpression.Evaluate().c_str() );
  }

  int mult_div_pos;
  for ( mult_div_pos = 0; mult_div_pos < (int) expression.size(); mult_div_pos++ )
    if ( expression[mult_div_pos] == '*' ||
	 expression[mult_div_pos] == '/' ) {

      if ( verbose )
	cout << "  Found '" << expression[mult_div_pos] << "' at " << mult_div_pos << "." << endl;

      if ( mult_div_pos == 0 ||
	   mult_div_pos == (int) expression.size() - 1 )
	return "";

      int first_operand_start, first_operand_length;
      int second_operand_start, second_operand_length;

      if ( ! expression.FindOperands_( mult_div_pos,
				       first_operand_start, first_operand_length,
				       second_operand_start, second_operand_length,
				       verbose ) )
	return "";
		     
      int first_operand = expression.substr(first_operand_start, 
					    first_operand_length).Int();
      int second_operand = expression.substr(second_operand_start,
					     second_operand_length).Int();

      String replacement;
      if ( expression[mult_div_pos] == '*' ) {
        replacement = ToString( first_operand * second_operand ).c_str();
      } 
      else { // ( expression[mult_div_pos] == '/' )
	if ( second_operand == 0 )
	  return "";
        replacement = ToString( first_operand / second_operand ).c_str();
      }

      if ( verbose )
	cout << "  Replacing with " << replacement << "." << endl;

      expression.replace( first_operand_start, 
			  first_operand_length + 1 + second_operand_length,
			  replacement.c_str() );
      mult_div_pos = first_operand_start + replacement.size() - 1; 
    }
  
  int add_sub_pos;
  for ( add_sub_pos = 0; add_sub_pos < (int) expression.size(); add_sub_pos++ )
    if ( expression[add_sub_pos] == '+' ||
	 expression[add_sub_pos] == '-' ) {

      if ( verbose ) 
	cout << "  Found '" << expression[add_sub_pos] << "' at " << add_sub_pos << "." << endl;

      if ( expression[add_sub_pos] == '-' )          // if it's a minus sign
	if ( add_sub_pos == 0 ) {                    // and it's at the beginning
	  if ( expression.size() > 1 &&              // and it's not the only character and
	     isdigit(expression[add_sub_pos+1]) )    // it's followed by a digit
	    continue;                                // then it's a negation, so nevermind
	  else                                       // but if it is the only character or
	                                             // it's not followed by a digit
	    return "";                               // then it's malformed, so bail
	}
	else                                           // but if it's not at the beginning
	  if ( ! isdigit(expression[add_sub_pos-1] ) ) // and it's just after an operator
	    continue;                                  // then it's a negation, so nevermind
	     
      if ( add_sub_pos == 0 ||
	   add_sub_pos == (int) expression.size() - 1 )
	return "";

      int first_operand_start, first_operand_length;
      int second_operand_start, second_operand_length;

      if ( ! expression.FindOperands_( add_sub_pos,
				       first_operand_start, first_operand_length,
				       second_operand_start, second_operand_length,
				       verbose ) )
	return "";

      int first_operand = expression.substr(first_operand_start, 
					    first_operand_length).Int();
      int second_operand = expression.substr(second_operand_start,
					     second_operand_length).Int();
      String replacement;
      if ( expression[add_sub_pos] == '+' ) {
        replacement = ToString( first_operand + second_operand ).c_str();
      } 
      else { // ( expression[add_sub_pos] == '-' )
        replacement = ToString( first_operand - second_operand ).c_str();
      }

      if ( verbose )
	cout << "  Replacing with " << replacement << "." << endl;

      expression.replace( first_operand_start, 
			  first_operand_length + 1 + second_operand_length,
			  replacement.c_str() ) ;
      add_sub_pos = first_operand_start + replacement.size() - 1;
    }
  return expression;
}
	
bool String::FindOperands_( const int operator_pos,
				int& first_operand_start, int& first_operand_length,
				int& second_operand_start, int& second_operand_length,
				const bool verbose ) const
{
  for ( first_operand_start = operator_pos-1; 
	first_operand_start >= 0; 
	first_operand_start-- )
    if ( ! isdigit((*this)[first_operand_start]) ) // if it's not a digit AND
      if ( ! (*this)[first_operand_start] == '-' ) // it's not a minus sign
	break;                                     // then it's an operator, so break
      else                                         // but if it's a minus sign
	if ( first_operand_start == 0 )            // and it's the first character
	  if ( first_operand_start != operator_pos-1 ) // and the next character is not the op
	    continue;                              // then it's a negation, so keep going
	  else                                     // but if the next character is the op
	    return false;                          // then it's malformed
	else                                       // but if it's not the first character
	  if ( isdigit( (*this)[first_operand_start-1] ) ) // and the previous char is a digit
	    break;                                         // then it's an operator, so break
	  else                                             // but if the previous char is an op
	    if ( first_operand_start != operator_pos-1 )   // and the next char is not the op
	      continue;                                    // then it's a negation, so keep going
	    else                                           // but if the next char is the op
	      return false;                                // then it's malformed
  
  first_operand_start++;
  
  int second_operand_end;
  for ( second_operand_end = operator_pos+1; 
	second_operand_end < (int) this->size(); 
	second_operand_end++ )
    if ( ! isdigit((*this)[second_operand_end]) ) // if it's not a digit AND
      if ( ! (*this)[second_operand_end] == '-' ) // it's not a minus sign
	break;                                    // then it's an operator, so break
      else                                        // but if it's a minus sign
	if ( second_operand_end == operator_pos+1 ) // and it's just after the operator
	  continue;                                 // then it's a negation, so keep going
	else                                        // but if it's not just after the operator
	  break;                                    // then it's an operator, so break

  second_operand_end--;
  
  if ( first_operand_start == operator_pos ||
       second_operand_end == operator_pos )
    return false;
  
  first_operand_length = operator_pos - first_operand_start;
  second_operand_start = operator_pos + 1;
  second_operand_length = second_operand_end - operator_pos;
  
  if ( verbose ) {
    cout << "  First operand starts at " << first_operand_start << "." << endl;
    cout << "  First operand has length " << first_operand_length << "." << endl;
    cout << "  Second operand starts at " << second_operand_start << "." << endl;
    cout << "  Second operand has length " << second_operand_length << "." << endl;
  }
  
  return true;
}	  
  
String ToString( int x )
{    
  char dec[20];
  sprintf( dec, "%d", x );
  return String(dec);    
}

String ToString( unsigned int x )
{
  char dec[20];
  sprintf( dec, "%d", x );
  return String(dec);   
}

String ToString( longlong x )
{
  string answer;
  strstream s;
  s << x;
  s >> answer;
  return answer;    
}

String ToString( ulonglong x )
{
  string answer;
  strstream s;
  s << x;
  s >> answer;
  return answer;    
}

String ToString( float x, int precision )
{
  char dec[20];
  String conv = "%." + ToString(precision) + "f";
  sprintf( dec, conv.c_str( ), x );
  return String(dec);    
}

String ToString( double x, int precision )
{
  char dec[20];
  String conv = "%." + ToString(precision) + "f";
  sprintf( dec, conv.c_str( ), x );
  return String(dec);   
}


String ToStringAbbrev( const longlong x )
{
  if ( x < 1000 )
    return ToString( x );

  double xDouble = x;
  xDouble /= 1000.;
  String suffix = "K";
  int precision = 0;

  if ( xDouble > 1000. ) {
    xDouble /= 1000.;
    suffix = "M";
    ++precision;
  }
  if ( xDouble > 1000. ) {
    xDouble /= 1000.;
    suffix = "G";
    ++precision;
  }
  return ToString( xDouble, precision ) + suffix;
}


String ToLower (const String & s) {
  String ret(s);
  ret.ToLower();
  return ret;
}

String ToUpper (const String & s) {
  String ret(s);
  ret.ToUpper();
  return ret;
}

void DeleteLeadingWhiteSpace( String& s )
{
  while( s.size( ) > 0 && isspace( s[0] ) ) 
    s.erase(0, 1);   
}

void DeleteTrailingWhiteSpace( String& s )
{ 
  while( s.size( ) > 0 && isspace( s[ s.size( ) - 1 ] ) ) 
    s.erase( s.size( ) - 1, 1 );    
}

String WhiteSpaceFree( String s )
{
  String answer;
  for ( unsigned int i = 0; i < s.size( ); i++ )
    if ( !isspace( s[i] ) ) 
      answer += s[i];
  return answer;    
}

void String::ReplaceBy( const String& from, const String& to )
{
  for ( unsigned int i = 0; i < size( ); i++ )
  {
    unsigned int j;
    for ( j = 0; j < from.size( ); j++ )
      if ( (*this)[i + j] != from[j] ) break;
    if ( j == from.size( ) ) 
    {
      replace( i, from.size( ), to.c_str( ) );
      return;
    }
  }
  FatalErr( "ReplaceBy: could not find \"" << from << "\" in \""
	    << *this << "\"." );    
}

void String::GlobalReplaceBy( const String& from, const String& to )
{    if ( to.Contains(from) ) ReplaceBy( from, to );
     else 
     {    while( Contains(from) ) ReplaceBy( from, to );    }    }

Bool cmp_numeric( const String& s1, const String& s2 )
{    for ( int i = 0; i < (int) s1.size( ); i++ )
     {    if ( i == (int) s2.size( ) ) return False;
          if ( !isdigit( s1[i] ) || !isdigit( s2[i] ) )
          {    if ( s1[i] < s2[i] ) return True;
               if ( s1[i] > s2[i] ) return False;
               continue;    }
          int j1, j2;
          for ( j1 = i + 1; j1 < (int) s1.size( ); j1++ )
               if ( !isdigit( s1[j1] ) ) break;
          for ( j2 = i + 1; j2 < (int) s2.size( ); j2++ )
               if ( !isdigit( s2[j2] ) ) break;
          longlong n1 = 0, n2 = 0;
          for ( int k = i; k < j1; k++ )
               n1 = ( 10 * n1 ) + ( s1[k] - '0' );
          for ( int k = i; k < j2; k++ )
               n2 = ( 10 * n2 ) + ( s2[k] - '0' );
          if ( n1 < n2 ) return True;
          if ( n1 > n2 ) return False;
          if ( j1 < j2 ) return False;
          if ( j1 > j2 ) return True;
          i = j1 - 1;    }
     return s1.size( ) < s2.size( );    }

String BaseAlpha( int n )
{    ForceAssertGe( n, 0 );
     String answer; 
     static String letter( "A" );
     while(1)
     {    letter[0] = 'A' + (n % 26);
          answer = letter + answer;
          n /= 26; 
          if ( n == 0 ) break;
          --n;    } 
     return answer;    }

int String::Freq( const String& s )
{    int count = 0;
     for ( int i = 0; i < isize( ); i++ )
          if ( this->Contains( s, i ) ) ++count;
     return count;    }

