// Copyright (c) 2000-2003 Whitehead Institute for Biomedical Research
// 


#ifndef STLEXTENSIONS_H
#define STLEXTENSIONS_H

using namespace std;

#include <functional>

#include <math.h>

/// minimum<T> is a function object.  
///
/// If f is an object of class minimum<T> and x and y are objects of
/// class T, then f(x,y) returns a copy of whichever of x or y is lower
/// as defined by T::operator<().
///
/// Example:
///
/// vector<int> V(10);
/// iota( V.begin(), V.end(), 1 );
///
/// copy( V.begin(), V.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///
/// transform( V.begin(), V.end(), V.begin(), bind2nd(minimum<int>(), 5) );
///  
/// copy( V.begin(), V.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///
/// The preceding code produces the following output:
///
/// 1 2 3 4 5 6 7 8 9 10
/// 1 2 3 4 5 5 5 5 5 5
 
template <class T>
struct minimum : public binary_function<T, T, T> 
{
  T operator() (const T& x, const T& y ) const { return ( x<y ? x : y ); }
};

/// maximum<T> is a function object.  
///
/// If f is an object of class maximum<T> and x and y are objects of
/// class T, then f(x,y) returns a copy of whichever of x or y has the
/// highest value as defined by T::operator<().
///
/// Example:
///
/// vector<int> V(10);
/// iota( V.begin(), V.end(), 1 );
///
/// copy( V.begin(), V.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///
/// transform( V.begin(), V.end(), V.begin(), bind2nd(maximum<int>(), 5) );
///  
/// copy( V.begin(), V.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///
/// The preceding code produces the following output:
///
/// 1 2 3 4 5 6 7 8 9 10
/// 5 5 5 5 5 6 7 8 9 10
 
template <class T>
struct maximum : public binary_function<T, T, T> 
{
  T operator() (const T& x, const T& y ) const { return ( x<y ? y : x ); }
};

/// modulus<float> is a function object that implements the template modulus<T>
/// modulus<double> is a function object that implements the template modulus<T>
///
/// The default modulus<T> defines its operator() as
///
/// { return x % y; }
///
/// floats and doubles do not have a % operator defined.  They require specific
/// math functions to perform modulus operations.  These implementations are provided
/// below.

namespace std {
    template <>
    struct modulus<float> : public binary_function<float, float, float>
    {
        float operator() (const float& x, const float& y) const { return fmodf( x, y ); }
    };

    template <>
    struct modulus<double> : public binary_function<double, double, double>
    {
        double operator() (const double& x, const double& y) const { return fmod( x, y ); }
    };
}

/// address_of<T> and dereference<T> are function objects.
///
/// If f is an object of class address_of<T> and x is an object of
/// class T, then f(x) returns the address of x, i.e. a T*.
///
/// If f is an object of class dereference<T> and x is a pointer to
/// an object of class T, then f(x) returns a copy of *x.
///
/// Example:
///
/// vector<int> V1(10);
/// iota( V1.begin(), V1.end(), 1 );
///
/// copy( V1.begin(), V1.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///
/// vector<int*> V_ptrs(10);
/// transform( V1.begin(), V1.end(), V_ptrs.begin(), address_of<int>() );
///
/// vector<int> V2(10);
/// transform( V_ptrs.begin(), V_ptrs.end(), dereference<int>() );
///
/// copy( V2.begin(), V2.end(), ostream_iterator<int>( cout, " " ) );
/// cout << endl;
///  
/// The preceding code will produce the following output:
///
/// 1 2 3 4 5 6 7 8 9 10
/// 1 2 3 4 5 6 7 8 9 10
///
/// Note that address_of<T> is nearly always inexpensive, while 
/// dereference<T> calls the copy constructor of T.
 
template <class T>
struct address_of : public unary_function<T, T*> 
{
  const T* operator() (const T& x ) const { return &x; }
  T* operator() ( T& x ) const { return &x; }
};

template <class T>
struct dereference : public unary_function<T*, T> 
{
  T operator() (const T* x ) const { return *x; }
};


// g++ 3.x defines select1st and select2nd in <ext/functional>

#if __GNUC__ > 2
#include <ext/functional>
using __gnu_cxx::select1st;
using __gnu_cxx::select2nd;
#endif

// cxx does not define select1st and select2nd, so we provide them here.

#if defined( __DECCXX_VER )

/*
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Hewlett-Packard Company makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 *
 * Copyright (c) 1996
 * Silicon Graphics Computer Systems, Inc.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Silicon Graphics makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */

template <class T_Pair>
struct Select1st : public unary_function<T_Pair, typename T_Pair::first_type> {
  const typename T_Pair::first_type& operator()(const T_Pair& __x) const {
    return __x.first;
  }
};

template <class T_Pair>
struct Select2nd : public unary_function<T_Pair, typename T_Pair::second_type>

  const typename T_Pair::second_type& operator()(const T_Pair& __x) const {
    return __x.second;
  }
};

template <class T_Pair> 
struct select1st
    : public Select1st<T_Pair> {};
template <class T_Pair> 
struct select2nd 
    : public Select2nd<T_Pair> {};

/*
 * End HP/SGI copyrighted material.
 */

#endif

// g++ 3.x defines is_sorted in <ext/algorithm>
// g++ 3.x defines iota in <ext/numeric>

#if __GNUC__ > 2
#include <ext/algorithm>
using __gnu_cxx::is_sorted;
#include <ext/numeric>
using __gnu_cxx::iota;
#else
#if __GNUC__ <= 2
#include <algorithm>
#include <numeric>
#endif
#endif

// cxx does not have iota or is_sorted, so we provide it here.

#if defined( __DECCXX_VER )

template <class ForwardIterator, class T>
void iota(ForwardIterator first, ForwardIterator last, T value) {
  while (first != last) *first++ = value++;
}

/*
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Hewlett-Packard Company makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 *
 * Copyright (c) 1996
 * Silicon Graphics Computer Systems, Inc.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Silicon Graphics makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */

/// is_sorted, a predicated testing whether a range is sorted in
/// nondescending order.  This is an extension, not part of the C++
/// standard.

template <class _ForwardIter>
bool is_sorted(_ForwardIter __first, _ForwardIter __last)
{
  if (__first == __last)
    return true;

  _ForwardIter __next = __first;
  for (++__next; __next != __last; __first = __next, ++__next) {
    if (*__next < *__first)
      return false;
  }

  return true;
}

/// is_sorted, a predicated testing whether a range is sorted in
/// nondescending order.  This is an extension, not part of the C++
/// standard.
template <class _ForwardIter, class _StrictWeakOrdering>
bool is_sorted(_ForwardIter __first, _ForwardIter __last,
               _StrictWeakOrdering __comp)
{
  if (__first == __last)
    return true;

  _ForwardIter __next = __first;
  for (++__next; __next != __last; __first = __next, ++__next) {
    if (__comp(*__next, *__first))
      return false;
  }

  return true;
}


/*
 * End HP/SGI copyrighted material.
 */

#endif

///returns true if the order is strictly ascending.
template <class _ForwardIter>
bool is_sorted_strict(_ForwardIter __first, _ForwardIter __last)
{
  if (__first == __last)
    return true;

  _ForwardIter __next = __first;
  for (++__next; __next != __last; __first = __next, ++__next) {
    if (*__next <= *__first)
      return false;
  }

  return true;
}

///returns true if the order is strictly ascending.
template <class _ForwardIter, class _StrictWeakOrdering>
bool is_sorted_strict(_ForwardIter __first, _ForwardIter __last,
               _StrictWeakOrdering __comp)
{
  if (__first == __last)
    return true;

  _ForwardIter __next = __first;
  for (++__next; __next != __last; __first = __next, ++__next) {
    if (!(__comp(*__first, *__next) ) )
      return false;
  }

  return true;
}


/// copy_if is an algorithm.
///
/// copy_if is so tremendously useful, it's hard to understand why it's
/// not in the standard.  For notes on this implementation, see
/// Effective STL (Meyers, 2001), Item 36, pp.154-156.

template<typename InputIterator, typename OutputIterator, typename Predicate>
OutputIterator copy_if( InputIterator begin, InputIterator end,
			OutputIterator destBegin, Predicate p )
{
  while ( begin != end )
  {
    if ( p(*begin) )
      *destBegin++ = *begin;
    ++begin;
  }

  return destBegin;
}


/// dereference_compare is a functor which allows comparison of
/// iterators using the operator< of the objects they point to.
///
/// usage example: suppose Foo::operator< exists.  Then:
///   list<Foo> foo_list;
///   vec< list<Foo>::iterator > foo_iters;
///   sort( foo_iters.begin(), foo_iters.end(), 
///         dereferenced_compare<list<Foo>::iterator>() );

template<typename Iterator>
struct dereferenced_compare : 
  public binary_function<Iterator, Iterator, bool>
{
  bool operator()(Iterator lhs, Iterator rhs) const {
    return( *lhs < *rhs );
  }
};

/*
   Template: update_min

   Update a running minimum: compare a value to the current minimum and update the minimum if the value is smaller.
 */
template <typename NumericType>
void update_min( NumericType& currentMin, const NumericType& val ) {
  if (val < currentMin )
    currentMin = val;
}

/*
   Template: update_max

   Update a running maximum: compare a value to the current maximum and update the maximum if the value is smaller.
 */
template <typename NumericType>
void update_max( NumericType& currentMax, const NumericType& val ) {
  if (val > currentMax )
    currentMax = val;
}

#define ForEach_Mut(x,containerType,c) \
    for ( containerType::iterator x = c.begin(); x != c.end(); x ++ ) 

#define ForEach(x,containerType,c) \
    for ( containerType::const_iterator x = c.begin() ; x != c.end(); x ++ )

template <class CONTAINER, class T> inline
  bool STLContains( const CONTAINER& c, const T& x ) {
  return c.find(x) != c.end();
}


#endif
