/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/**
\file FeudalTemplate.h
This file contains the code which needs to be compiled for each instantiation
of class mastervec.
*/


#ifndef FEUDAL_TEMPLATE
#define FEUDAL_TEMPLATE

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "CoreTools.h"
#include "Feudal.h"
#include "FeudalReadPart.h"

#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <math.h>

#include <sys/mman.h>

//#define DEBUG_PRINT( thing ) cerr << #thing << ": " << thing << endl

template<class X, class A> 
mastervec<X,A>::mastervec( )
  : raw_data_size_( 0 ),
    raw_data_capacity_( 0 ),
    raw_data_( 0 ),
    objects_( 0 ),
    objects_size_( 0 ),
    objects_capacity_( 0 ),
    fd_( -1 )
{
}


template<class X, class A> 
mastervec<X,A>::mastervec( int n )
  : raw_data_size_( 0 ),
    raw_data_capacity_( 0 ),
    raw_data_( 0 ),
    objects_( new X[n] ),
    objects_size_( n ),
    objects_capacity_( n ),
    fd_( -1 )
{
}


template<class X, class A> 
mastervec<X,A>::mastervec( const String &filename )
  : raw_data_size_( 0 ),
    raw_data_capacity_( 0 ),
    raw_data_( 0 ),
    objects_( 0 ),
    objects_size_( 0 ),
    objects_capacity_( 0 ),
    fd_( -1 )
{
  ReadRange( filename, 0, MastervecFileObjectCount( filename ) );
}


template<class X, class A> 
mastervec<X,A>::~mastervec( )
{ 
  if ( fd_ >= 0 ) {
    int rc = munmap( raw_data_, raw_data_size_ * sizeof(A) );
    if ( rc != 0 ) {
      cout << "munmap() failed: " << ErrnoExplanation( errno ) << endl;
      PRINT3( fd_, raw_data_, raw_data_size_ );
      TracebackThisProcess();
    }
    close( fd_ );
    raw_data_ = 0;
    raw_data_size_ = 0;
    raw_data_capacity_ = 0;
    fd_ = -1;
  }
  else {
    if ( raw_data_ != 0 ) { 
      delete [ ] raw_data_;
      raw_data_= 0;
      raw_data_size_ = 0;
      raw_data_capacity_ = 0;   
    }
  }

  if ( objects_ != 0 ) {   
    delete [ ] objects_;
    objects_ = 0;
    objects_size_ = 0;
    objects_capacity_ = 0;    
  }
}


template<class X, class A> 
void
mastervec<X,A>::destroy( )
{
  if ( fd_ >= 0 ) {
    int rc = munmap( raw_data_, raw_data_size_ * sizeof(A) );
    if ( rc != 0 ) {
      cout << "munmap() failed: " << ErrnoExplanation( errno ) << endl;
      PRINT3( fd_, raw_data_, raw_data_size_ );
      TracebackThisProcess();
    }
    close( fd_ );
    raw_data_ = 0;
    raw_data_size_ = 0;
    raw_data_capacity_ = 0;
    fd_ = -1;
  }
  else {
    if ( raw_data_ != 0 ) {
      delete [ ] raw_data_;
      raw_data_= 0;
      raw_data_size_ = 0;
      raw_data_capacity_ = 0;  
    }
  }
  fd_ = -1;

  if ( objects_ != 0 ) { 
    delete [ ] objects_;
    objects_ = 0;
    objects_size_ = 0;
    objects_capacity_ = 0;   
  }
}

template<class X, class A> 
void
mastervec<X,A>::clear( )
{
  ForceAssertLt( fd_, 0 );
  raw_data_size_ = 0;
  for ( int i = 0; i < objects_size_; i++ )
    objects_[i].Reinitialize( );
  objects_size_ = 0;   
}


template<class X, class A> 
void
mastervec<X,A>::resize( int n )
{
  if ( n <= objects_capacity_ ) {
    for ( int i = n; i < objects_size_; i++ )
      objects_[i].Reinitialize( );  
  }
  else {
    X* new_objects_data = new X[n];
    memcpy( new_objects_data, objects_, objects_size_ * sizeof(X) );
    for ( int i = 0; i < objects_size_; i++ )
      objects_[i].Blank( );
    delete [ ] objects_;
    objects_ = new_objects_data;
    objects_capacity_ = n;   
  }
  objects_size_ = n;   
}

template<class X, class A> 
void
mastervec<X,A>::reserve( int n )
{
  if (n > objects_capacity_)
    reserveCleanup(reserveInternal(n));
}

template<class X, class A>
void mastervec<X,A>::reserveCleanup( pair<int, X *> cleanup ) {
  for ( int i = 0; i < cleanup.first; i++ )
    cleanup.second[i].Blank( );
  delete [] cleanup.second;
}

template<class X, class A> 
pair<int, X *> mastervec<X,A>::reserveInternal( int n )
{
  //The following check should be made outside the method
  //to avoid generating and copying a lot of null pairs.
  AssertGt(n,objects_capacity_);

  pair<int, X*> ret(0,0);
  X* new_objects_data = new X[n];
  memcpy( new_objects_data, objects_, objects_size_ * sizeof(X) );
  X * toDelete = objects_;
  objects_ = new_objects_data;
  objects_capacity_ = n;
  return make_pair(objects_size_, toDelete);
}

template<class X, class A> 
A * mastervec<X,A>::ReserveInternal( longlong raw_mem_size, int n_objects) {
  if ( raw_mem_size < 1 )
    raw_mem_size = 1; // always allocate at least 1 byte
  if ( raw_mem_size <= raw_data_capacity_ ) 
    return 0; //no need to reserve.

  A* new_raw_data = new A[raw_mem_size];
  SafeMemcpy( new_raw_data, raw_data_, raw_data_size_ * sizeof(A) );
  for ( int i = 0; i < objects_size_; i++ )
    if ( !objects_[i].SelfOwned( ) )
      objects_[i].ShiftStartOfDynamicData( new_raw_data, raw_data_ );
  A * toDelete = raw_data_;
  raw_data_ = new_raw_data; 
  raw_data_capacity_ = raw_mem_size;   
  return toDelete;   
}

template<class X, class A> 
void
mastervec<X,A>::Reserve( longlong raw_mem_size, int n_objects )
{
  A * toDelete = ReserveInternal(raw_mem_size, n_objects);  
  mastervec<X,A>::reserve(n_objects);  
  delete [] toDelete;
}

template<class X, class A>
void
mastervec<X,A>::push_back_reserve( const X& obj, int extra_space, 
                                   double increase ) {
  static const longlong LIMIT= (longlong)1000000 * (longlong) 1000000;
  ForceAssertLt( raw_data_size_, LIMIT);
  A * adel = 0;
  pair<int, X *> resret(0,0);

  //extend capacity if needed
  longlong dynabytes = obj.SizeOfDynamicData();
  longlong sizeneeded = rawsize() + dynabytes + longlong(extra_space);
  if (sizeneeded > rawcapacity()) {
    adel = ReserveInternal( longlong(increase * sizeneeded), 0 );
  }
  if (capacity() <= size()) {
    resret = reserveInternal(int(ceil(increase * capacity() + 1)));
  }

  //copy in the object
  objects_[ objects_size_ ].SetExternal( obj.StartOfStaticData( ),
       raw_data_ + raw_data_size_, dynabytes, extra_space );
  memcpy( raw_data_ + raw_data_size_, obj.StartOfDynamicData( ),
          dynabytes * sizeof(A) );


  //adjust sizes and delete old data.  
  ++objects_size_;
  raw_data_size_ += dynabytes + extra_space;
  ForceAssertLt( fd_, 0 );
  reserveCleanup(resret);
  delete [] adel;
} 
  


/**
\todo make this more efficient by copying the internal memory in one
big block
*/
template<class X, class A>
void
mastervec<X,A>::Append( const mastervec & orig, int from, int to,
                        longlong extraRaw, int extraSize) {
  AssertGe(from,0);
  AssertGe(to,from);
  AssertGe(orig.size(),to);
  extraRaw += rawsize() + orig.rawsize(from,to);
  extraSize += size() + to-from;
  Reserve(extraRaw, extraSize);
  for (int i = from; i != to; ++i) {
    push_back(orig[i], orig[i].ExtraSpaceForDynamicData());
  }
}

template<class X, class A>
void
mastervec<X,A>::Append( const mastervec & orig, const vec<int> & entries,
                        longlong extraRaw, int extraSize) {
  extraRaw += rawsize() + orig.rawsize(entries);
  extraSize += size() + entries.size();
  Reserve(extraRaw, extraSize);
  int end = entries.size();
  for (int i = 0; i != end; ++i) {
    push_back(orig[entries[i]], orig[entries[i]].ExtraSpaceForDynamicData());
  }
}

template<class X, class A> 
void 
mastervec<X,A>::Append( const mastervec & orig ) {
  longlong extraRaw = rawsize() + orig.rawsize( );
  longlong extraSize = size() + orig.size( );
  Reserve(extraRaw, extraSize);
  for (int i = 0; i != orig.size( ); ++i) {
    push_back_reserve(orig[i]);
  }
}

template<class X, class A>
longlong
mastervec<X,A>::rawsize( int from, int to ) const
{ 
  AssertGe(from,0);
  AssertGe(to,from);
  AssertGe(objects_size_,to);
  if (from == to) return 0; //needed to avoid problems in case of (0,0).
  vec<int> entries(to-from);
  for (int i = from; i != to; ++i) entries[i-from] = i;
  return rawsize(entries);
}

template<class X, class A>
longlong
mastervec<X,A>::rawsize( const vec<int> & entries ) const
{ 
  longlong ret = 0;
  int end = entries.size();
  for (int i = 0; i != end; ++i) {
    ret += objects_[i].SizeOfDynamicData();
    ret += objects_[i].ExtraSpaceForDynamicData();
  }
  return ret * sizeof(A);
}

template<class X, class A>
void
mastervec<X,A>::Write( const String& filename,
		       int from,
		       int to ) const
{    
  // Define filenames.

  const String& filename1 = filename;
  String filename2 = filename + "..offsets";
  String filename3 = filename + "..static";

     // Set control bytes of filename1.

  if ( !IsRegularFile(filename1) ) { 
    ForceAssert( !IsRegularFile(filename2) && !IsRegularFile(filename3) );
    mv_file_control_block header;
    header.n = to-from;
    header.nfiles(3);
    header.SetVecSize(sizeof(X));
    header.SetDataSize(sizeof(typename X::value_type));
    int fd1 = OpenForWrite( filename1, 0666 );
    WriteBytes( fd1, &header, 24 );
    close(fd1);
    int fd2 = OpenForWrite( filename2, 0666 );
    longlong f1size = FileSize(filename1);
    WriteBytes( fd2, &f1size, 8 );
    close(fd2);   
  }
  else {
    int fd1 = OpenForRead( filename1 );
    mv_file_control_block header;
    read( fd1, &header, 24 );
    close(fd1);
    ForceAssertEq( header.nfiles(), 3 );
    header.n += to - from;
    header.SetVecSize(sizeof(X));
    header.SetDataSize(sizeof(typename X::value_type));
    fd1 = Open( filename1, O_WRONLY);
    WriteBytes( fd1, &header, 24 );
    close(fd1);   
  }

  // Open files.

  int fd1 = Open( filename1, O_WRONLY ^ O_APPEND );
  int fd2 = Open( filename2, O_WRONLY ^ O_APPEND );
  int fd3 = Open( filename3, O_WRONLY ^ O_CREAT ^ O_APPEND, 0666 );

  // Determine static block size.

  X an_X;
  const int static_block_size = an_X.SizeOfStaticData( );

  // Write the data.

  const int batch_size = 512;
  longlong offset[batch_size];
  longlong current_offset = FileSize(filename1);
  for ( int i = from; i < to; i += batch_size ) {
    int this_batch_size = min( batch_size, to - i );
    for ( int j = 0; j < this_batch_size; j++ ) {
      int dynabytes = objects_[i+j].SizeOfDynamicData( ) * sizeof(A);
      current_offset += dynabytes;
      offset[j] = current_offset;   
    }
    WriteBytes( fd2, (char*) &offset, this_batch_size * 8 );   
  }

  char* statics = new char[batch_size * static_block_size];
  if ( static_block_size != 0 ) { 
    for ( int i = from; i < to; i += batch_size ) {
      int this_batch_size = min( batch_size, to - i );
      for ( int j = 0; j < this_batch_size; j++ )
        memcpy( statics + static_block_size * j,
                objects_[i+j].StartOfStaticData( ), static_block_size );
      WriteBytes( fd3, statics, 
                  this_batch_size * static_block_size );  
    } 
  }
  delete [] statics;

  static vector<A> dynabuffer;
  longlong maxdyna = 0;
  for ( int i = from; i < to; i += batch_size ) { 
    int this_batch_size = min( batch_size, to - i );
    longlong thisdyna = 0;
    for ( int j = 0; j < this_batch_size; j++ )
      thisdyna += objects_[i+j].SizeOfDynamicData( );
    maxdyna = max( maxdyna, thisdyna );  
  }
  dynabuffer.reserve(maxdyna);
  for ( int i = from; i < to; i += batch_size ) {  
    int this_batch_size = min( batch_size, to - i );
    char* dynaptr = (char*) &dynabuffer[0];
    for ( int j = 0; j < this_batch_size; j++ ) {  
      int dynabytes = objects_[i+j].SizeOfDynamicData( ) * sizeof(A);
      if ( dynabytes > 0 )
        memcpy( dynaptr, (char*) objects_[i+j].StartOfDynamicData( ), 
                dynabytes );
      dynaptr += dynabytes;  
    }
    char* target = (char*) &dynabuffer[0];
    WriteBytes( fd1, target, dynaptr - target );  
  }

  // Close files.

  close(fd1);
  close(fd2);
  close(fd3);  
}

template<class X, class A>
void
mastervec<X,A>::Read( const String& filename,
		      const vec<int>& entries, 
		      int extra,
		      Bool pre_reserved,
		      Bool pre_resized,
                      Bool acceptCompFile)
{    
  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  if (!acceptCompFile) AssertEq(control.isCompFile(), false);
  X * dummy=0;
  if ( !IsGoodFeudalFile(filename, dummy, true, true) )
    FatalErr( filename << " is not in feudal (mastervec) format." );

  // Are entries garbage?

  for ( unsigned int i = 0; i < entries.size( ); i++ ) {  
    ForceAssertGe( entries[i], 0 );
    ForceAssertLt( entries[i], control.n );
    //This assert requires that entries be (almost) sorted ascending, and 
    // appears to not be strictly required for proper functioning. If it were
    // required, it should be i > 0. In fact, the method seems to work fine 
    // without the assert for small unordered entries vectors. 
    // However, it is probably less efficient with unsorted entries.
    if ( i > 1 )
      ForceAssert( entries[i] > entries[i-1] );  
  }

  // Open files.

  const String& filename1 = filename;
  String filename2 = filename + "..offsets";
  String filename3 = filename + "..static";
  int fd1, fd2, fd3;
  fd1 = Open( filename1, O_RDONLY );
  if ( control.nfiles() == 3 ) {  
    fd2 = Open( filename2, O_RDONLY );
    fd3 = Open( filename3, O_RDONLY );   
  }
  else
    fd2 = fd3 = fd1;
  read_part m1(fd1), m2(fd2), m3(fd3);

  // Reserve space.

  if ( !pre_reserved ) {
    longlong new_rawmem = 0, offseta, offsetb;
    for ( int i = 0; i < (int) entries.size( ); i++ ) { 
      m2.Fetch( control.offsets_start + (longlong) entries[i] * (longlong) 8, 8, 
                (char*) &offseta );
      int j;
      for ( j = i + 1; j < (int) entries.size( ); j++ )
        if ( entries[j] != entries[j-1] + 1 )
          break;
      m2.Fetch( control.offsets_start 
                + (longlong) (entries[j-1] + 1) * (longlong) 8, 8, (char*) &offsetb );
      new_rawmem += (offsetb - offseta) / sizeof(A);
      i = j - 1;    
    }
    Reserve( raw_data_size_ + new_rawmem, 
             objects_size_ + entries.size( ) );   
  }
     
  int next_object_idx;
  if ( !pre_resized ) {
    next_object_idx = objects_size_;
    resize( objects_size_ + entries.size() );
  }
  else
    next_object_idx = 0;

  // Determine static block size.

  X an_X;
  const int static_block_size = an_X.SizeOfStaticData( );

  // Read in the data.

  const int batch_size = 512;
  longlong offset[ batch_size + 1 ];
  char* static_data = new char[ batch_size * static_block_size ];
  for ( int i = 0; i < (int) entries.size( ); i++ ) {  
    int top = min( i + batch_size, (int) entries.size( ) );
    int j;
    for ( j = i + 1; j < top; j++ )
      if ( entries[j] != entries[j-1] + 1 ) 
        break;
    int this_batch_size = j - i;
    
    // Get the offsets and dynamic block size.  Reserve space for dynamic data.
    
    m2.Fetch( control.offsets_start + (longlong) entries[i] * (longlong) 8,
              (this_batch_size + 1) * 8, (char*) offset );
    longlong dynabytes = offset[this_batch_size] - offset[0];
    longlong new_raw_data_size = raw_data_size_ + dynabytes/sizeof(A);
    
    // Get the dynamic data.
    
    m1.Fetch( offset[0], dynabytes, (char*) (raw_data_ + raw_data_size_) );
    
    // Get the static data.
    
    if ( static_block_size > 0 )
      m3.Fetch( control.static_start 
                + (longlong) entries[i] * (longlong) static_block_size,
                this_batch_size * static_block_size, static_data );
    
    // Create objects.
    
    for ( int k = 0; k < this_batch_size; k++ ) { 
      if ( !( offset[k] <= offset[k+1] ) )
        cout << "offset problem reading " << filename << endl;
      ForceAssertLe( offset[k], offset[k+1] );
      objects_[next_object_idx++].
        SetExternal( &static_data[static_block_size * k], 
                     raw_data_ + raw_data_size_ + (offset[k] - offset[0]) / sizeof(A),
                     (offset[k+1] - offset[k]) / sizeof(A), extra );  
    }

    // Update raw_data_size_.

    ForceAssertLt( fd_, 0 );
    raw_data_size_ = new_raw_data_size;
    
    i = j - 1;   
  }
  
  delete [ ] static_data;

  // Close files.

  close(fd1);
  if ( control.nfiles() == 3 ) {
    close(fd2);
    close(fd3);  
  } 
}


template<class X, class A>
void
mastervec<X,A>::SparseRead( const String& filename,
			    const vec<int>& entries, 
			    int extra,
			    Bool pre_reserved,
                            Bool acceptCompFile)
{    
  int num_objects_to_save = min( entries.size(), (size_t) this->size() );

  // Save objects that could be overwritten.

  X* saved_objects = new X[num_objects_to_save];
  for ( int i = 0; i < num_objects_to_save; ++i )
    objects_[i].Swap( saved_objects[i] );
       
  // Resize the vector of objects to the size of the full vector.

  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  if ( control.n > this->size() )
    this->resize( control.n );

  // Read in the entries.

  A* old_raw_data = raw_data_;
  this->Read( filename, entries, extra, pre_reserved, True, acceptCompFile );

  // Move the entries (which will be in indices 0 to entries.size()
  // of objects_) into their real places.  We start from the end to
  // avoid overwriting not-yet-moved entries.

  for ( int i = (int) entries.size() - 1; i >= 0; --i )
    objects_[ entries[i] ].Swap( objects_[ i ] );

  // Adjust the pointers for the saved objects, if necessary.
  for ( int i = 0; i < num_objects_to_save; i++ )
    if ( !saved_objects[i].SelfOwned( ) )
      saved_objects[i].ShiftStartOfDynamicData( raw_data_, old_raw_data );

  // Restore the saved objects, except ones that should be overwritten.
  unsigned int entry_idx = 0;
  for ( int i = 0; i < (int) num_objects_to_save; ++i ) {
    if ( i > entries[entry_idx] )
      ++entry_idx;
    if ( i != entries[entry_idx] )
      objects_[i].Swap( saved_objects[i] );
  }
     
  delete [] saved_objects;
}


template<class X, class A>
void
mastervec<X,A>::ReadRange( const String& filename,
			   int from,
			   int to,
			   int extra,
			   Bool pre_reserved,
			   Bool pre_resized,
                           Bool acceptCompFile)
{
  static vec<int> entries;
  entries.resize( to - from );
  for ( int i = from; i < to; i++ )
    entries[i-from] = i;
  Read( filename, entries, extra, pre_reserved, pre_resized, acceptCompFile);  
}


template<class X, class A>
void
mastervec<X,A>::SparseReadRange( const String& filename,
				 int from,
				 int to,
				 int extra,
				 Bool pre_reserved,
                                 Bool acceptCompFile )
{
  static vec<int> entries;
  entries.resize( to - from );
  for ( int i = from; i < to; i++ )
    entries[i-from] = i;
  SparseRead( filename, entries, extra, pre_reserved, acceptCompFile );   
}



template<class X, class A> 
void
mastervec<X,A>::ReadAll( const String& filename,
			 Bool append,
                         Bool acceptCompFile )
{    
  if ( !IsRegularFile(filename) && !IsRegularFile( filename + ".gz" ) )
    FatalErr( "Neither " << filename << " nor " << filename << ".gz exists." );
  if ( IsRegularFile(filename) && IsRegularFile( filename + ".gz" ) )
    FatalErr( "Both " << filename << " and " << filename << ".gz exist."
	      << "  This confuses me." );

  if ( IsRegularFile( filename + ".gz" ) )
    System( "gzip -d " + filename + ".gz" );

  X * dummy=0;
  if (!IsGoodFeudalFile(filename, dummy, true, true) ) {
      FatalErr( "ReadAll: It would appear that " << filename << "\n"
		<< "is not in proper mastervec format.\n");
  }

  int fd = Open( filename, O_RDONLY );
  mv_file_control_block control;
  read( fd, &control, sizeof(control) );
  	      
  if ( control.nfiles() != 1 )
    FatalErr( "ReadAll: " << filename << " was expected to be in "
	      << "mastervec single-file format, but it has "
	      << "control.nfiles() = 3." );

  lseek( fd, control.offsets_start, SEEK_SET );
  longlong offset0, offsetn;
  read( fd, &offset0, 8 );
  lseek( fd, control.offsets_start + (longlong) 8 * (longlong) control.n, SEEK_SET );
  read( fd, &offsetn, 8 );
  close(fd);
  if ( !append ) clear( );
  ForceAssertLe( offset0, offsetn );
  Reserve( raw_data_size_ + ( (offsetn - offset0) / sizeof(A) ), 
	   size( ) + control.n );
  ReadRange( filename, 0, control.n, 0, True, False, acceptCompFile );  
}

template<class X, class A> 
void 
mastervec<X,A>::WriteAll( const String& filename ) const
{
  Remove( filename );
  Remove( filename + ".gz" );
  Remove( filename + "..offsets" );
  Remove( filename + "..static" );
  Write( filename, 0, size( ) );
  MergeMastervecFiles( filename );  
}

template<class X, class A>
void
mastervec<X,A>::MapAll( const String& filename ) 
{
  clear();

  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  Assert(!control.isCompFile());
  if ( control.nfiles() != 1 && control.nfiles() != 3 )
    FatalErr( filename << " is not in feudal (mastervec) format." );

  // Open files.

  const String& filename1 = filename;
  String filename2 = filename + "..offsets";
  String filename3 = filename + "..static";
  int fd2, fd3;
  fd_ = Open( filename1, O_RDONLY );
  if ( control.nfiles() == 3 ) {  
    fd2 = Open( filename2, O_RDONLY );
    fd3 = Open( filename3, O_RDONLY );   
  }
  else
    fd2 = fd3 = fd_;
  read_part m2(fd2), m3(fd3);

  // mmap() the dynamic data.
  m2.Fetch( control.offsets_start + (longlong) control.n * (longlong) 8,
            8, (char*) &raw_data_size_ );
  raw_data_ = (A*) mmap( 0, raw_data_size_, PROT_READ, MAP_SHARED, fd_, 0 );
  raw_data_size_ /= sizeof(A);

  if ( raw_data_ == 0 ) {
    cout << "mmap() failed: " << ErrnoExplanation( errno ) << endl;
    TracebackThisProcess();
  }

  // Reserve space.

  resize( control.n );

  int next_object_idx = 0;

  // Determine static block size.

  X an_X;
  const int static_block_size = an_X.SizeOfStaticData( );

  // Read in the data.

  const int batch_size = getpagesize();
  longlong* offset = new longlong[ batch_size + 1 ];
  char* static_data = new char[ batch_size * static_block_size ];
  for ( int i = 0; i < control.n; i += batch_size ) {  
    int top = min( i + batch_size, (int) control.n );
    int this_batch_size = top - i;
    
    // Get the offsets and dynamic block size.
    
    m2.Fetch( control.offsets_start + (longlong) i * (longlong) 8,
              (this_batch_size + 1) * 8, (char*) offset );
    
    // Get the static data.
    
    if ( static_block_size > 0 )
      m3.Fetch( control.static_start + (longlong) i * (longlong) static_block_size,
                this_batch_size * static_block_size, static_data );
    
    // Create objects.
    
    for ( int k = 0; k < this_batch_size; k++ ) { 
      if ( !( offset[k] <= offset[k+1] ) )
        cout << "offset problem reading " << filename << endl;
      ForceAssertLe( offset[k], offset[k+1] );
      objects_[next_object_idx++].
        SetExternal( &static_data[static_block_size * k], 
                     raw_data_ + ( offset[k] / sizeof(A) ),
                     (offset[k+1] - offset[k]) / sizeof(A), 
                     0 );  
    }
  }
  
  delete [ ] offset;
  delete [ ] static_data;

  // Close files.

  if ( control.nfiles() == 3 ) {
    close(fd2);
    close(fd3);  
  } 
}


template<class X, class A> 
longlong
mastervec<X,A>::MastervecFileRawCount( const String& filename ) 
{
  return ::MastervecFileRawCount(filename, sizeof(A));
}

template<class X>
void
GetOneFeudal(const String & filename, int index, 
             X & value, bool acceptCompFile) 
{
  mastervec<X, typename X::value_type> v;
  v.ReadOne(filename, index, 0, false, false, acceptCompFile);
  value = v[0];
}
  
template<class X, class A> 
mastervec<X,A>&
mastervec<X,A>::operator=( const mastervec& x )
{ 
  clear( );
  Reserve( x.actual_rawsize( ), x.size( ) );
  for ( int i = 0; i < x.size( ); i++ )
    push_back( x[i] );
  return *this;   
}
  
template<class X, class A> 
mastervec<X,A>::mastervec( const mastervec& x )
  : raw_data_size_( 0 ),
    raw_data_capacity_( 0 ),
    raw_data_( 0 ),
    objects_( 0 ),
    objects_size_( 0 ),
    objects_capacity_( 0 ),
    fd_( -1 )
{
  if ( ! x.empty() ) {
    Reserve( x.actual_rawsize( ), x.size( ) );
    for ( int i = 0; i < x.size( ); i++ )
      push_back( x[i] );
  }
}

template<class X>
bool
IsGoodFeudalFile( const String & filename, const X * dummy, bool verbose,
		  bool ok3) 
{
  if (!IsGoodFeudalFile(filename, verbose, ok3)) return false;

  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  if (control.IsOldFormat()) return true;

  if ( control.vecSize() != sizeof(X)) {
    if (verbose) { 
      cerr << "IsGoodFeudalFile: Failed vecSize test" << endl;
      PRINT2(int(control.vecSize()), sizeof(X));
    }
    return false;
  }
  if ( control.dataSize() != sizeof(typename X::value_type)) {
    if (verbose) {
      cerr << "IsGoodFeudalFile: Failed dataSize test" << endl;
      PRINT2(int(control.dataSize()), sizeof(typename X::value_type));
    }
    return false;
  }  
  return true;
}

template<class X, class A> 
void
mastervec<X,A>::BinaryRead( int fd )
{    
  destroy( );
  ReadBytes( fd, &raw_data_size_, sizeof(longlong) );
  if ( raw_data_size_ > (longlong)(1000 * 1000) * (longlong)(1000 * 1000) ) {
    FatalErr( "mastervec::BinaryRead sees raw_data_size_ = "
              << raw_data_size_ << ", possibly a sign of file corruption." );    
  }
  ReadBytes( fd, &objects_size_, sizeof(int) );
  A* raw_data_old = 0;
  ReadPointer( fd, raw_data_old );
  raw_data_ = new A[raw_data_size_];
  ReadBytes( fd, raw_data_, raw_data_size_ * sizeof(A) );
  objects_ = new X[objects_size_];
  ReadBytes( fd, objects_, objects_size_ * sizeof(X) );
  for ( int i = 0; i < objects_size_; i++ )
    (*this)[i].ShiftStartOfDynamicData( raw_data_, raw_data_old );    
}

template<class X, class A> 
void 
mastervec<X,A>::BinaryWrite( int fd ) const
{    
  for ( int i = 0; i < size( ); i++ )
    ForceAssert( !( (*this)[i].SelfOwned( ) ) );
  WriteBytes( fd, &raw_data_size_, sizeof(longlong) );
  WriteBytes( fd, &objects_size_, sizeof(int) );
  WritePointer( fd, raw_data_ );
  WriteBytes( fd, raw_data_, raw_data_size_ * sizeof(A) );
  WriteBytes( fd, objects_, objects_size_ * sizeof(X) );    
}

template<class X, class A>
void
mastervec<X,A>::QuickSort( const int lower, const int upper )
{
  if ( upper - lower < qsort_cutoff )
    return;
  this->SwapElements( lower, lower + random() % ( upper - lower + 1 ) );
  int i = lower, j = upper+1;
  while ( 1 ) {
    do ++i; while ( i <= upper && objects_[i] < objects_[lower] );
    do --j; while ( objects_[lower] < objects_[j] );
    if ( i > j )
      break;
    this->SwapElements( i, j );
  }
  this->SwapElements( lower, j );
  QuickSort( lower, j-1 );
  QuickSort( j+1, upper );
}



// ======================= INSTANTIATIONS ======================

// Here is an example of how to instantiate a feudal template.

// #ifdef __DECCXX_VER
// #pragma define_template mastervec<serfvec<int>, int>
// #else
// INSTANTIATE_MASTERVEC( serfvec<int>, int )
// #endif

#ifndef __DECCXX_VER

#define INSTANTIATE_MASTERVEC(X,A)                                            \
     template void mastervec< X, A >::Reserve( longlong raw_mem_size,         \
          int n_objects );                                                    \
     template mastervec<X,A>::mastervec( );                                   \
     template mastervec<X,A>::mastervec( int n );                             \
     template mastervec<X,A>::mastervec( const String &filename );            \
     template mastervec<X,A>::mastervec( const mastervec& x );                \
     template mastervec<X,A>::~mastervec( );                                  \
     template mastervec<X,A>& mastervec<X,A>::operator=( const mastervec& x );\
     template void mastervec<X,A>::destroy( );                                \
     template void mastervec<X,A>::clear( );                                  \
     template void mastervec<X,A>::resize( int n );                           \
     template void mastervec<X,A>::reserve( int n );                          \
     template void mastervec<X,A>::push_back( const X& obj, int extra_space );\
     template void mastervec<X,A>::push_back_reserve( const X& obj,           \
                   int extra_space, double increase );                        \
     template void mastervec<X,A>::BinaryRead( int fd );                      \
     template void mastervec<X,A>::BinaryWrite( int fd ) const;               \
     template void mastervec<X,A>::Append( const mastervec<X,A> & orig,       \
int from, int to, longlong extraRaw, int extraSize);                          \
     template void mastervec<X,A>::Append( const mastervec<X,A> & orig,       \
const vec<int> & entries, longlong extraRaw, int extraSize);                  \
     template void mastervec<X,A>::Append( const mastervec<X,A> & orig );     \
     template longlong mastervec<X,A>::rawsize( int from, int to ) const;     \
     template longlong mastervec<X,A>::rawsize( const vec<int> & entries )    \
              const;\
     template void mastervec<X,A>::Read( const String& filename,              \
const vec<int>& entries, int extra, Bool pre_reserved, Bool pre_resized,      \
Bool acceptCompFile );                                                        \
     template void mastervec<X,A>::SparseRead( const String& filename,        \
const vec<int>& entries, int extra, Bool pre_reserved, Bool acceptCompFile  );\
     template void mastervec<X,A>::ReadRange( const String& filename, int to, \
int from, int extra, Bool pre_reserved, Bool pre_resized,Bool acceptCompFile);\
     template void mastervec<X,A>::SparseReadRange( const String& filename,   \
          int to, int from,int extra, Bool pre_reserved, Bool acceptCompFile);\
     template void mastervec<X,A>::Write( const String& filename, int from,   \
          int to ) const;                                                     \
     template void mastervec< X, A >::ReadAll( const String& filename,        \
          Bool append, Bool acceptCompFile  );                                \
     template void mastervec< X, A >::WriteAll(const String& filename ) const;\
     template longlong mastervec< X, A >::                                    \
          MastervecFileRawCount( const String& filename );                    \
template void GetOneFeudal<X>(const String& filename, int index, X & value,   \
          bool acceptCompFile);                                               \
template bool IsGoodFeudalFile<X>(const String & filename, const X * dummy,   \
          bool verbose, bool ok3);                                            \
template void mastervec<X,A>::MapAll( const String& filename );
#endif


#endif
