// Copyright (c) 2000-2003 Whitehead Institute for Biomedical Research

// #define USE_TIMERS

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include <errno.h>

#include "CoreTools.h"
#include "Feudal.h"
#include "FeudalTemplate.h"
#include "math/Functions.h"
#include "STLExtensions.h"


Bool IsGoodFeudalFile( const String& filename, bool verbose, bool ok3 )
{
  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  bool ret = ( (1 == control.nfiles() || (ok3 && 3==control.nfiles())) 
		&& control.offsets_start <= control.static_start 
		&& control.static_start <= FileSize( filename ) );
  if (!ret && verbose) {
    if (!ok3 && 1 != control.nfiles()) {
      cout<< "control.nfiles() should be 1 but is "<< control.nfiles()<< endl;
    }
    if (ok3 && 1 != control.nfiles() && 3 != control.nfiles())  {
      cout<< "control.nfiles() should be 1 or 3 but is " 
	  << control.nfiles()<< endl;
    }
    if (control.offsets_start > control.static_start) {
      PRINT2(control.offsets_start, control.static_start);
    }
    if (control.static_start > FileSize( filename ) ) {
      PRINT2(control.static_start, FileSize( filename ));
    }
  }
  return ret;
}

void MergeMastervecFiles( const String& filename )
{
  longlong main_size = FileSize(filename);
  CpAppend2( filename + "..offsets", filename );
  longlong main_size_plus = FileSize(filename);
  CpAppend2( filename + "..static", filename );
  Remove( filename + "..offsets" );
  Remove( filename + "..static" );

  int fd = Open( filename, O_RDWR );
  mv_file_control_block header;
  ReadBytes(fd, &header, sizeof(header));
  header.nfiles( 1 );
  header.offsets_start = main_size;
  header.static_start = main_size_plus;

  lseek( fd, 0, SEEK_SET );
  WriteBytes( fd, &header, sizeof(header) );
  close(fd);    
}

void DivideMastervecFile( const String& filename ) {

  int fd = Open( filename, O_RDWR );
  mv_file_control_block header, header2;
  ReadBytes(fd, &header, sizeof(header));
  header2 = header;
  header2.nfiles( 3 );
  header2.offsets_start = 0;
  header2.static_start = 0;
  lseek( fd, 0, SEEK_SET );
  WriteBytes( fd, &header2, sizeof(header2) );
  lseek(fd, header.offsets_start, SEEK_SET);

  int fdoff = OpenForWrite(filename + "..offsets");
  longlong towrite = header.static_start - header.offsets_start;
  CopyBytes(fd, fdoff, towrite);
  close (fdoff);

  int fdstatic = OpenForWrite(filename + "..static");
  towrite = FileSize(filename) - header.static_start;
  CopyBytes(fd, fdstatic, towrite);
  close(fdstatic);

  //Truncate the main file
  ftruncate(fd, header.offsets_start);
  close(fd);    
}

longlong MastervecFileRawCount( const String& filename, int dataSize )
{    
  if ( !IsRegularFile(filename) && !IsRegularFile( filename + ".gz" ) )
    FatalErr( "Neither " << filename << " nor " << filename << ".gz exists." );
  if ( IsRegularFile(filename) && IsRegularFile( filename + ".gz" ) )
    FatalErr( "Both " << filename << " and " << filename << ".gz exist."
	      << "  This confuses me." );

  if ( IsRegularFile( filename + ".gz" ) )
    System( "gzip -d " + filename + ".gz" );
  int fd = Open( filename, O_RDONLY );
  mv_file_control_block control;
  read( fd, &control, sizeof(control) );

  //get the data size from the file header, if we can.
  if ( 0 == dataSize ) dataSize = control.dataSize();
  Assert(dataSize);

  if ( control.nfiles() != 1 && control.nfiles() != 3 )
    FatalErr( "ReadAll: It would appear that " << filename << "\n"
	      << "is not in proper mastervec format.\n"
	      << "The value of control.nfiles() "
	      << "is " << control.nfiles() << ", whereas "
	      << "it should be 1 or 3,\nand for ReadAll, it should be 1.\n"
	      << "There is probably something very wrong with the file." );
  if ( control.nfiles() != 1 )
    FatalErr( "ReadAll: " << filename << " was expected to be in "
	      << "mastervec single-file format, but it has "
	      << "control.nfiles() = 3." );

  lseek( fd, control.offsets_start, SEEK_SET );
  longlong offset0, offsetn;
  read( fd, &offset0, 8 );
  lseek( fd, control.offsets_start + 8 * control.n, SEEK_SET );
  read( fd, &offsetn, 8 );
  close(fd);
  return (offsetn - offset0) / dataSize;  
}

int MastervecFileObjectCount( const String& filename )
{
  int fd = Open( filename, O_RDONLY );
  int n;
  if ( read( fd, &n, 4 ) != sizeof(n) )
    FatalErr( "read failed" );
  close(fd);
  return n;    
}

void ConcatenateMastervecData( const String& bottom, const String& top )
{    
  if ( !IsRegularFile(bottom) ) {
    Cp2( top, bottom );
    Cp2( top + "..offsets", bottom + "..offsets" );
    Cp2( top + "..static", bottom + "..static" );   
  }
  else {    
    // Copy dynamic data.
    
    int n1, n2;
    int fd1 = Open( bottom, O_RDONLY );
    read( fd1, &n1, 4 );
    close(fd1);
    
    int fd2 = Open( top, O_RDONLY );
    read( fd2, &n2, 4 );
    fd1 = Open( bottom, O_WRONLY );
    n1 += n2;
    WriteBytes( fd1, &n1, 4 );
    
    lseek( fd1, 0, SEEK_END );
    lseek( fd2, 24, SEEK_SET );
    
    int pagesize = getpagesize( );
    vec<char> buf(pagesize);
    
    while (1) { 
      int bytes_read = read( fd2, &buf[0], pagesize );
      if ( bytes_read <= 0 )
        break;
      
      WriteBytes( fd1, &buf[0], bytes_read );
      if ( bytes_read < pagesize )
        break;   
    }
    close(fd1);
    close(fd2);

    // Copy offsets.  We delete the first offset from top, and add the
    // last offset from bottom minus 24 to every other offset from top.
    
    fd1 = Open( bottom + "..offsets", O_RDONLY );
    lseek( fd1, -8, SEEK_END );
    longlong last_offset, offset;
    read( fd1, &last_offset, 8 );
    close(fd1);
    
    fd1 = Open( bottom + "..offsets", O_WRONLY );
    lseek( fd1, 0, SEEK_END );
    fd2 = Open( top + "..offsets", O_RDONLY );
    lseek( fd2, 8, SEEK_SET );
    
    while (1) {
      int bytes_read = read( fd2, &offset, 8 );
      if ( bytes_read < 8 )
        break;
      
      offset += last_offset - 24;
      WriteBytes( fd1, &offset, 8 );   
    }
    close(fd1);
    close(fd2);
    
    // Copy static data.
    
    CpAppend2( top + "..static", bottom + "..static" );    
  }
}


#ifdef __DECCXX_VER
#pragma define_template mastervec<serfvec<unsigned char>, unsigned char>
#else
INSTANTIATE_MASTERVEC( serfvec<unsigned char>, unsigned char )
#endif

#ifdef __DECCXX_VER
#pragma define_template mastervec<serfvec<unsigned short>, unsigned short>
#else
INSTANTIATE_MASTERVEC( serfvec<unsigned short>, unsigned short )
#endif

#ifdef __DECCXX_VER
#pragma define_template mastervec<serfvec<float>, float>
#pragma define_template mastervec<serfvec<int>, int>
#else
INSTANTIATE_MASTERVEC( serfvec<float>, float )
INSTANTIATE_MASTERVEC( serfvec<int>, int )
#endif

