/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// bitvector: a class which hold bits, and can be formed into a feudal vecbitvector

#ifndef BITVECTOR
#define BITVECTOR

#include "CoreTools.h"
#include "Feudal.h"

class bitvector {

     public:
  
  typedef unsigned int        value_type;
  
     unsigned int size() const { return length_; }
     int isize() const { return length_; }

     // The following methods provide unrestricted access to
     // the  data_  member of the  bitvector  class.  For the
     // sake of efficiency, no check is done on the value of
     // the argument -- hence there is the high risk of indexing
     // out-of-range in using these methods.
     // BE VERY SURE THAT YOU KNOW WHAT YOU ARE DOING BEFORE YOU
     // USE THESE METHODS.
     unsigned char DataAsBytes( int n ) const 
     {  return ((unsigned char*) data_)[n];    }
     unsigned int  DataAsInts( int n ) const 
     {  return data_[n];  }

     unsigned char Initialized( ) const { return data_ != 0; }

     unsigned int SelfOwnedBit( ) const
     {    return extra_space_and_self_owned_ & (1 << 31);    }

     Bool SelfOwned( ) const
     {    return SelfOwnedBit( ) != 0;    }

     unsigned int ExtraSpace( ) const
     {    return extra_space_and_self_owned_ & Bits31;    }

     bitvector( ) 
     {    data_ = 0;
          extra_space_and_self_owned_ = 1u << 31;    }

     void Reinitialize( )
     {    if ( SelfOwned( ) && data_ != 0 ) delete [ ] data_;
          data_ = 0;
          extra_space_and_self_owned_ = 1u << 31;    }

     void Blank( )
     {    data_ = 0;
          extra_space_and_self_owned_ = 1u << 31;    }

     ~bitvector( ) 
     {    if ( SelfOwned( ) && data_ != 0 ) delete [ ] data_;
          data_ = 0;
          extra_space_and_self_owned_ = 1u << 31;    }

     explicit bitvector( unsigned int n ) : length_(n)
     {    int words_needed = (n + 31) / 32;
          int actual_extra = words_needed * 32 - n;
          extra_space_and_self_owned_ = (1 << 31) ^ actual_extra;
          data_ = new unsigned int[words_needed];    }

     bitvector( unsigned int n, Bool value );

     void ReverseMe( );

     void Reverse( const bitvector& b );

     bitvector& operator=(const bitvector& b);

     void Swap( bitvector& b )
     {   swap( data_, b.data_ );
	 swap( length_, b.length_ );
	 swap( extra_space_and_self_owned_, b.extra_space_and_self_owned_ );    }

     void SetData( unsigned int* data )
     {    for ( unsigned int i = 0; i < (length_ + 31)/32; i++ )
               data_[i] = data[i];    }

     void Zero( )
     {    for ( unsigned int i = 0; i < (length_ + 31)/32; i++ )
               data_[i] = 0;    }

     void SetData( char* data )
     {    memcpy( data_, data, (length_ + 3)/4 );    }

     /// Setsize: resize, destroying contents

     void Setsize(unsigned int n, unsigned int extra = 0);

     /// Resize.

     void resize(unsigned int n);

     /// Resize.  If enlarged, assign bit to each new bit.

     void resize(unsigned int n, unsigned char bit);

     bitvector(const bitvector& b);

     // Return the ith element of a bitvector (for read access only).

     unsigned char operator [ ] (unsigned int i) const 
     {    AssertLt( i, length_ );
          return ( ( DataAsBytes( i>>3 ) >> (i&7)) & 1 );    }

     // Set bit i to value "bit".

     void Set(unsigned int i, unsigned char bit)
     {    AssertLt( i, length_ );
          AssertLt( bit, 2 );
          unsigned char& target = DataAsBytes( )[ i>>3 ];
          int shift = i&7;
          target = ( target & ( ~(1 << shift) ) ) | (bit << shift);    }

     /// SetToSubOf(orig_bv, start_pos, len):  Set *this to the length len
     /// sub-bitvector of  orig_bv, starting at position  start_pos.  The
     /// case where *this = orig_bv is allowed.  Not efficiently implemented.

     void SetToSubOf(const bitvector& orig_bv, const unsigned int start_pos, 
          const int len );

     // Set bits x (i <= x < j) to the same value "bit".  If reverse = True,
     // treat range as being on reverse of the bitvector.

     void Set(unsigned int i, unsigned int j, unsigned char bit, 
          Bool reverse = False);

     friend Bool operator==( const bitvector& b1, const bitvector& b2 )
     {    if ( b1.size( ) != b2.size( ) ) return False;
          for ( unsigned int i = 0; i < b1.size( ); i++ )
               if ( b1[i] != b2[i] ) return False;
          return True;    }

     const unsigned int* StartOfDynamicData( ) const
     {    return data_;    }

     unsigned int* StartOfDynamicData( )
     {    return data_;    }

     longlong SizeOfDynamicData( ) const
     {    return (length_ + 31) / 32;    }

     longlong ExtraSpaceForDynamicData( ) const
     {    return ExtraSpace( );    }

     void ShiftStartOfDynamicData( unsigned int* a1, unsigned int* a2 )
     {    data_ += a1 - a2;    }

     const char* StartOfStaticData( ) const
     {    return (char*) &length_;    }

     const int SizeOfStaticData( ) const
     {    return sizeof(length_);    }

     void SetExternal( const char* start_of_static_data, 
          unsigned int* pointer_to_dynamic_data, int size_of_dynamic_data, 
          int extra_space_for_dynamic_data )
     {    data_ = pointer_to_dynamic_data;
          length_ = *( (unsigned int*) start_of_static_data );
          extra_space_and_self_owned_ = 
               (length_ + 31)/32 * 32 - length_
               + 32 * extra_space_for_dynamic_data;    
     }

     private:

     unsigned char* DataAsBytes( ) const
     {  return (unsigned char*) data_;  }

// For full compatibility across 32-bit and 64-bit architectures, we use a
// union to pad the data_ pointer so that 8 bytes are always used.  Of course
// this wastes space on 32-bit systems.

     union {
       unsigned int* data_;///< The bits themselves, stored at 8 bits per character.
       longlong unused_;
     };

     unsigned int length_; // The number of bits.
     // Keep track of space for more bits and self-ownership:
     unsigned int extra_space_and_self_owned_;

};

typedef mastervec< bitvector, unsigned int > vecbitvector;

float Coverage( const vecbitvector& v );

/// Specialization needed because sizeof(bitvector) differs between archs.

template<> bool IsGoodFeudalFile<bitvector>( const String & filename,
 const bitvector * dummy, bool verbose, bool ok3);

void And(vecbitvector & result, const vecbitvector & first, 
	 const vecbitvector & second);
 
void Or(vecbitvector & result, const vecbitvector & first, 
	const vecbitvector & second);
 
#endif
