/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif
#define STRING_FAST_EXECUTE

#include <string.h>

#include "Basevector.h"
#include "CoreTools.h"
#include "FeudalTemplate.h"
#include "math/Functions.h"

basevector::basevector(const basevector& b)
{ 
  if ( b.data_ == 0 ) 
    { 
      data_ = 0;
      length_ = 0;
      extra_space_and_self_owned_ = 1u << 31;   
    }
  else
    {
      length_ = b.length_;
      extra_space_and_self_owned_ = (1 << 31) | b.ExtraSpace( );
      data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 15)/16];
      memcpy( data_, b.data_, sizeof(int) * (((int) length_+15)/16) );   
    }
}


// Note that operator= for class basevector will try to avoid allocating
// memory.  So long as there is enough space in *this, new is not called.

basevector&
basevector::operator = (const basevector& b)
{  
  ForceAssert( b.data_ != 0 );
  if ( data_ == 0 )
    {
      length_ = b.length_;
      extra_space_and_self_owned_ = (1 << 31) | b.ExtraSpace( );
      data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 15)/16];   
    }
  else if ( b.size( ) <= length_ + ExtraSpace( ) )
    {    
      extra_space_and_self_owned_ = 
	SelfOwnedBit( ) | ( length_ + ExtraSpace( ) - b.size( ) );
      length_ = b.size( );   
    }
  else
    {
      if ( SelfOwned( ) )
	delete [ ] data_;
      length_ = b.length_;
      extra_space_and_self_owned_ = (1 << 31) | b.ExtraSpace( );
      data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 15)/16];   
    }
  memcpy( data_, b.data_, sizeof(int) * (((int) length_+15)/16) );
  return *this;   
}


void
basevector::resize(unsigned int n)
{
  if ( data_ == 0 )
    {
      data_ = new unsigned int[(n + 10 + 15)/16];
      length_ = n;
      extra_space_and_self_owned_ = (1u << 31) | 10;   
    }
  else if ( n <= length_ + ExtraSpace( ) )
    {
      extra_space_and_self_owned_ =
	SelfOwnedBit( ) | ( ExtraSpace( ) + length_ - n );
      length_ = n;   
    }
  else
    {
      unsigned int* data_new = new unsigned int[(n + 10 + 15)/16];
      int nwords = (Min(n,length_)+15)/16;
      memcpy( data_new, data_, sizeof(int) * nwords );
      if ( SelfOwned( ) )
	delete [ ] data_;
      data_ = data_new;
      length_ = n;
      extra_space_and_self_owned_ = (1u << 31) | 10; 
    }
}


istream&
operator >> (istream& s, basevector& b)
{ 
  if (!s)
    {
      cerr << "Basevector read failed.\n";
      exit(1);
    }

  unsigned int len;
  s >> len;
  b.resize(len);

  char c;
  s.get(c);
  for ( unsigned int i = 0; i < (len+3)/4; i++ )
    s.get( ((char*) b.data_)[i] );

  return s; 
}


ostream&
operator << (ostream& s, const basevector& b)
{  
  if (!s)
    {
      cerr << "Basevector write failed.\n";
      exit(1);
    }

  unsigned int len = b.size();
  s << len << "\n";

  for ( unsigned int i = 0; i < (len+3)/4; i++ )
    s.put( ((char*) b.data_)[i] );

  return s;   
}


String
basevector::ToString( ) const
{
  String s;
  s.resize( size( ) );
  for ( unsigned int i = 0; i < size( ); i++ )
    {
      if ( (*this)[i] == 0 )
	s[i] = 'A';
      else if ( (*this)[i] == 1 )
	s[i] = 'C';
      else if ( (*this)[i] == 2 )
	s[i] = 'G';
      else if ( (*this)[i] == 3 )
	s[i] = 'T';   
    }
  return s;   
}


void
basevector::PrintBases( ostream& out,
			int start,
			int nbases,
			Bool rc ) const
{ 
  for ( int j = 0; j < nbases; j++ )
    {
      if ( j > 0 && j % 80 == 0 )
	out << "\n";
      if ( !rc )
	out << as_base( (*this)[start+j] );
      else 
	out << as_base( 3 - (*this)[ start + nbases - j - 1 ] );   
    }
  out << "\n";   
}


void
basevector::Print( ostream& out ) const
{
  PrintBases( out, 0, size(), false ); 
}
	      

void
basevector::Print( ostream& out, int id ) const
{
  out << ">sequence_" << id << "\n";
  Print( out );
}

void
basevector::Print( ostream& out, String id ) const
{
  out << ">" << id << "\n";
  Print( out );
}

unsigned int basevector::Find(const basevector & other, unsigned int start, 
		     unsigned int end) const {
  end = min(end, size());
  for (unsigned int i=start; i <= end - other.size(); ++i) {
    bool found = true;
    for (unsigned int j=0; j !=other.size(); ++j) {
      if ((*this)[i+j] != other[j]) {
	found = false;
	break;
      }
    }
    if (found) return i;
  }
  return size();
}

void
basevector::Cap( int n )
{
  static vec<unsigned char> capped_bases;
  capped_bases.reserve( size( ) );
  capped_bases.clear( );
  for ( int i = 0; i < (int) size( ); i++ )
    {
      int j;
      for ( j = i + 1; j < (int) size( ); j++ )
	if ( (*this)[j] != (*this)[i] )
	  break;
      int count = Min( j - i, n );
      for ( int k = 0; k < count; k++ )
	capped_bases.push_back( (*this)[i] );
      i = j - 1;   
    }
  Setsize( capped_bases.size( ) );
  for ( int i = 0; i < capped_bases.isize( ); i++ )
    Set( i, capped_bases[i] );   
}


basevector
Cat( const basevector& left, const basevector& right )
{
  static basevector join;
  join.Setsize( left.size( ) + right.size( ) );
  for ( unsigned int i = 0; i < left.size( ); i++ )
    join.Set( i, left[i] );
  for ( unsigned int i = 0; i < right.size( ); i++ )
    join.Set( i + left.size( ), right[i] );
  return join;   
}

basevector Cat( const basevector& b1, const basevector& b2, const basevector& b3 )
{    static basevector join;
     join.Setsize( b1.size( ) + b2.size( ) + b3.size( ) );
     for ( unsigned int i = 0; i < b1.size( ); i++ )
          join.Set( i, b1[i] );
     for ( unsigned int i = 0; i < b2.size( ); i++ )
          join.Set( i + b1.size( ), b2[i] );
     for ( unsigned int i = 0; i < b3.size( ); i++ )
          join.Set( i + b1.size( ) + b2.size( ), b3[i] );
     return join;    }

int
RoundDown( int n, int d )
{
  return n - (n % d); 
}

int
RoundUp( int n, int d )
{
  if ( n % d == 0 )
    return n;
  return n - (n % d) + d;    
}


void
CopyBases( const basevector& from,
	   int from_start,
	   basevector& to,
	   int to_start,
	   int count,
	   Bool rc_from )
{
  ForceAssert( from.Initialized( ) );
  ForceAssert( to.Initialized( ) );
  ForceAssertLe( from_start + count, (int) from.size( ) );
  ForceAssertLe( to_start + count, (int) to.size( ) );
  ForceAssertGe( from_start, 0 );
  ForceAssertGe( to_start, 0 );
  ForceAssertGe( count, 0 );
  static basevector b, from_rc;
  if (rc_from)
    {
      from_rc.Setsize( from.size( ) );
      from_rc.ReverseComplement(from);   
    }
  const basevector& source = ( rc_from ? from_rc : from );
  if ( count < 4 )
    {
      for ( int i = 0; i < count; i++ )
	to.Set( to_start + i, source[ from_start + i ] );
      return;   
    }
  int raise = RoundUp( to_start, 4 ) - to_start;
  for ( int i = 0; i < raise; i++ )
    to.Set( to_start + i, source[ from_start + i ] );
  int carry = RoundDown( count - raise, 4 );
  b.Setsize(carry);
  b.SetToSubOf( source, from_start + raise, carry );
  memcpy( to.DataAsBytesRW( ) + (to_start + raise)/4, b.DataAsBytes( ), carry/4 );
  for ( int i = 0; i < count - raise - carry; i++ )
    to.Set( to_start + raise + carry + i, 
	    source[ from_start + raise + carry + i ] );   
}


void
StringReverseComplement( const String &seq, String &rc_seq )
{
  rc_seq = seq;
    
  // reverse
  for (int ii=0; ii<(int)rc_seq.size()/2; ++ii) 
    swap( rc_seq[ii], rc_seq[ (rc_seq.size()-1)-ii ] );
    
  // complement
  for (int ii=0; ii<(int)rc_seq.size(); ++ii) 
    switch ( rc_seq[ii] ) 
      {
      case 'A': rc_seq[ii] = 'T'; break;
      case 'C': rc_seq[ii] = 'G'; break;
      case 'G': rc_seq[ii] = 'C'; break;
      case 'T': rc_seq[ii] = 'A'; break;
      case 'a': rc_seq[ii] = 't'; break;
      case 'c': rc_seq[ii] = 'g'; break;
      case 'g': rc_seq[ii] = 'c'; break;
      case 't': rc_seq[ii] = 'a'; break;
      case 'B': rc_seq[ii] = 'V'; break;
      case 'D': rc_seq[ii] = 'H'; break;
      case 'H': rc_seq[ii] = 'D'; break;
      case 'K': rc_seq[ii] = 'M'; break;
      case 'M': rc_seq[ii] = 'K'; break;
      case 'N': rc_seq[ii] = 'N'; break;
      case 'R': rc_seq[ii] = 'Y'; break;
      case 'S': rc_seq[ii] = 'S'; break;
      case 'V': rc_seq[ii] = 'B'; break;
      case 'W': rc_seq[ii] = 'W'; break;
      case 'Y': rc_seq[ii] = 'R'; break;
      default: break;
      }
}



// The lookup table used by the method
//     basevector::ReverseComplement(...)
// The value at index  b  is computed as follows:
//      unsigned char n = ~b;
//      return ( (n & 3) << 6) |
//             ( (n & (3<<2)) << 2) |
//             ( (n & (3<<4)) >> 2) |
//             ( (n & (3<<6)) >> 6);

const
unsigned char
basevector::RCtable[] =
{
  0xff, 0xbf, 0x7f, 0x3f, 0xef, 0xaf, 0x6f, 0x2f, 
  0xdf, 0x9f, 0x5f, 0x1f, 0xcf, 0x8f, 0x4f, 0x0f, 
  0xfb, 0xbb, 0x7b, 0x3b, 0xeb, 0xab, 0x6b, 0x2b, 
  0xdb, 0x9b, 0x5b, 0x1b, 0xcb, 0x8b, 0x4b, 0x0b, 
  0xf7, 0xb7, 0x77, 0x37, 0xe7, 0xa7, 0x67, 0x27, 
  0xd7, 0x97, 0x57, 0x17, 0xc7, 0x87, 0x47, 0x07, 
  0xf3, 0xb3, 0x73, 0x33, 0xe3, 0xa3, 0x63, 0x23, 
  0xd3, 0x93, 0x53, 0x13, 0xc3, 0x83, 0x43, 0x03, 
  0xfe, 0xbe, 0x7e, 0x3e, 0xee, 0xae, 0x6e, 0x2e, 
  0xde, 0x9e, 0x5e, 0x1e, 0xce, 0x8e, 0x4e, 0x0e, 
  0xfa, 0xba, 0x7a, 0x3a, 0xea, 0xaa, 0x6a, 0x2a, 
  0xda, 0x9a, 0x5a, 0x1a, 0xca, 0x8a, 0x4a, 0x0a, 
  0xf6, 0xb6, 0x76, 0x36, 0xe6, 0xa6, 0x66, 0x26, 
  0xd6, 0x96, 0x56, 0x16, 0xc6, 0x86, 0x46, 0x06, 
  0xf2, 0xb2, 0x72, 0x32, 0xe2, 0xa2, 0x62, 0x22, 
  0xd2, 0x92, 0x52, 0x12, 0xc2, 0x82, 0x42, 0x02, 
  0xfd, 0xbd, 0x7d, 0x3d, 0xed, 0xad, 0x6d, 0x2d, 
  0xdd, 0x9d, 0x5d, 0x1d, 0xcd, 0x8d, 0x4d, 0x0d, 
  0xf9, 0xb9, 0x79, 0x39, 0xe9, 0xa9, 0x69, 0x29, 
  0xd9, 0x99, 0x59, 0x19, 0xc9, 0x89, 0x49, 0x09, 
  0xf5, 0xb5, 0x75, 0x35, 0xe5, 0xa5, 0x65, 0x25, 
  0xd5, 0x95, 0x55, 0x15, 0xc5, 0x85, 0x45, 0x05, 
  0xf1, 0xb1, 0x71, 0x31, 0xe1, 0xa1, 0x61, 0x21, 
  0xd1, 0x91, 0x51, 0x11, 0xc1, 0x81, 0x41, 0x01, 
  0xfc, 0xbc, 0x7c, 0x3c, 0xec, 0xac, 0x6c, 0x2c, 
  0xdc, 0x9c, 0x5c, 0x1c, 0xcc, 0x8c, 0x4c, 0x0c, 
  0xf8, 0xb8, 0x78, 0x38, 0xe8, 0xa8, 0x68, 0x28, 
  0xd8, 0x98, 0x58, 0x18, 0xc8, 0x88, 0x48, 0x08, 
  0xf4, 0xb4, 0x74, 0x34, 0xe4, 0xa4, 0x64, 0x24, 
  0xd4, 0x94, 0x54, 0x14, 0xc4, 0x84, 0x44, 0x04, 
  0xf0, 0xb0, 0x70, 0x30, 0xe0, 0xa0, 0x60, 0x20, 
  0xd0, 0x90, 0x50, 0x10, 0xc0, 0x80, 0x40, 0000
};

void
basevector::ReverseComplement()
{
  ForceAssert( Initialized() );

  const int num_bytes = (size() + 3) / 4;

  // reverse complement the whole  data_  array
  // using the table  RCtable  for lookup

  for (int i = 0; i < num_bytes/2; i++)
    {
      register unsigned char RCbyte = RCtable[ DataAsBytes(i) ];
      DataAsBytes()[i] = RCtable[ DataAsBytes(num_bytes-1-i) ];
      DataAsBytes()[num_bytes-1-i] = RCbyte;   
    }
  if (num_bytes % 2 == 1)
    {
      DataAsBytes()[num_bytes/2] = RCtable[ DataAsBytes(num_bytes/2) ];
    }

  // When the length of the basevector is not divisible by 4,
  // we have to shift the bases down accordingly.
  // The picture is as follow:
  //
  //      uintL
  //      |
  //      |   uintR
  //      |   |
  // ---|---|---|--------------------------------------------
  //      ^
  //      |
  //      i
  //
  // The two adjacent unsigned ints  uintL  and  uintR  are used together
  // to compute the shifted-down value to be put at position  i  of  data_. 

  const int num_ints = (size() + 15) / 16;

  register unsigned int uintL = data_[0];
  register unsigned int uintR;

    // macro for shifting bases in the data_ array
#define RC_SHIFT_LOOP_CASE(_case_)                        \
    case _case_:                                          \
      {                                                   \
	for (int i = 0; i < num_ints-1; i++)              \
	  {                                               \
	    uintR = data_[i+1];                           \
	    data_[i] = (uintL >> 8 - 2*_case_) |          \
                       (uintR << 32 - (8 - 2*_case_));    \
	    uintL = uintR;                                \
	  }                                               \
	data_[num_ints-1] = (uintL >> 8 - 2*_case_);      \
        break;                                            \
      }                                                   \
						       
						       
  // loop through the  data_  array and shift the bases down
  // according to the remainder of  size()  modulo 4.
  switch (size() % 4)
    {
    case  0:
      {
	break; 	// Do nothing.
      }
    RC_SHIFT_LOOP_CASE( 1);
    RC_SHIFT_LOOP_CASE( 2);
    RC_SHIFT_LOOP_CASE( 3);
    }
}





// The lookup table used by the method
//     basevector::GcBases(...)
const
unsigned char
basevector::GCtable[] =
{
  0, 1, 1, 0, 1, 2, 2, 1, 
  1, 2, 2, 1, 0, 1, 1, 0, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  0, 1, 1, 0, 1, 2, 2, 1, 
  1, 2, 2, 1, 0, 1, 1, 0, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  2, 3, 3, 2, 3, 4, 4, 3, 
  3, 4, 4, 3, 2, 3, 3, 2, 
  2, 3, 3, 2, 3, 4, 4, 3, 
  3, 4, 4, 3, 2, 3, 3, 2, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  2, 3, 3, 2, 3, 4, 4, 3, 
  3, 4, 4, 3, 2, 3, 3, 2, 
  2, 3, 3, 2, 3, 4, 4, 3, 
  3, 4, 4, 3, 2, 3, 3, 2, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  0, 1, 1, 0, 1, 2, 2, 1, 
  1, 2, 2, 1, 0, 1, 1, 0, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  1, 2, 2, 1, 2, 3, 3, 2, 
  2, 3, 3, 2, 1, 2, 2, 1, 
  0, 1, 1, 0, 1, 2, 2, 1, 
  1, 2, 2, 1, 0, 1, 1, 0
};

unsigned int
basevector::GcBases( int start, int end ) const {
  if (start < 0 )
    start = 0;
  if (end < 0 || end > (int) this->size())
    end = this->size();
  if ( end<=start ) {
    return 0;
  }

  unsigned int gc = 0;
  // If start is not on a byte-border...
  if ( start % 4 != 0 ) {
    // Find the next byte-border...
    int byteborder = start + 4 - start%4;
    // If that is past the end...
    if ( byteborder >= end ) {
      // Just calculate the GC the obvious way.
      for ( int j = start; j < end; ++j )
        if ( (*this)[j] == 1 || (*this)[j] == 2 )
          ++gc;
      return gc;
    }
    else {
      // Otherwise, tally the GCs between start and the byte-border.
      for ( int j = start; j < byteborder; ++j )
        if ( (*this)[j] == 1 || (*this)[j] == 2 )
          ++gc;
      start = byteborder;
    }
  }

  // If end is not on a byte-border...
  if ( end % 4 != 0 )
    // Tally the GCs between the byte-border and end.
    for ( int j = end - end%4; j < end; ++j )
      if ( (*this)[j] == 1 || (*this)[j] == 2 )
        ++gc;

  // Tally the GCs by byte.
  for ( int j = start/4; j < end/4; j++ )
    gc += GCtable[DataAsBytes(j)];

  return gc;
}


void
basevector::SetToSubOf(const basevector& orig_bv,
		       unsigned int start_pos,
		       int len,
		       int extra)
{
  if ( this == &orig_bv )
  {    basevector x;
       x.SetToSubOf( orig_bv, start_pos, len, extra );
       *this = x;
       return;    }

  ForceAssertGe( len, 0 );
  ForceAssertLe( start_pos + len, orig_bv.size() );
  ForceAssertLe( start_pos, orig_bv.size() ); 
  // That last one catches a passed start_pos<0 converted to unsigned

  // make sure there is enough extra space
  // to facilitate the memcpy later
  const int num_ints = ((start_pos%16)+len+15)/16;
  extra += num_ints * 16 - len;
  Setsize(len, extra);
  const int num_bytes = 4 * num_ints;
  // data_ is now an array of at least num_bytes many bytes

  unsigned char* src = orig_bv.DataAsBytes( );
  unsigned char* tgt = DataAsBytes( );

  // SetToSubOf should not be called with  orig_bv == *this
  Assert( src != tgt );

  // copy the right number of bytes from src to tgt
  memcpy( tgt, src + (start_pos/4), num_bytes );


  // When the start position of the sub-basevector is not
  // divisible by 4, we have to shift the bases down accordingly.
  // The picture is as follow:
  //
  //      unitL
  //      |
  //      |   uintR
  //      |   |
  // ---|---|---|--------------------------------------------
  //      ^
  //      |
  //      i
  //
  // The two adjacent unsigned ints  uintL  and  uintR  are used together
  // to compute the shifted-down value to be put at position  i  of  data_. 

  register unsigned int uintL = data_[0];
  register unsigned int uintR;

    // macro for shifting bases in the data_ array
#define SUB_SHIFT_LOOP_CASE(_case_)                 \
    case _case_:                                    \
      {                                             \
	for (int i = 0; i < num_ints-1; i++)        \
	  {                                         \
	    uintR = data_[i+1];                     \
	    data_[i] = (uintL >> 2*_case_) |        \
                       (uintR << 32 - 2*_case_);    \
	    uintL = uintR;                          \
	  }                                         \
	data_[num_ints-1] = (uintL >> 2*_case_);    \
        break;                                      \
      }                                             \


  // loop through the data_ array and shift the bases down
  // according to the remainder of  start_pos  modulo 4.
  switch (start_pos % 4)
    {
    case  0:
      {
	break; 	// Do nothing.
      }
    SUB_SHIFT_LOOP_CASE( 1);
    SUB_SHIFT_LOOP_CASE( 2);
    SUB_SHIFT_LOOP_CASE( 3);
    }
}

void BinaryWrite( int fd, const basevector& b )
{    WriteBytes( fd, &b.length_, sizeof(b.length_) );
     WriteBytes( fd, b.data_, sizeof(unsigned int) * ( (b.length_ + 15)/16 ) );    }

void BinaryRead( int fd, basevector& b )
{    ReadBytes( fd, &b.length_, sizeof(b.length_) );
     b.Setsize(b.length_);
     ReadBytes( fd, b.data_, sizeof(unsigned int) * ( (b.length_ + 15)/16 ) );    }
  
unsigned int GcBases( const basevector& b, int start, int end ) {    
  return b.GcBases(start, end);
}

float GcPercent( const basevector& b, int start, int end ) {    
  if ( start < 0 ) start = 0;
  if ( end < 0 || end > (int)b.size() ) end = b.size();
  unsigned int gc = b.GcBases(start, end);
  return 100.0f * gc / (end-start);
}

float GcPercent( const String& b ) {    
  float gc = 0;
  for ( int j = 0; j < (int) b.size( ); j++ )
    if ( b[j] == 'G' || b[j] == 'C' )
      ++gc;
  return 100.0 * gc / b.size( );
}

Bool Overlap( const basevector& s, const basevector& t, int r )
{    if ( s.isize( ) < r || t.isize( ) < r ) return False;
     for ( int j = 0; j < r; j++ )
          if ( s[ s.isize( ) - r + j ] != t[j] ) return False;
     return True;    }

template void mastervec<basevector, unsigned int>::QuickSort(const int, const int);

void ReverseComplement( vecbasevector& s )
{    for ( int i = 0; i < s.size( ); i++ )
          s[i].ReverseComplement( );    }

longlong SizeSum( const vecbasevector& s )
{    longlong sum = 0;
     for ( int i = 0; i < s.size( ); i++ )
          sum += s[i].size( );
     return sum;    }


template<> bool IsGoodFeudalFile<basevector>( const String & filename, 
 const basevector * dummy, bool verbose, bool ok3) {
  if (!IsGoodFeudalFile(filename, verbose, ok3)) return false;

  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  if (control.IsOldFormat()) return true;

  if ( control.vecSize() != 12 && control.vecSize() != sizeof(basevector)) {
    if (verbose) {
      cerr << "IsGoodFeudalFile: Failed vecSize test" << endl;
      cerr << "Should be 16 (12 for old files), but is " 
      << int(control.vecSize()) << endl;
    }
    return false;
  }
  return true;
}


#ifdef __DECCXX_VER
#pragma define_template mastervec<basevector, unsigned int>
#else
INSTANTIATE_MASTERVEC( basevector, unsigned int )
#endif
