/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/** Implementations for classes for reading SFF files, in namespace sff.

\file SFFTypes.cc

*/


#include "454/sff/SffTypes.h"

namespace sff {

  istream & CommonHeader::Read(istream & is) {
    is.read((char *)(&magic_number),sizeof(magic_number));
    magic_number = FromBigEndian(magic_number);
    AssertEq(magic_number, MAGIC_NUMBER);

    is.read((char *)(version),sizeof(version));
    AssertEq(0, strncmp(version,"\0\0\0\1",4));//only version 1 is valid.

    is.read((char *)(&index_offset),sizeof(index_offset));
    index_offset = FromBigEndian(index_offset);
    is.read((char *)(&index_length),sizeof(index_length));
    index_length = FromBigEndian(index_length);
    is.read((char *)(&number_of_reads),sizeof(number_of_reads));
    number_of_reads = FromBigEndian(number_of_reads);
    is.read((char *)(&header_length),sizeof(header_length));
    header_length = FromBigEndian(header_length);
    is.read((char *)(&key_length),sizeof(key_length));
    key_length = FromBigEndian(key_length);
    is.read((char *)(&number_of_flows_per_read),sizeof(number_of_flows_per_read));
    number_of_flows_per_read = FromBigEndian(number_of_flows_per_read);
    is.read((char *)(&flowgram_format_code),sizeof(flowgram_format_code));

    flow_chars.resize(number_of_flows_per_read);
    is.read((char *)(&flow_chars[0]),number_of_flows_per_read);
    key_sequence.resize(key_length);
    is.read((char *)(&key_sequence[0]),key_length);

    int padding = header_length-(31 + number_of_flows_per_read + key_length);
    if (padding) 
      is.ignore(padding);
    Assert(is);
    return is;
  }
      
  istream & ReadHeader::Read(istream & is) {
    is.read((char *)(&read_header_length),sizeof(read_header_length));
    read_header_length = FromBigEndian(read_header_length);
    is.read((char *)(&name_length),sizeof(name_length));
    name_length = FromBigEndian(name_length);
    is.read((char *)(&number_of_bases),sizeof(number_of_bases));
    number_of_bases = FromBigEndian(number_of_bases);
    is.read((char *)(&clip_qual_left),sizeof(clip_qual_left));
    clip_qual_left = FromBigEndian(clip_qual_left);
    is.read((char *)(&clip_qual_right),sizeof(clip_qual_right));
    clip_qual_right = FromBigEndian(clip_qual_right);
    is.read((char *)(&clip_adapter_left),sizeof(clip_adapter_left));
    clip_adapter_left = FromBigEndian(clip_adapter_left);
    is.read((char *)(&clip_adapter_right),sizeof(clip_adapter_right));
    clip_adapter_right = FromBigEndian(clip_adapter_right);

    name.resize(name_length);
    is.read((char *)(&name[0]),name_length);

    int padding = read_header_length-(16 + name_length);
    if (padding)
      is.ignore(padding);
    Assert(is);
    return is;
  }


  istream & ReadData::Read(istream & is, int nflows, int nbases) {
    flows.resize(nflows);
    is.read((char *)(&flows[0]),nflows * sizeof(flows[0]));
    for (int i=0; i != flows.size(); ++i)
      flows[i]=FromBigEndian(flows[i]);

    correspondence.resize(nbases);
    is.read((char *)(&correspondence[0]),nbases);
    bases.resize(nbases);
    is.read(bases.c_str_mutable(),nbases);
    quals.resize(nbases);
    is.read((char *)(&quals[0]),nbases);

    int padding =  8 - (nflows * sizeof(flows[0]) + 3 * nbases) % 8;
    if (8 != padding) 
      is.ignore(padding);
    Assert(is);
    return is;
  }

  int DataLength(int nbases, const CommonHeader & chead) {
    int len = nbases * 3 + chead.number_of_flows_per_read * 2;
    len += (0 == len % 8) ? 0 : (8 - len % 8);
    return len;
  }

  void SkipIndex(istream & is, const CommonHeader & chead) {
    if (uint64_t(is.tellg()) == chead.index_offset) {
      is.seekg(chead.index_length, ios::cur);
    }
  }
    
  pair<longlong, longlong> CountBasesAndNames(istream & is) {
    CommonHeader chead;
    chead.Read(is);
    uint16_t read_header_length;
    uint16_t name_length;
    uint32_t number_of_bases;
    pair<longlong, longlong> ret(0,0);

    for (unsigned int i=0; i != chead.number_of_reads; ++i) {
      SkipIndex(is, chead);
      is.read((char *)(&read_header_length),sizeof(read_header_length));
      read_header_length = FromBigEndian(read_header_length);
      is.read((char *)(&name_length),sizeof(name_length));
      name_length = FromBigEndian(name_length);
      is.read((char *)(&number_of_bases),sizeof(number_of_bases));
      number_of_bases = FromBigEndian(number_of_bases);
      int advance = DataLength(number_of_bases, chead) + read_header_length
	-sizeof(read_header_length) - sizeof(name_length)
	-sizeof(number_of_bases);
      is.ignore(advance);
      ret.first += number_of_bases;
      ret.second += name_length;
    }
    return ret;
  }


} //end of namespace sff



ostream & operator<<(ostream & os, const sff::CommonHeader & h) {
  //ugly hack: redirect cout so we can use PRINT4.
  streambuf *savebuf = cout.rdbuf();
  streambuf *buf1 = os.rdbuf();
  cout.rdbuf(buf1);
  PRINT4(h.magic_number, int(h.version[3]), h.index_offset, h.index_length);
  PRINT5(h.number_of_reads, h.header_length, h.key_length, h.number_of_flows_per_read, int(h.flowgram_format_code));
  os << "floworder: " ; 
  copy(h.flow_chars.begin(), h.flow_chars.end(),ostream_iterator<char>(os)); 
  os << "\nkey: ";
  copy(h.key_sequence.begin(),h.key_sequence.end(),ostream_iterator<char>(os));
  os << endl;
  cout.rdbuf(savebuf);
  return os;
}



ostream & operator<<(ostream & os, const sff::ReadHeader & h){
  //ugly hack: redirect cout so we can use PRINT4.
  streambuf *savebuf = cout.rdbuf();
  streambuf *buf1 = os.rdbuf();
  cout.rdbuf(buf1);
  PRINT3(h.read_header_length, h.name_length, h.number_of_bases);
  PRINT4(h.clip_qual_left, h.clip_qual_right, h.clip_adapter_left, h.clip_adapter_right);
  os << h.name << endl;
  cout.rdbuf(savebuf);
  return os;
}

ostream & operator<<(ostream & os, const sff::ReadData & d) {
  copy(d.flows.begin(), d.flows.end(), 
      ostream_iterator<uint16_t>(os," ")); os << endl;
  copy(d.correspondence.begin(), d.correspondence.end(), 
      ostream_iterator<int>(os," ")); os << endl;
  os << d.bases << endl; 
  copy(d.quals.begin(), d.quals.end(), 
      ostream_iterator<int>(os," ")); os << endl;
  return os;
}



