#ifndef SFF_READ_H
#define SFF_READ_H
/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "454/sff/PropertyVector.h"
#include "454/sff/SffTypes.h"
#include "454/qual/PredictorParameterHandler.h"
#include "Basevector.h"
#include "Qualvector.h"


class SffInfo;

/** Parameters needed for producing error predictors.
\class SffPredictorParameters
*/
struct SffPredictorParameters: public PredictorParameterHandler {
  //parameters used in SffRead::GetPredictors:
  static int cafieValue_;/// See PushCafie();
  static int otherWidth_;/// See Other();
  static int otherType_;/// See Other();
  static int flowalign_;/// See CacheFlowAlign();
  static int flowalignRadius_;/// See CacheFlowAlign();
  static int flowalignMaxRadius_;/// See Max3FlowAlign();
  static int posType_;///< See PosForQuality();
  static int posStart_;///< See PosForQuality();

  virtual ~SffPredictorParameters() {}
  virtual String GetPredictorParameters() const;
  virtual void SetPredictorParameters(const String & params);

  void SetFromFile(const String & fname);
};


/** Give quick access to privileged information in an SffInfo.

\class SffRead

This class gives direct access to bases, quals, flows, etc...
for an SffInfo. It represents one row of the SffInfo, as pointed
to by Index(). At construction, it loads pointers to 
the appropriate vectors from the SffInfo.

It can function like an iterator to run through a whole SffInfo,
and has prefix ++ and -- operators. Note that there are some 
cached values, and that ClearCache() should be called any time
the index_ is changed.

Since the vector pointers are not owned, it is acceptable to 
copy and assign SffRead objects with the default
copy constructor and assignment operator.
*/
class SffRead {
private:
  int index_; ///<which row of the SffInfo we represent.
  SffInfo & info_;

  //all vectors are borrowed from the SffInfo
  vecbasevector * vbases;
  vecqualvector * vquals;
  vecString * vname;
  vecvecushort * vflows;
  vecvecushort * vcorrespondence;
  vec< short> * vclipQualLeft;
  vec< short> * vclipQualRight;
  vec< short> * vclipAdapLeft;
  vec< short> * vclipAdapRight;

public:  
  ///Create a read from an SffInfo.
  SffRead(SffInfo & info, int index = 0);

  unsigned int size() const { return info_.size(); }

  //direct access to information at a particular read index.
  basevector & Bases() { return (*vbases)[index_]; }
  qualvector & Quals() { return (*vquals)[index_]; }
  vecushort & Flows() { return (*vflows)[index_]; }
  vecushort & Corr() { return (*vcorrespondence)[index_]; }
  String & Name() { return (*vname)[index_]; }
  short & ClipQualLeft() { return (*vclipQualLeft)[index_]; } ;
  short & ClipQualRight() { return (*vclipQualRight)[index_]; };
  short & ClipAdapLeft() { return (*vclipAdapLeft)[index_]; };
  short & ClipAdapRight() { return (*vclipAdapRight)[index_]; };

  unsigned short FlowAtBase(int i) {
    return Flows()[Corr()[i]];
  }

  void SetIndex(unsigned int i) {
    ForceAssertLt(longlong(i), longlong(size()));
    index_ = i;
    ClearCache();
  }

  void operator++() { 
    AssertLt (index_,longlong(size())); ClearCache(); ++index_; 
  }
  void operator--() { AssertGt (index_,0); ClearCache(); --index_; }

  int Index() const { return index_; }
  friend inline bool operator==(const SffRead & l, const SffRead & r) {
    return r.vbases == l.vbases && r.Index() == l.Index();
  }

  /// Return the clipping positions in base space.
  /// This is a half-open, 0-based interval.
  pair< short,  short> Clips() {
     short left = max(short(0), max(ClipQualLeft(), ClipAdapLeft()));
     short right = 
      min(( NO_CLIP == ClipAdapRight() ? Bases().size() : ClipAdapRight() ),
	  ( NO_CLIP == ClipQualRight() ? Bases().size() : ClipQualRight() )); 
    if (left > right) {
      PRINT3(Index(), left, right);
      PRINT4(ClipQualLeft(), ClipAdapLeft(),ClipQualRight(), ClipAdapRight());
      ForceAssert(0);
    }
    return make_pair(left, right);
  }

  /// Return the clipping positions in flow space.
  /// This is a half-open, 0-based interval.
  pair<short, short> FlowClips() {
   pair<short,short> clips = Clips();
   clips.first = Corr()[clips.first];
   clips.second = ( Corr().size() == clips.second ) 
     ? Flows().size()
     : Corr()[clips.second];
   return clips;
  }

  static bool HasOther() { return SffPredictorParameters::otherType_ != 0; }

  static int CafieNumber() { 
    return 4 == SffPredictorParameters::cafieValue_ ? 2 : 1; 
  }

  ///Calculate the separation between 0 and 1 signals. 
  ///This is what we used to call the overlap score.
  ///start and end are measured in flows.
  float SeparationScore(unsigned int start = 0, unsigned int end = 0, 
                        unsigned int maxPasses = 2);

  ///Length of current homopolymer
  float Homopol(int base) {
    int ret = sff::Call(FlowAtBase(base));
    if (0 == ret) PRINT4(base, FlowAtBase(base), Corr()[base], *this);
    return sff::Call(FlowAtBase(base)); }

  ///Incomplete extension: length of homopolymer in previous flow cycle.
  float Ie(int base) { 
    return Corr()[base] < 4 ? 0 : sff::Call(Flows()[Corr()[base]-4]);
  }

  ///Carryforward: length of homopolymer in next flow cycle.
  float CarryForward(int base) {
    return Corr()[base] + 4 >= Flows().size() ? 0 : 
      sff::Call(Flows()[Corr()[base]+4]);
  }

  ///Adjust position so it increases with decreasing quality.
  float PosForQuality(int base) { 
    switch (SffPredictorParameters::posType_) {
      case 0: 
	return abs(SffPredictorParameters::posStart_ - base); 
      case 1: 
	return abs(SffPredictorParameters::posStart_ - Corr()[base]);
      default:
	FatalErr("Unknown posType_ " << SffPredictorParameters::posType_);
    }
  }

  ///Calculate flow alignment score: mean of all flow errors (differences
  /// between actual flow measured and closest integer) from the previous
  /// base to the next base inclusive.
  float FlowAlign(int base) {
    if (flowAlignCache_.empty()) CacheFlowAlign();
    return flowAlignCache_[base];
  }

  /// Calculate worst-of-3 flow alignment scores centered at base.
  /// The radius is actually SffPredictorParameters::flowalignMaxRadius_,
  /// not necessarily 1.
  float Max3FlowAlign(int base)
  { return MaxFlowAlign(base, SffPredictorParameters::flowalignMaxRadius_); }

  /// Return the maximum flow align score in a window of bases.
  /// Window is centered on base and has the given width.
  float MaxFlowAlign(int base, int width=1);

  /// Return the maximum flow align score in a window of bases.
  /// Window is centered on base and has the given width.
  float MeanFlowAlign(int base, int width=1);

  /// Get the standard quality predictors for this base.
  /// This code is meant to centralize picking predictors, although
  /// synchronization with ConstructPhredTableFromSff is still needed.  
  void GetPredictors(vec<float> & values, int base);

  ///Set quals for this read from a PhredTable object.
  ///Assumes the columns in the table are PosForQuality, Max3FlowAlign,
  /// -SeparationScore, Homopol and Ie.
  /// Sets quals for every base, but uses the Clips() to determine what
  /// flows to look at to calculate the SeparationScore().
  void SetQualsFrom(const PhredTableReader & p, unsigned char MIN, 
      unsigned char MAX, int SUBTRACT);

  ///Fix all clips to go along with a new correspondence vector.
  /// This is only used in AllBasesFromFlows;
  void FixClips(const serfvec<unsigned short> & newCorr);

  void ClippedBases(basevector & b) {
    pair<short,short> clips = Clips();
    b.SetToSubOf(Bases(), clips.first, clips.second - clips.first);
  }

  SffInfo & Info() { return info_; };

  ///Print out name, bases, and flows.
  ///Non-const because the methods are non-const.
  friend ostream & operator<<(ostream & os, SffRead & r);

	private:
  vec<float> flowErrors_; ///difference between float flow value and integer
  vec<float> flowAlignCache_; ///flowAligns for individual bases.
  vec<float> sepCache_; ///Separation scores for individual bases.

  ///Clear any cached information that depends on index_.
  void ClearCache() { flowErrors_.clear(); flowAlignCache_.clear(); sepCache_.clear();}
  ///Write into flowErrors_ and flowAlignCache_;
  void CacheFlowAlign();

  /// Fix one clip to go along with a new correspondence vector.
  /// We first figure out which flow the base used to correspond to, and 
  /// work back from there to figure out which new base corresponds to that
  /// flow.
  void FixClip( short & clip, 
      const serfvec<unsigned short> & newCorr);

  /// Add appropriate Cafie predictors depending on cafieValue_.
  void PushCafie(vec<float> & values, int base);

  /// Add appropriate other predictors depending on otherType_, otherWidth_.
  void PushOther(vec<float> & values, int base);

};

#endif //SFF_READ_H
