#ifndef SFF_INFO_H
#define SFF_INFO_H
/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////


/// \file SffInfo.h
///
/// \class SffInfo
/// Class that contains and gives access to a table of information
/// about flowreads, including but not limited to the flows, bases,
/// crossreference, and quality score information for each read.
///
/// It is implemented as a set of maps from Strings to vectors of
/// various types.  All the vectors stored by a particular instance
/// are of the same length: one entry per read.  The supported types
/// are listed below.
///
/// It also includes a String-to-String map for storing information
/// about the entire data set, including flow order and key sequence.

#include "454/sff/PropertyVector.h"
#include "454/sff/SffTypes.h"
#include "Basevector.h"
#include "Qualvector.h"

class PhredTableReader;

namespace sff {
  // A typedef needed by SffInfoTypeTable so that it doesn't have to
  // use commas in naming types
  typedef mastervec<serfvec<unsigned short>, unsigned short> vecvecushort;
  // Corresponding contained vector.
  typedef serfvec<unsigned short> vecushort;
}

using sff::vecvecushort;
using sff::vecushort;
using sff::NO_CLIP;

class SffInfo {
  friend class SffInfoTester;
  friend class SffRead;

  typedef map< String, String > Smap;
  Smap strings_;
  unsigned int size_;
 
public:

  //Privileged names for columns read from the 454 SFF format, defined
  //here to avoid confusion.
  static const String NAME;
  static const String BASES;
  static const String QUALS;
  static const String FLOWS;
  static const String CORR; ///<correspondence between bases and flows.
  static const String CLIP_QUAL_LEFT; ///< 0-based (changed from 454)
  static const String CLIP_QUAL_RIGHT; ///< 0-based (changed from 454)
  static const String CLIP_ADAP_LEFT; ///< 0-based (changed from 454)
  static const String CLIP_ADAP_RIGHT; ///< 0-based (changed from 454)
  static const String FLOW_ORDER; ///< floworder stored as string property

  static const String SIZE; ///! String property used to avoid reading
			    ///data to find out the size
  /// We use a macro to define a PropertyVector of each type specified
  /// in SffInfoTypeTable.h, along with appropriately-named accessors
  /// that point to the appropriate PropertyVector and a convenience
  /// typedef NameVector.
#define SFF_IMPL(Name, MemberType, VecType)				    \
 private:								    \
  PropertyVector<MemberType, VecType> Name ## map_;			    \
									    \
 public:								    \
  typedef VecType Name ## Vector;					    \
  const VecType & Get ## Name ## Vector(const String &property) const	    \
  { return Name ## map_.GetVector(property); }				    \
  VecType & GetMutable ## Name ## Vector(const String &property)	    \
  { return Name ## map_.GetMutableVector(property); }			    \
  VecType & GetNew ## Name ## Vector(const String &property, MemberType deflt)\
  { return Name ## map_.GetNewVector(property, deflt); }		    \
  VecType & Get ## Name ## VectorForUpdate(const String &property,	    \
					MemberType deflt)		    \
  { return Name ## map_.GetVectorForUpdate(property, deflt); }		    \
  void Set ## Name ## Vector(const String &property, const VecType &data) { \
    Name ## map_.SetVector(property, data);                                 \
    if (empty()) setSize(Name ## map_.size());                               \
    else ForceAssertEq(longlong(Name ## map_.size()), longlong(size())); }  \

#include "454/sff/SffInfoTypeTable.h"
#undef SFF_IMPL

public:
  ///Create an empty SffInfo class.
  SffInfo() : size_(0) { }

  ///Load a SffInfo class from disk.
  explicit SffInfo( const String &filename): size_(0) {
    Read(filename);
  }

  ///Save to disk.
  void Write(const String &filename) const;

  ///Read from disk.  Implementation also depends on SffInfoTypeTable.h
  void Read(const String & filename);

  /// Read from a 454-generated SFF file. 
  /// The correspondence are clips are originally 1-based.
  /// Both become 0-based when we read them in, because all our
  /// code counts from 0. The clips are set to sff::NO_CLIP if they 
  /// were 0. Also, the clip intervals become half-open rather than
  /// closed.
  /// if allBasesFromFlows = true, call AllBasesFromFlows to remove Ns,
  /// adjust the correspondence, and set all quals to 0.
  sff::CommonHeader ReadFromSff(const String & filename,
				bool allBasesFromFlows = true);
  
  ///The length of each vector of information.
  unsigned int size() const { return size_; }

  ///The length of each vector of information.
  int isize() const { return size_; }

   void setSize(unsigned int size) {
    if (size_==0) {
      size_ = size;
      strings_[SIZE] = ToString(size_);
      // Send each PropertyVector a resize
#define SFF_IMPL(Name, MemberType, VecType)				    \
    Name ## map_.setSize(size)
#include "454/sff/SffInfoTypeTable.h"
#undef SFF_IMPL
    } else {
      ForceAssertEq(0,size_);
    }
  }

  bool empty() const { return 0 == size_; }


  //------- Methods that use SffRead and expect bases, flows, etc...----//

  ///Write out all basevectors to a fasta file.
  void WriteFasta(const String & filename, bool useClip = true);

  ///Write out all basevectors to a fasta file.
  void WriteFasta(ostream & os, bool useClip = true);

  ///Write out all quals to a fasta file.
  void WriteQuals(const String & filename, bool useClip = true);

  ///Write out all quals to a fasta file.
  void WriteQuals(ostream & os, bool useClip = true);

  /// Recalculate all bases and correspondences from the flows.
  /// Assumes a score of x50 or below means x bases.
  /// Sets all quals to 0.
  /// This function exists because 454 does a couple of funky things
  /// in basecalling: they call dots as Ns (not always a good idea)
  /// and they also call before rounding, so that a flow score of
  /// 150 sometimes means 1 base and sometimes means 2 bases.
  void AllBasesFromFlows( const int MAX_HEIGHT=50, const int MAX_BASES=600);

  ///Set quals for all data from a PhredTable object.
  ///Assumes the columns in the table are PosForQuality, Max3FlowAlign,
  /// -SeparationScore, Homopol and Ie.
  /// Sets quals for every base, but uses the Clips() to determine what
  /// flows to look at to calculate the SeparationScore().
  void SetQualsFrom(const PhredTableReader & p, unsigned char MIN = 5, 
			   unsigned char MAX = 35, int SUBTRACT = 1);

  



  ////---------- String-to-String mapping methods. -----------------////

  ///Getters for properties.
  bool HasStringProperty(const String &property) const { 
    return strings_.find(property)!= strings_.end(); 
  }
  set<String> StringProperties() const;

  ///Get const reference to existing string property.
  const String & GetString(const String & property) const {
    Smap::const_iterator which = strings_.find(property);
    if (which == strings_.end()) // force diagnostic error message
      ForceAssertEq("SffInfo: did not find string property ", property);
    return which->second;
  }
    
  void SetString(const String & property, const String & value="") {
    strings_[property] = value;
  }

  ///Get mutable reference to existing string property.
  String & GetMutableString(const String & property) {
    ForceAssert(HasStringProperty(property));
    return GetNewString(property);
  }

  ///Parallel to GetVectorForUpdate, actually == GetNewString.
  String & GetStringForUpdate(const String & property) {
    return GetNewString(property);
  }

  ///Get mutable reference to new or existing string property.
  String & GetNewString(const String & property) {
    return strings_[property];
  }
};



///Return the sizes of the contigs in the reference for this key.
///Prerequisites: the analinfo file has been loaded into the
/// "analinfo" String property.
vec<int> GetReferenceSizes(const SffInfo & info, const String & key="TCAG");



#endif //SFF_INFO_H
