#ifndef PROPERTY_VECTOR_H
#define PROPERTY_VECTOR_H
/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/// \file PropertyVector.h
///
/// \class PropertyVector
///
/// Generic class that maintains a map from Strings ("properties") to
/// vectors of some type.  A generalization and simplification of the
/// FlowInfo class--now works with Mastervecs like vecString and
/// vecvec<T>.
///

#include "CoreTools.h"
#include "Feudal.h"
#include "Basevector.h"
#include <iosfwd>
#include <set>
#include <map>
#include "system/System.h"

/// Adapter to handle differences between vector types

template<class MemberType, class VecType>
struct VectorAdapter
{
  static void Initialize(VecType &vec, unsigned int size, MemberType deflt);
  static void WriteVector(const String &filename, const VecType &vec);
  static void ReadVector(const String &filename, VecType &vec);
};

// Now the specific implementations for particular types. 

/// vecvec. Needed because vecvec is a subclass of mastervec, not a typedef.
template<class MemberType>
struct VectorAdapter
<MemberType, vecvec<typename MemberType::value_type> >
{
  typedef vecvec<typename MemberType::value_type> VecType;
  /// NOTE that the implementation of mastervec doesn't allow
  /// us to make a mastervec that acutally has size_ entries.  The
  /// closest we can come is to make a mastervec that has the space
  /// reserved for that many entries.  You still have to push_back to
  /// make them.
  static void Initialize (VecType &vec, unsigned int size, MemberType deflt)
  {
    ForceAssertGt(size, 0);
    vec.Reserve(deflt.size() * size, size);
  }
  static void WriteVector(const String &filename, const VecType &vec )
  {
    vec.WriteAll(filename);
  }
  static void ReadVector(const String &filename, VecType &vec)
  {
    vec.ReadAll(filename);
  }
};

/// Basevector: 
template<>
struct VectorAdapter
<basevector, vecbasevector >
{
  typedef basevector MemberType;
  typedef vecbasevector VecType;
  /// NOTE that the implementation of mastervec doesn't allow
  /// us to make a mastervec that acutally has size_ entries.  The
  /// closest we can come is to make a mastervec that has the space
  /// reserved for that many entries.  You still have to push_back to
  /// make them.
  static void Initialize (VecType &vec, unsigned int size, MemberType deflt)
  {
    ForceAssertGt(size, 0);
    vec.Reserve(deflt.size() * size, size);
  }
  static void WriteVector(const String &filename, const VecType &vec )
  {
    vec.WriteAll(filename);
  }
  static void ReadVector(const String &filename, VecType &vec)
  {
    vec.ReadAll(filename);
  }
};

/// Mastervec: 
template<class MemberType>
struct VectorAdapter
<MemberType, mastervec<MemberType, typename MemberType::value_type> >
{
  typedef mastervec<MemberType, typename MemberType::value_type> VecType;
  /// NOTE that the implementation of mastervec doesn't allow
  /// us to make a mastervec that acutally has size_ entries.  The
  /// closest we can come is to make a mastervec that has the space
  /// reserved for that many entries.  You still have to push_back to
  /// make them.
  static void Initialize (VecType &vec, unsigned int size, MemberType deflt)
  {
    ForceAssertGt(size, 0);
    vec.Reserve(deflt.size() * size, size);
  }
  static void Initialize (VecType &vec, unsigned int size, longlong rawsize)
  {
    ForceAssertGt(size, 0);
    vec.Reserve(size, rawsize);
  }
  static void WriteVector(const String &filename, const VecType &vec )
  {
    vec.WriteAll(filename);
  }
  static void ReadVector(const String &filename, VecType &vec)
  {
    vec.ReadAll(filename);
  }
};

///Vec:
template<class MemberType>
struct VectorAdapter<MemberType, vec<MemberType> >
{
  typedef vec<MemberType> VecType;
  static void Initialize(VecType &vec, unsigned int size, MemberType deflt)
  {
    ForceAssertGt(size, 0);
    vec.resize(size, deflt);
  }
  static void ReadVector(const String &filename, VecType &vec)
  {
    BinaryRead3(filename, vec);
  }
  static void WriteVector(const String &outname, const VecType & vec)
  {
    BinaryWrite3(outname, vec);
  }
};

/// The PropertyVector holds a map from Strings to DataVectors which
/// are thin wrappers around the VecType that support lazy reading and
/// writing.  The data is marked dirty if a mutable reference to it
/// has ever been obtained.
template<class MemberType, class VecType>
class DataVector {
private:
  // The data members are mutable because we lazy read and write the
  // vector, so the object may be changed by getting logically-const
  // operations such as getting a const reference to data or writing
  // to disk.
  mutable VecType data_;
  mutable bool dirty_;
  mutable String filename_;
public:
  explicit DataVector(const String &filename = String()) :
    dirty_(false), filename_(filename)
  { }
  VecType & data()
  {
    Read();
    dirty_ = true;
    return data_;
  }
  const VecType & data() const {
    Read();
    return data_;
  }
  /// Size of vector, if available; returns default if not yet read to
  /// avoid reading data just to check its size.
  typename VecType::size_type size(typename VecType::size_type deflt) const {
    if (data_.empty())
      return deflt;
    return data_.size();
  }
  /// Finalize read from disk, if available and not yet done.
  void Read() const {
    if (data_.empty() && !filename_.empty())
      VectorAdapter<MemberType,VecType>::ReadVector(filename_, data_);
  }
  /// Write to disk.  If new filename differs from old, finalize any
  /// lazy reads and be sure to write.
  void Write(const String &newfilename) const {
    if (filename_.empty())
      filename_ = newfilename;
    else if (newfilename!=filename_) {
      Read();
      dirty_=true;
      filename_ = newfilename;
    }
    if (!data_.empty() && dirty_) {
      ForceAssert(!filename_.empty());
      VectorAdapter<MemberType,VecType>::WriteVector(filename_, data_);
    }
    dirty_ = false;
  }
};

/// Here's the meat of this file, PropertyVector

template<class MemberType, class VecType>
class PropertyVector {
  friend class PropertyVectorTester;

  typedef DataVector<MemberType, VecType> DataType;
  typedef map< String, DataType > Map;
  // The following 2 members are mutable because we lazy-read the
  // vectors from disk, so they may be set by what are overtly const
  // operations.
  mutable Map M_; //!< The collection of property vectors.
  mutable unsigned int size_; //!< The number of elements of each vector.
  
  ///Save a particular vector to disk.
  void WriteVector(const String &property, const String &filename) const;

 public:
  ///Create an empty PropertyVector class.
  PropertyVector() : size_(0) { }

  ///Load a PropertyVector class from disk.
  PropertyVector( const String &filename): size_(0) {
    Read(filename);
  }

  ///Save PropertyVector to disk.
  void Write(const String &filename) const;

  ///Read PropertyVector from disk.
  void Read(const String & filename);

  ///The length of each vector of information.
  unsigned int size() const {
    if (0!=size_ || M_.empty())
      return size_;
    // Must read something to find out the size
    size_ = M_.begin()->second.data().size();
    return size_;
  }

  bool empty() const { return 0 == size(); }

  /// How to set the size of a property vector.  Note that once the
  ///size is set it is no longer empty().  This cannot be used to
  ///change the size of a nonempty property vector; if you want that,
  ///start a new PropertyVector.
  void setSize(unsigned int size)
  {
    if (size_==0)
      size_ = size;
    else
      ForceAssertEq(size_,size);
  }

  /// Check whether a property exists.  
  bool HasProperty(const String &property) const
  { return (M_.find(property)!=M_.end()); }

  /// Assert that a property exists.  
  void AssertHasProperty(const String &property) const
  {
    if (M_.find(property)==M_.end()) {
      // force diagnostic error message and exit
      FatalErr("PropertyVector: did not find property " << property);
    }
  }

  /// Set of properties available.  Does not cause lazy reads to get finalized.
  set<String> Properties() const;  

  /// Check that the lengths of all vectors are the same. 
  void ValidateLengths() const;
  
  ///Get/set methods for vectors
  ///Get const reference to existing property vector.
  const VecType & GetVector(const String &property) const
  {
    AssertHasProperty(property);
    return M_[property].data();
  }
  
  ///Get a modifyable reference to existing property vector.  
  VecType & GetMutableVector(const String &property)
  {
    AssertHasProperty(property);
    return M_[property].data();
  }
  
  ///Create a new property vector, assuming correct size is known;
  ///initialize to deflt.  If this property is already present,
  ///reinitialize rather than raising an error; this is assumed to be
  ///caused by rerunning a process and we want that to be idempotent.
  VecType & GetNewVector(const String &property, MemberType deflt)
  {
    ForceAssert(size()>0);
    VecType &newvec = M_[property].data();
    VectorAdapter<MemberType, VecType>::Initialize(newvec, size_, deflt);
    return newvec;
  }
  
  ///Create a new property vector or get mutable reference to existing
  ///one.  If this property is already present, do not reinitialize.
  VecType & GetVectorForUpdate(const String &property, MemberType deflt)
  {
    if (HasProperty(property))
      return M_[property].data();
    else
      return GetNewVector(property, deflt);
  }
  
  /// Copy data into property vector, creating one if needed.  If the
  /// PropertyVector was empty() before, this will determine its size.
  /// After that the size cannot change.
  void SetVector(const String &property, const VecType &data) // copies data
  {
    ForceAssert(0==size_ || size_==static_cast<unsigned int>(data.size()));
    M_[property].data() = data;
    setSize(data.size());
  }

  friend bool operator==(const PropertyVector & l, const PropertyVector & r) {
    return (l.M_ == r.M_);
  }

};

/// Implementations.  First, the generic ones.

///Load a PropertyVector class from disk: because we use lazy
///initialization, just read in the available properties and store filenames. 
template<class MemberType, class VecType >
void PropertyVector<MemberType,VecType>::Read( const String &filename)  {
  vector<String> files = AllFilesWithPrefix(Basename(filename+"."),
					    Dirname(filename));
  String basename = filename + ".";
  for (unsigned int i=0; i<files.size(); ++i) {
    String property = files[i].After(basename);
    M_[property] = DataType(files[i]);
  }
}

///Save to disk.
template<class MemberType, class VecType >
void PropertyVector<MemberType,VecType>::Write(const String &filename) const
{
  ValidateLengths();
  for (typename Map::const_iterator it=M_.begin(); it!=M_.end(); ++it) {
    WriteVector(it->first, filename);
  }
}

template<class MemberType, class VecType >
set<String> PropertyVector<MemberType,VecType>::Properties() const
{
  set<String> ans;
  typename Map::const_iterator it = M_.begin();
  while (it!=M_.end()) {
    ans.insert(it->first);
    ++it;
  }
  return ans;
}

template<class MemberType, class VecType >
void PropertyVector<MemberType,VecType>::ValidateLengths() const
{
  typename Map::const_iterator it = M_.begin();
  for (; it!=M_.end(); ++it) {
    ForceAssertEq(size_, it->second.size(size_));
  }
}

template<class MemberType, class VecType>
void PropertyVector<MemberType,VecType>::WriteVector(const String &property,
						     const String &filename) const
{
  M_[property].Write(filename + "." + property);
}


#if 0
/// Currently-unused methods that might be worth making available

  ///Create a histogram of the data in the vector with name "property".
  ///By default, will cover the whole range and have 10 bins.
  void GetHistogram(const String & property, histogram<MemberType> & h,
		    int nbins, MemberType startBin, MemberType binSize) const;

  ///Print a text histogram of the data in the vector with name "property".
  ///By default, will cover the whole range and have 10 bins.
  ///Calls GetHistogram.
  void PrintHistogram(ostream & os, const String & property, 
		      int nbins, MemberType startBin, MemberType binSize) const;

  friend ostream & operator<<(ostream &, const PropertyVector<MemberType,VecType> &);
  
// Write out a PropertyVector object as a table that can be inspected.
ostream & operator<<(ostream &, const PropertyVector<MemberType,VecType> &);

void PropertyVector<MemberType,VecType>::GetHistogram(const String & property, histogram<float> & h,
		    int nbins, float startBin, float binSize) const {
  const vec<float> & v0 = GetVector(property);
  vec<float> v;
  copy_if(v0.begin(), v0.end(), back_inserter(v), finite);
  if (isnan(startBin)) {
    startBin = *min_element(v.begin(), v.end());
  }
  if (isnan(binSize)) {
    binSize = (*max_element(v.begin(), v.end()) - startBin)
      / nbins;
  }
  h.AddLinearProgressionOfBins(startBin, binSize, nbins);
  h.AddData(v.begin(), v.end());
}

void PropertyVector<MemberType,VecType>::PrintHistogram(ostream & os, const String & property,
		    int nbins, float startBin, float binSize) const {
  os << "Histogram for property " << property << endl;
  histogram<float> h;
  GetHistogram(property, h, nbins, startBin, binSize);
  os << h;
}

ostream & operator<<(ostream &out, const PropertyVector<MemberType,VecType> &F)
{
  //First, write the string map.
  out << "StringProperty\tValue\n";
  for (PropertyVector<MemberType,VecType>::Smap::const_iterator i= F.strings_.begin(); 
       i != F.strings_.end(); ++i) {
    out << i->first << "\t" << i->second << "\n";
  }
  
  PropertyVector<MemberType,VecType>::Map::const_iterator it = F.M_.begin();
  const PropertyVector<MemberType,VecType>::Map::const_iterator last = F.M_.end();
  for ( ; it!=last ; ++it) {
    out << (it->first) << "\t";
  }
  out << "\n";
  for (unsigned int i=0; i<F.size(); ++i) {
    it = F.M_.begin();
    for ( ; it!=last ; ++it) {
      out << (it->second[i]) << "\t";
    }
    out << "\n";
  }
  return out;
}

  // Some conversion operators for built-in types are straightforward.

  template <typename X>
  struct ConvertToMemberType {
    MemberType operator()(X in) { return static_cast<MemberType>(in); }
  };

  template <typename X>
  struct ConvertFromMemberType {
    X operator()(MemberType in) { return static_cast<X>(in); }
  };

  ///Template get/set methods for types that can be translated to/from MemberType.
  /// GetVectorX: op(MemberType y) returns an X constructed from y.
  template <typename X, typename Op>
  void GetVectorX(const String &property, vector<X> &out, Op op)
  {
    const VecType &in = GetVector(property);
    out.resize(in.size());
    transform(in.begin(), in.end(), out.begin(), op);
  }
  /// SetVectorX: op(X x) returns a MemberType representing x.
  template <typename X, typename Op>
  void SetVectorX(const String &property, const vector<X> &data, Op op)
  {
    VecType save(data.size());
    transform(data.begin(), data.end(), save.begin(), op);
    SetVector(property, save);
  }


#endif

#endif //PROPERTY_VECTOR_H
