/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/** Generate quality scores for data derived from an sff file.

\file QualsFromSff.cc

Expects the sff file(s) to have been processed first with MakeSffInfo

Saves quals in the sff binary file forest if SAVE is true.

Also saves quals in fasta format if a QUAL destination file is given.

*/

#ifndef FORCE_DEBUG
#define NDEBUG
#endif


#include "MainTools.h"
#include "454/sff/SffTypes.h"
#include "454/sff/SffInfo.h"
#include "454/sff/SffRead.h"
#include "454/qual/PhredTableReader.h"



int main( int argc, char *argv[])
{ 
  RunTime();

  BeginCommandArguments;
  // The sffinfo file with the data. This should be the header name for a
  // forest of binary sffinfo files that have been processed and aligned by
  // us. It must be a : separated list if MULTIPLE_SFFINFOS is True.
  CommandArgument_String(SFFINFO);
  // PhredTable in text format.
  CommandArgument_String(TABLE);
  // If we want text output, give a name for the file.
  CommandArgument_String_OrDefault(QUAL,"");
  // If true, save the just calculated quals
  CommandArgument_Bool_OrDefault(SAVE,True);
  // Minimum quality score
  CommandArgument_UnsignedInt_OrDefault(MIN,5);
  // Maximum quality score
  CommandArgument_UnsignedInt_OrDefault(MAX, 35);
  // Subtract this value from calculated score before capping at [MIN,MAX]
  CommandArgument_Int_OrDefault(SUBTRACT, 1);
  EndCommandArguments;


  cout << "Reading in sffinfo " << SFFINFO << endl;
  SffInfo info;
  info.Read(SFFINFO);

  cout << "Reading in table from " << TABLE << endl;
  SffPredictorParameters predParamHandler;
  PhredTableReader p(TABLE, false, &predParamHandler);
  p.ResetOverflow();

  cout << "Setting quality scores " << endl;
  info.SetQualsFrom(p, MIN, MAX, SUBTRACT);

  //Print to stdout a summary of the qualities of clipped reads.
  const vecqualvector & q = info.GetQualvecVector(SffInfo::QUALS);
  longlong total=0, q20=0, q25=0, q30=0;
  for (SffRead r(info); r.Index() != info.isize(); ++r) {
    pair<int,int> clips = r.Clips();
    for (int b= clips.first; b < clips.second;  ++b) {
      const unsigned char thisq = r.Quals()[b];
      ++total;
      if (thisq >=20) ++q20;
      if (thisq >=25) ++q25;
      if (thisq >=30) ++q30;
    }
  }
  cout << "\nTotal bases: " << total
       << ", q20 bases: " << q20
       << ", q25 bases: " << q25
       << ", q30 bases: " << q30
       << endl;

  if (!QUAL.empty()) info.WriteQuals(QUAL);
  if (SAVE) info.Write(SFFINFO);

  cout << p.Overflow() << "/" << total 
  	<< " bases beyond the table boundaries\n";
  return 0;
}
