/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "454/qual/PhredTableReader.h"

longlong PhredTableReader::outerLookupCounter =0;
longlong PhredTableReader::innerLookupCounter =0;
longlong PhredTableReader::outerFLookupCounter =0;
longlong PhredTableReader::innerFLookupCounter =0;

const unsigned char PhredTableReader::ZERO_BECOMES_CHAR_0 = 48;

PhredTableReader::~PhredTableReader() {}

PhredTableReader::PhredTableReader(const String & fname, 
				   bool verboseSetup,
				   PredictorParameterHandler * h):
  PhredTableBase(), firstBin_(), validLines_(), 
  compactedTable_(), maxBin_(), overflow_(0) {

  Ifstream(is, fname);
  PhredTableBase::ReadPhredTable(is, h);
  Init(verboseSetup);
  ReadCompacted(is);
}
  
PhredTableReader::PhredTableReader(const PhredTableBase & base):
  PhredTableBase(base), firstBin_(), validLines_(), 
  compactedTable_(), maxBin_(), overflow_(0) {
  Init();
}
  

int PhredTableReader::SlowLookup(vec<float> & values) const {
  AssertEq(values.size(), m_table[0].size() - 2);
  CapValues(values);
  
  const int TABLE_SIZE =  m_table.isize();
  const int NPREDICTORS = values.isize();
  for (int i=0; i != TABLE_SIZE; ++i) {
    bool found=true;
    const vec<float> & line = m_table[i];
    ++outerLookupCounter;
    for (int j=0; j != NPREDICTORS; ++j) {
      ++innerLookupCounter;
      if (values[j] > line[j]) {found=false; break;}
    }
    if (found) {
      return int(line.back());
    }
  }
  return 0;
}



int PhredTableReader::FastLookup(vec<float> & values) const {
  static vec<unsigned int> bins(values.size());
  AssertEq(values.size(), m_table[0].size() - 2);
  CapValues(values);
  const int TABLE_SIZE =  m_table.isize();
  const int NPREDICTORS = values.isize();
  
  ToBins(values, bins);

  int topbin = 0;
  for (int i= 0; i != NPREDICTORS; ++i) {
    if (bins[i] >= firstBin_[i].size()) {
      PRINT4(i, values, bins, firstBin_[i]);
      PRINT(binMapper_[i]);
      }
    topbin = max(firstBin_[i][bins[i]], topbin);
  }

  for (int i=topbin; i != TABLE_SIZE; ++i) {
    bool found=true;
    const vec<float> & line = m_table[i];
    ++outerFLookupCounter;
    for (int j=0; j != NPREDICTORS; ++j) {
      ++innerFLookupCounter;
      if (values[j] > line[j]) {found=false; break;}
    }
    if (found) {
      return int(line.back());
    }
  }
  return 0;
}

template<class V>
bool AllEqual(const V & v) {
  for (int i=1; i < int(v.size()); ++i) {
    if (v[i] != v[i-1]) return false;
  }
  return true;
}

/// Given the current best line for each predictor, 
/// find the worst one, and find lines that will go with that
/// worst one, and put then into nextlines.
/// Not implemented because CompactLookup is faster.
void PhredTableReader::FindNextLines(vec<int> & nextlines, 
		   const vec<vec<vec<int> > > & validLines,
		   const vec<unsigned int> & bins) const {}
  
/** Not completely implemented because CompactLookup is faster
The idea is that we have built a table of what lines are valid for what
threshold values for each predictor in validLines_, and knowing the 
values of the bins we look down that table, skipping down to the next
valid line for each, until we reach a line that is the same for all
the predictors.
*/
int PhredTableReader::VeryFastLookup(vec<float> & values) const {
  const int NPREDICTORS = npredictors();
  AssertEq(values.isize(), NPREDICTORS);
  CapValues(values);
  
  vec<unsigned int> bins(NPREDICTORS);
  ToBins(values, bins);

  vec<int> nextlines(NPREDICTORS);
  do {
    FindNextLines(nextlines, validLines_, bins);
  }
  while (!AllEqual(nextlines));

  if (nextlines[0] < m_table.isize()) return int(m_table[nextlines[0]].back());
  else return 0;
}

void PhredTableReader::InitBinMapper() {
  ForceAssert(!m_table.empty());
  binMapper_.clear();
  const int NPREDICTORS = npredictors();
  for (int i=0; i!= NPREDICTORS; ++i) {

    vec<float> column(m_table.size());
    for (int j=0; j != column.isize(); ++j) column[j]=m_table[j][i];

    //find all bin edges
    vec<float> uniquevalues(column);
    sort(uniquevalues.begin(), uniquevalues.end());
    uniquevalues.erase(unique(uniquevalues.begin(), uniquevalues.end()),
	uniquevalues.end());
    //PRINT2(i,uniquevalues);

    //set up bin mapper
    binMapper_.push_back(FloatMap(uniquevalues));
  }
}



//need to set up firstBin_, maxBin_, binMapper_ and validLines_
void PhredTableReader::Init(bool print) {
  InitBinMapper();
  ForceAssert(!m_table.empty());
  const int NPREDICTORS = npredictors();
  maxBin_.resize(NPREDICTORS);
  firstBin_.resize(NPREDICTORS);
  validLines_.resize(NPREDICTORS);
  dimensions_.resize(NPREDICTORS);
  //PRINT2(m_table.size(), m_table[0].size())
  for (int i=0; i!= NPREDICTORS; ++i) {

    vec<float> column(m_table.size());
    for (int j=0; j != column.isize(); ++j) column[j]=m_table[j][i];

    //find all bin edges
    vec<float> uniquevalues(column);
    sort(uniquevalues.begin(), uniquevalues.end());
    uniquevalues.erase(unique(uniquevalues.begin(), uniquevalues.end()),
		       uniquevalues.end());
    //PRINT2(i,uniquevalues);

    const int NBINS = uniquevalues.size();
    dimensions_[i] = NBINS;
    maxBin_[i] = *max_element(uniquevalues.begin(), uniquevalues.end());

    //set up firstBin array.
    firstBin_[i].assign(NBINS, -1);
    validLines_[i].resize(NBINS);
    for (int bin = 0; bin != NBINS; ++bin) {
      for (int line = 0; line != m_table.isize(); ++line) {
	if (m_table[line][i] >= uniquevalues[bin]) {
	  if (-1 == firstBin_[i][bin]) firstBin_[i][bin] = line;
	  validLines_[i][bin].push_back(line);
	}
      }
    } 
  } //end of for each predictor loop

  if (!print) return;

  //print out internal tables for visual verification.
  for (int i=0; i != NPREDICTORS; ++i) {
    PRINT2(i, binMapper_[i]);
    copy(firstBin_[i].begin(), firstBin_[i].end(),
	 ostream_iterator<float>(cout,"\t"));
    cout << endl;
    for (int valid=0; valid != validLines_[i][0].isize(); ++valid) {
      for (int bin = 0; bin != firstBin_[i].isize(); ++bin) {
	if (valid < validLines_[i][bin].isize()) {
	  cout << validLines_[i][bin][valid];
	}
	cout << "\t";
      }
      cout << "\n";
    }
  } 
}		  

void PhredTableReader::Compact() {
  ForceAssert(isInited());
  ulonglong MAXDIM=1;
  for (int i = 0; i != binMapper_.isize(); ++i) {
    MAXDIM *= dimensions_[i];
  }
  compactedTable_.assign(MAXDIM, 0);

  vec<unsigned int> bins;
  vec<float> values(dimensions_.size());
  for (ulonglong i = 0; i != MAXDIM; ++i) {
    bins = FindIndices(i, dimensions_);
    for (int b=0; b!= bins.isize(); ++b) values[b] = binMapper_[b].Bin(bins[b]);
    compactedTable_[i] = FastLookup(values);
  }
}

int PhredTableReader::CompactLookup(vec<float> & values) const {
  vec<unsigned int> bins(values.size());
  CapValues(values);
  ToBins(values, bins);
  int tuple = FindTuple(bins, dimensions_);
  return compactedTable_[tuple];
}
  
void PhredTableReader::WriteCompacted(ostream & os) {
  const unsigned int N=compactedTable_.size();
  ForceAssert(N);
  vec<unsigned char> readable(N);
  //qual 0 becomes the char '0' and so on, so this is sort of human-readable
  for (unsigned int i=0; i !=N; ++i) {
    readable[i] = compactedTable_[i] + ZERO_BECOMES_CHAR_0;
  }
  os.write((char *) &readable[0], N);
  os << "\n";
}

void PhredTableReader::ReadCompacted(istream & is) {
  ForceAssert(isInited());
  unsigned int tablesize = 
    accumulate(dimensions_.begin(), dimensions_.end(),
	       1, multiplies<unsigned int>());
  compactedTable_.resize(tablesize);
  is.read((char *) &compactedTable_[0], tablesize);
  if (is.good()) {
    for (unsigned int i=0; i != tablesize; ++i) { 
      compactedTable_[i] = compactedTable_[i] - ZERO_BECOMES_CHAR_0;
    }
  }
  else FatalErr("could not read compacted table.");
}
    
    
