/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef PHRED_TABLE_BASE
#define PHRED_TABLE_BASE

#include "CoreTools.h"
#include "AlignErr.h"
#include "454/qual/PredictorParameterHandler.h"

#include <functional>
#include <numeric>
#include <algorithm>
#include <map>
#include <sstream>

/// Base class for Writers and Readers of PhredTables.
///
/// \class PhredTableBase
///
/// Provides some common instance variables and methods used
/// by both PhredTableWriter and PhredTableReader

class PhredTableBase {

protected:
  // Default handler does nothing.
  struct NullPredictorHandler: public PredictorParameterHandler {
    virtual String GetPredictorParameters() const { return ""; }
    virtual void SetPredictorParameters(const String & params) {}
    virtual ~NullPredictorHandler() {}
  };

  class FloatMap;//forward declaration

  friend class TestPhredTable;
  vec<vec<float> > m_table; ///< table of thresholds, sizes, and qualities.
  vec<unsigned int> dimensions_; ///< the dimensions for the combo vector
  vec<FloatMap> binMapper_; ///< to translate predictor values to bin indices.

public:

  virtual ~PhredTableBase();

  ///Don't declare this as abstract, because it's useful to create an instance.
  virtual int npredictors() const { return -1; }

 /// Return the index in a 1-D matrix of the item with the N-dimensional
  /// coordinates indices in the matrix with the given dimensions.
  /// FindIndices does the inverse transformation.
  ulonglong FindTuple(const vec<unsigned int> & indices,
			 const vec<unsigned int> & dimensions) const {
    AssertEq(indices.size(), dimensions.size());
    ulonglong index = 0;
    ulonglong blocksize = 1;
    for (unsigned int i = 0; i != indices.size(); ++i) {
      index += indices[i] * blocksize;
      blocksize *= dimensions[i];
    }
    return index;
  }

  /// Return the index in a 1-D matrix of the item with the
  /// N-dimensional coordinate indices that are the element-by-element
  /// minimum of the two index vectors passed in, in the matrix with
  /// the given dimensions.  
  ulonglong FindTuple(const vec<unsigned int> & indices1,
		      const vec<unsigned int> & indices2,
		      const vec<unsigned int> & dimensions) const {
    AssertEq(indices1.size(), dimensions.size());
    AssertEq(indices1.size(), indices2.size());
    ulonglong index = 0;
    ulonglong blocksize = 1;
    for (unsigned int i = 0; i != indices1.size(); ++i) {
      index += min(indices1[i], indices2[i]) * blocksize;
      blocksize *= dimensions[i];
    }
    return index;
  }

  /// Inverse of FindTuple: given the index in a 1-D matrix that is a 
  /// linearization of an N-dim matrix, return a vector containing the
  /// indices into the N-dim matrix.
  vec<unsigned int> FindIndices(ulonglong tuple,
				const vec<unsigned int> & dimensions) const {
    ulonglong block=1;
    vec<unsigned int> indices(dimensions.size());
    vec<ulonglong> blocksizes;
    for (unsigned int i=0; i != dimensions.size(); ++i) {
      blocksizes.push_back(block);
      block *= dimensions[i];
    }
    for (int i=dimensions.isize()-1; i >=0; --i) {
      indices[i] = tuple / blocksizes[i];
      tuple -= indices[i] * blocksizes[i];
    }
    return indices;
  }


  // Must be kept in sync with ReadPhredTable.
  void PrintPhredTable(ostream &out,  PredictorParameterHandler *h =
		       new NullPredictorHandler()) const {
    out << m_table.size() << " lines " << npredictors() << " predictors\t";
    if (h) {
      out << "PredictorParameters:\t" << h->GetPredictorParameters()
	  << "\n";
    }
    for (int i=0; i != m_table.isize(); ++i) {
      copy(m_table[i].begin(), m_table[i].end(),
	   ostream_iterator<float>(out," "));
      out << "\n";
    }
  }

  void OldReadPhredTable(istream &in)  {
    string inl;
    getline(in, inl);
    float tmp;
    while (in) {
      m_table.push_back(vec<float>());
      istringstream iss(inl);
      iss >> tmp;
      while (iss) {
	m_table.back().push_back(tmp);
	iss >> tmp;
      }
      getline(in, inl);
    }
  }

  // Must be kept in sync with PrintPhredTable
  void ReadPhredTable(istream &in, PredictorParameterHandler * h 
		      = new NullPredictorHandler())  {
    int lines, predictors;
    //use std::string to avoid the fact that operator>> for String
    //consumes the following whitespace.
    string verifyl, verifyp; 
    in >> lines >> verifyl >> predictors >> verifyp;
    ForceAssert(verifyl == "lines");
    ForceAssert(verifyp == "predictors");
    String params;
    getline(in, params);
    if (params.Contains("PredictorParameters:")) {
      Assert(h);
      h->SetPredictorParameters(params.After("PredictorParameters:"));
    }

    m_table.resize(lines);
    const int PER_LINE= predictors + 2;
    for (int i=0; i != lines; ++i) {
      m_table[i].resize(PER_LINE);
      for (int j=0; j != PER_LINE; ++j) {
	in >> m_table[i][j];
      }
    }
    ForceAssert(in.good());
    getline(in, verifyl); //get rid of the last newline before compactedTable.
  }  

protected:

  /// transform a vector of floats into the corresponding integer bins.
  void ToBins(const vec<float> & values, vec<unsigned int> & bins) const;

  /// Return the index of a float in a sorted vector thereof.
  /// If the float is not in the vector, return index of the next
  /// highest element. If there is no higher element, return
  /// size() (this should never happen if the data have been
  /// passed through CapValues()).
  class FloatMap: public unary_function<float, int> {
  private:
    vec<float> bins_;
    int TOP_BIN;

  public:
    FloatMap(const vec<float> & bins): bins_(bins) {
      sort(bins_.begin(), bins_.end());
      bins_.erase(unique(bins_.begin(), bins_.end()), bins_.end());
      TOP_BIN = bins_.size() - 1;
    }

    int size() { return bins_.size(); }

    int operator()(float f) const {
      int ret = lower_bound(bins_.begin(), bins_.end(), f) - bins_.begin();
      return min(ret, TOP_BIN);
    }

    float Bin(int i) { return bins_[i]; }

    friend ostream & operator<<(ostream & os, const FloatMap & m) {
      copy(m.bins_.begin(), m.bins_.end(), ostream_iterator<float>(os, " "));
      return os;
    }
  };

};

#endif //PHRED_TABLE_BASE
