#ifndef FLOWVECTOR_OPS_H
#define FLOWVECTOR_OPS_H
/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// FlowVectorOps.h

/// Perform operations on FlowVectors.
/// \file FlowVectorOps.h
/// 
/// FlowVector allows for trimming at the end, but it never actually 
/// removes trimmed flows: they can be retrieved with UntrimEnd() in 
/// case they are needed later.


class String;
class FlowKey;
class FlowVector;
class FlowOrder;

template<class T> class vec;
template<class T> class serfvec;

struct normal_distribution;
class basevector;

///Normalize so that the mean of the 1-peaks in the key = poskeyMean.
///Version with a known key.
/// if key is Bad(), then fvec is left unchanged.
void NormalizeToPosKey(double poskeyMean, FlowVector & fvec, 
                       const FlowKey & key);


///Normalize so that the mean of the 1-peaks in the key = poskeyMean.
///This version will calculate the key.
void NormalizeToPosKey(double poskeyMean, FlowVector & fvec, 
                       const FlowOrder & order);

///Normalize so that the mean flow score for the indices is mean.
/// That is, ensure that the average vec[i] for in in the input
/// vec indices equals mean.
void NormalizeToIndices(double mean, const vec<int> & indices,
                        FlowVector & vec);

/// Use affine transform to set mean0 to 0.0 and mean1 to 1.0.
/// and set all
/// other values accordingly. The idea is that we want the noise centered
/// at 0, the singlet centered at 1, and therefore a pretty good 
/// approximation of the n-peak centered at n.
void NormalizeToZeroOne( double mean0, double mean1, FlowVector & fvec);

  /// Normalize so that the middle of each cutoff range becomes an integer.
  /// That integer is the index for that cutoff range. The cutoffs 
  /// must be given in a scale of value_type, not in a scale of double.
  /// performs a piecewise affine transform
  /// based on the cutoffs so that the cutoffs become 0.5,1.5,2.5, etc... in
  /// the modified flowvector. The cutoffs need to be in ascending order, but
  /// do not need to be equally spaced.
  void NormalizeWithCutoffs(const vec<double> & cutoffs, FlowVector & vec);

  void MeanStdev(const vec<const FlowVector *> & fvecs,
		 vec<normal_distribution> & distributions);

  /// Transform into a vec<double> (for compatibility with other code).
  void TransformToDoubles(const FlowVector & fvec, vec<double> & result);

  /// Transform into a vec<float> (for compatibility with other code).
  void TransformToFloats(const FlowVector & fvec, vec<float> & result);

  /// Transform into a serfvec<float> (for compatibility with other code).
  void TransformToFloats(const FlowVector & fvec, serfvec<float> & result);

  /// Normalize into a vec<double> using cheewhye's normalization.
  /// If normalization fails, returns false.
  bool NormalizeToDoubles(const FlowVector & fvec, const FlowOrder &order,
			  const String &key, vec<double> & result);

  /// Normalize into a vec<double> using cheewhye's normalization.  If
  /// normalization succeeds, return 0; if it fails, returns positive
  /// error code.  Be careful switching between these two versions,
  /// because NormalizeToDoubles() is equivalent to !NormalizeToDoublesVerbose().
  int NormalizeToDoublesVerbose(const FlowVector & fvec, const FlowOrder &order,
				const String &key, vec<double> & result);

  /// Create a vec<float> representing the flows we would get from
  /// "bases".  Note that the result always starts with a nonzero
  /// flow, hence does not necessarily start with the first base in
  /// order.

  vec<float> FlowsFromBases(const FlowOrder & order, const basevector & bases,
			    const double noise=0.0, vec<int> * correspondence = 0);

  ///count consecutive elements above and below the cutoff.
  /// class V must provide size() and operator[] , with the type
  /// returned being comparable to double.
template<class V>
void CountConsecutive(const V & v, double cutoff, 
		      int & noise, int & signal) {
  int n=0, s=0;
  noise=0; 
  signal=0;
  const int S=v.size();
  for (int i=0; i != S; ++i) {
    if (v[i] > cutoff) {
      if (++s > signal) signal=s;
      n=0;
    } else {
      if (++n > noise) noise=n;
      s=0;
    }
  }
}

/// Find the last good position in vector.
/// good defined as : must always have fewer than max_below in a row below 
/// the cutoff, must always have less than max_above in a row above the
/// cutoff 
int LastGood(const FlowVector & v, double cutoff, 
	      int max_below, int max_above);


#endif // FLOWVECTOR_OPS_H
