/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "454/flowdata/FlowVectorOps.h"
#include "454/flowdata/FlowVector.h"
#include "454/flowdata/FlowOrder.h"
#include "454/flowdata/FlowKey.h"
#include "454/NormalizeFlowVec.h"
#include "STLExtensions.h"

#include "Vec.h"
#include "math/Functions.h"
#include "Basevector.h"

#include <deque>

void NormalizeToPosKey(double poskeyMean, FlowVector & fvec, 
    const FlowKey & key) {
  //Don't do anything if the key is bad.
  if (key.Bad()) return;
  const int S = fvec.size();
  double factor = poskeyMean / key.PosMean(fvec);
  vec<FlowVector::value_type> data(fvec.begin(), fvec.untrimmedEnd());
  data *= factor;
  fvec.SetData(1.0, data);
  fvec.SetNewEnd(S);
}

void NormalizeToPosKey(double poskeyMean, FlowVector & fvec, 
    const FlowOrder & order) {
  FlowKey key(order, fvec);
  NormalizeToPosKey(poskeyMean, fvec, key);
}

void NormalizeToIndices( double mean, const vec<int> & indices,
    FlowVector & fvec) {
  Assert(!indices.empty());
  double current = 0;
  for (int i = 0; i != indices.isize(); ++i) {
    current += fvec[indices[i]];
  }
  current /= indices.size();
  double factor = mean/current;
  const int S = fvec.size();
  vec<FlowVector::value_type> data(fvec.begin(), fvec.untrimmedEnd());
  data *= factor;
  fvec.SetData(1.0, data);
  fvec.SetNewEnd(S);
}  

void NormalizeToZeroOne( double mean0, double mean1, FlowVector & fvec) {
  //use affine transform to set mean0 to 0.0 and mean1 to 1.0
  //a*mean0 +b = 0, a*mean1+b = 1 =>
  // a(mean1 - mean0) = 1 => 
  double a = 1/(mean1 - mean0);
  double b = -mean0/(mean1 - mean0);
  vec<double> data(fvec.untrimmedSize());
  const int S = fvec.size();
  for (int i=0; i !=data.isize(); ++i) {
    data[i] = a * fvec.Get(i) + b;
    if (data[i] < 0) data[i]= 0;
  }
  fvec.SetData(data);
  fvec.SetNewEnd(S);
}

void NormalizeWithCutoffs( const vec<double> & cutoffs,
    FlowVector & fvec) {
  ForceAssert(cutoffs.size() > 2);
  ForceAssert(is_sorted(cutoffs.begin(), cutoffs.end()));
  vec<float> data(fvec.untrimmedSize());
  const int S = fvec.size();
// We normalize everything! That means we have to use Get(i)
// instead of fvec[i].
  for (int i = 0; i != fvec.untrimmedSize(); ++i) {
    //find the right cutoff.
    int c=0;
    while (c < cutoffs.isize() && fvec.Get(i) > cutoffs[c] ) {
      ++c;
    }
    if (c==cutoffs.isize()) {//above our cutoff: cap to max.
      data[i] = cutoffs.size();
    } else {
      //need to normalize noise values between 0 and 0.5, not
      //between -0.5 and 0.5, so we set bottom to a negative number.
      double bottom = c > 0 ? cutoffs[c-1] : -cutoffs[0];
      double top = cutoffs[c];

      //now adjust the value within that cutoff.
      double pos = (fvec.Get(i) - ((top+bottom)/2)) / (top - bottom);
      data[i] = c + pos;
    }
  }
  fvec.SetData(data);
  fvec.SetNewEnd(S);//now we set the end back where it used to be.
}  



void MeanStdev(const vec<const FlowVector *> & fvecs,
    vec<normal_distribution> & distributions) {
  const int VECS = fvecs.size();
  const int S= fvecs[0]->size();
  distributions.resize(S);
  vec<float> current(VECS);
  for (int i = 0; i != S; ++i) {
    for (int j = 0; j != VECS; ++j) {
      current[j] = (*fvecs[j])[i];
    }
    distributions[i] = SafeMeanStdev(current);
  } 
}   

void TransformToDoubles(const FlowVector & fvec, vec<double> & result) {
  const int S= fvec.size();
  result.resize(S);
  for (int i=0; i != S; ++i) { result[i] = fvec(i); }
}

void TransformToFloats(const FlowVector & fvec, vec<float> & result) {
  const int S= fvec.size();
  result.resize(S);
  for (int i=0; i != S; ++i) { result[i] = fvec(i); }
}

void TransformToFloats(const FlowVector & fvec, serfvec<float> & result) {
  const int S= fvec.size();
  result.resize(S);
  for (int i=0; i != S; ++i) { result[i] = fvec(i); }
}

bool NormalizeToDoubles(const FlowVector & fvec, const FlowOrder &order,
			const String &key, vec<double> & result) {
  return 0==NormalizeToDoublesVerbose(fvec, order, key, result);
}

int NormalizeToDoublesVerbose(const FlowVector & fvec, const FlowOrder &order,
			      const String &key, vec<double> & result) {
  TransformToDoubles(fvec, result);
  FlowStats stats;
  return FlowVecNormalizer(order, key)(result, stats);
}

///helper method for FlowsFromBases.
/// noise for 0 flows is compressed to 0-0.5, for others is
/// [h-0.5,h+0.5).
float FlowNoise(double noise, int height) {
  if (0 == height) return float(noise * drand48() * 0.5);
  else return float(height + noise * (drand48() - 0.5));
}

vec<float> FlowsFromBases(const FlowOrder & order, const basevector & bases,
    const double noise, vec<int> * correspondence) {
  vec<int> c;
  const int N = bases.size();
  vec<float> ret;
  int o=-1;
  int height=0;
  // Skip o ahead to the first base in bases without adding flows
  while (order[++o] != as_base(bases[0])) 
    ;
  // Now process normally
  for (int b=0; b!= N;) {
    height=1;
    c.push_back(ret.size()); 
    while (++b != N && as_base(bases[b]) == order[o]) {
      ++height;
      c.push_back(ret.size());
    }
    ret.push_back(FlowNoise(noise, height));
    if (b!=N) {
      while (order[++o] != as_base(bases[b])) {
	ret.push_back(FlowNoise(noise,0));
      }
    }
  }
  AssertEq(c.size(), bases.size());
  if (correspondence) correspondence->swap(c);
  return ret;
}

int LastGood(const FlowVector & v, double cutoff, 
	      int max_below, int max_above) {
  int b = 0, a = 0;
  const int S=v.size();
  for (int i=0; i != S; ++i) {
    if (v[i] > cutoff) {
      if (++a > max_above) return i - max_above;
      b=0;
    } else {
      if (++b > max_below) return i - max_below;
      a=0;
    }
  }
  return S-1;
}

