// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowVectorManager.cc


#include "454/flowdata/FlowVectorManager.h"

#include "assembly/PrefetchStrategy.h"
#include "STLExtensions.h"

  FlowVectorManager::FlowVectorManager()
: mpFlowValuesMgr( new FlowValuesManager ),
  mpPrefetchStrategy( new NullPrefetchStrategy )
{
}


  FlowVectorManager::FlowVectorManager( const String & strFlowVectorsFile )
: mpFlowValuesMgr( new FlowValuesManager( strFlowVectorsFile ) ),
  mpPrefetchStrategy( new NullPrefetchStrategy )
{
}



FlowVectorManager::~FlowVectorManager()
{
  delete mpFlowValuesMgr;
  delete mpPrefetchStrategy;
}



const FlowOrder &
FlowVectorManager::GetFlowOrder() const
{
  return mpFlowValuesMgr->GetPlusData();
}



const FlowVector &
FlowVectorManager::GetFlowVector( const int id ) const
{
  EnsureLoaded(id);
  return mpFlowValuesMgr->GetData( id );
}



int
FlowVectorManager::GetLength( const int id ) const
{
  return GetFlowVector( id ).size();
}



  void
FlowVectorManager::SetFlowOrder( const FlowOrder & floworder )
{
  mpFlowValuesMgr->SetPlusData( floworder );
}



  void
FlowVectorManager::SetFlowVector( const int id,
    const FlowVector & flowvector )
{
  mpFlowValuesMgr->SetData( id, flowvector );
}



bool
FlowVectorManager::Verify( const int id ) const
{
  // Nothing to be done.
  // This method is here only for compatibility with  ReadSequenceManager,
  // which has to verify the consistency of the various data sets it maintains;
  // since we have only one FlowValuesMgr to handle, we have nothing
  // to verify.
  return true;
}



  void
FlowVectorManager::Write( const bool bOverwrite,
    const String & strFlowVectorsFile )
{
  mpFlowValuesMgr->Write( bOverwrite, strFlowVectorsFile );
}



  void
FlowVectorManager::SetPrefetchStrategy( PrefetchStrategy * pPrefetchStrategy )
{
  delete mpPrefetchStrategy;
  mpPrefetchStrategy = pPrefetchStrategy->Clone();
}

void FlowVectorManager::RemovePFlows() {
  mpFlowValuesMgr->LoadAllData();
  //find positions and remove p flows from the FlowOrder
  FlowOrder order = mpFlowValuesMgr->GetPlusData();
  vec<int> positions = order.PPositions();
  if (positions.empty()) return; //No P flows already! Good!

  
  Assert(is_sorted(positions.begin(), positions.end()));
  order.SetPPositions(vec<int>());
  mpFlowValuesMgr->SetPlusData(order);
  
  //remove P flows from each FlowVector
  const int S = mpFlowValuesMgr->Size();
  FlowVector v;
  for (int i = 0; i != S; ++i) {
    v = mpFlowValuesMgr->GetData(i);
    //because the positions vector is sorted, we can proceed from the 
    //back and not worry about the relative positions of the p flows 
    //changing.
    for (int j =positions.size()-1; j >= 0; --j) {
      v.Erase(positions[j]);
    }
    mpFlowValuesMgr->SetData(i, v);
  } 
}

  void FlowVectorManager::EnsureLoaded(int id) const {
    if ( ! mpFlowValuesMgr->IsLoaded( id ) )
    {
      if ( mpPrefetchStrategy->GetAll() )
      {
	mpFlowValuesMgr->LoadAllData();
      }
      else
      {
	vec<int> vecIds;
	mpPrefetchStrategy->GetIdsToLoad( id, vecIds );
	mpFlowValuesMgr->LoadData( vecIds );
      }
    }
  }


