// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowVector.h

/// Manages the collection of flow values which constitute a flow.
/// \class FlowVector
/// 
/// FlowVector allows for trimming at the end, but it never actually 
/// removes trimmed flows: they can be retrieved with UntrimEnd() in 
/// case they are needed later.

#ifndef FLOWDATA_FLOWVECTOR
#define FLOWDATA_FLOWVECTOR


#include "system/Types.h"
#include "Vec.h"
#include "Feudal.h"

class basevector;
class FlowOrder;

class FlowVector
{
  friend class TestFlowUtils;//for testing internals.
  friend class FlowVectorTester;//for testing internals.

 public:

  typedef unsigned short value_type;
  typedef value_type * iterator;

  /// Create an empty FlowVector
  FlowVector()
    : mCapacity_SelfOwnedBit( TopBit32 ),
      mpMantissas( 0 ),
      mMultiplier( 1.0 ),
      mLength( 0 ),
      mUntrimmedLength( 0 )
  {}


  /// Create a Flowvector of a given length and extra space.
  /// Both length and extra space are in units of value_type.
  FlowVector( unsigned int length,
              unsigned int extraspace )
  {
    mLength = length;
    mUntrimmedLength = length;
    mCapacity_SelfOwnedBit = (length + extraspace) | TopBit32;
    mpMantissas = new value_type [ length + extraspace ];
    mMultiplier = 1.0;
  }


  FlowVector( const FlowVector & fvec):
    mCapacity_SelfOwnedBit( TopBit32 | fvec.capacity() ),
    mpMantissas( new value_type[fvec.capacity()]),
    mMultiplier( fvec.mMultiplier ),
    mLength( fvec.mLength ),
    mUntrimmedLength( fvec.mUntrimmedLength )
  {
    ForceAssert( fvec.mpMantissas != 0 );
    memcpy(mpMantissas, fvec.mpMantissas,
           fvec.mUntrimmedLength * sizeof(value_type) );
  }

  FlowVector & operator = ( const FlowVector & flowvec );


  ~FlowVector()
  {
    Reinitialize();
  }


  bool empty() const
  {  return 0 == size();  }

  unsigned int size() const
  {  return mLength;  }

  ///Return size as an integer for typing loops fast (for (int i=....).
  int isize() const { return size(); }

  int untrimmedSize() const {
    return mUntrimmedLength;
  }

  int capacity() const {
    return mCapacity_SelfOwnedBit & Bits31;
  }

  iterator begin() { return mpMantissas; }

  iterator end() { return mpMantissas + size(); }

  const iterator begin() const { return mpMantissas; }

  const iterator end() const { return mpMantissas + size(); }

  const iterator untrimmedEnd() const {return mpMantissas + untrimmedSize();}

  void Swap( FlowVector & flowvec )
  {
    swap( mpMantissas, flowvec.mpMantissas );
    swap( mMultiplier, flowvec.mMultiplier );
    swap( mLength, flowvec.mLength );
    swap( mUntrimmedLength, flowvec.mUntrimmedLength );
    swap( mCapacity_SelfOwnedBit, flowvec.mCapacity_SelfOwnedBit );
  }



  /// given an index, return the unsigned short mantissa of the flow value
  value_type operator [] ( const int index ) const
  {
    AssertLt( index, static_cast<int>(mLength) );
    return mpMantissas[ index ];
  }

  /// given an index, return or assign the value_type mantissa.
  value_type & operator [] ( const int index )
  {
    AssertLt( index, static_cast<int>(mLength) );
    return mpMantissas[ index ];
  }

  ///Return value at index, even if it has been trimmed.
  value_type Get(const int index) const {
    AssertLt( index, static_cast<int>(mUntrimmedLength) );
    return mpMantissas[ index ];
  }

  ///Remove the element at i, and move down all others appropriately.
  void Erase(unsigned int i) {
    AssertLt(i, size());
    int bytes = (size() -i-1)* sizeof(value_type);
    memmove(mpMantissas+i, mpMantissas+i+1, bytes);
    --mLength;
    --mUntrimmedLength;
  }

  ///Remove the elements between [i,j), and move down others appropriately.
  void Erase(unsigned int i, unsigned int j) {
    AssertLt(j, size());
    AssertGt(j,i);
    int bytes = (size() -j)* sizeof(value_type);
    memmove(mpMantissas+i, mpMantissas+i+j, bytes);
    mLength -= j-i;
    mUntrimmedLength -= j-i;
  }

  ///If new size larger than old, new positions are not initialized.
  void resize(int n) {
    Assert( n >= 0 && n <= Bits31 );
    if ( n > capacity( ) ) {
      reserve(n);
      mUntrimmedLength = n;
    }
    mLength = n;
  }

  ///Reserve additional memory.
  ///Set self-owned to true if we need to allocate.
  void reserve(int n) {
    Assert( n >= 0 && n <= Bits31 );
    if ( n > capacity( ) ) {
      value_type * new_data = new value_type[n];
      //Copy all data up to the untrimmed length!
      memcpy( new_data, mpMantissas, 
              mUntrimmedLength * sizeof(value_type) );
      if ( SelfOwned( ) )
        delete [ ] mpMantissas;
      mpMantissas = new_data;
      mCapacity_SelfOwnedBit = TopBit32 ^ (n - size());
    }
  }



  /// given an index, returns the corresponding flow value 
  double operator () ( const int index ) const
  {
    AssertLt( index, static_cast<int>(mLength) );
    return mMultiplier * static_cast<double>(mpMantissas[ index ]);
  }

  /// given an index, returns the corresponding flow value, even if
  /// outside trim region
  double GetDouble ( const int index ) const
  {
    AssertLt( index, static_cast<int>(mUntrimmedLength) );
    return mMultiplier * static_cast<double>(mpMantissas[ index ]);
  }

  double Multiplier() const
  {  return mMultiplier;  }

  void SetMultiplier(double m) {
    mMultiplier = m;
  }


  ///Set this flowvector to flows that match the basevector.
  /// Noise can be added by setting that parameter. A noise value of 1.0
  /// will distribute the flow value of a singlet uniformly over 
  /// [0.5,1.5).
  void SetFromBases(const FlowOrder & order, 
      const basevector & bases, double noise = 0.0);

  ///Set data from a vector of other than value_type.
  ///The data will be converted to value_type first.
  template<class T>
  void SetData( const double multiplier, const vec<T> & mantissas) {
    ForceAssertGe(*min_element(mantissas.begin(), mantissas.end()),
		  numeric_limits<value_type>::min());
    ForceAssertLe(*max_element(mantissas.begin(), mantissas.end()),
		  numeric_limits<value_type>::max());
    vec<value_type> temp(mantissas.size());
    copy(mantissas.begin(), mantissas.end(), temp.begin());
    SetData(multiplier, temp);
  }

  ///Set data from a vector of other than value_type.
  ///The data will be converted to value_type first, and the multiplier 
  /// will be adjusted so that the range of the data is close to the
  /// range of value_type (to minimize loss of precision);
  template<class T>
  void SetData( const vec<T> & mantissas) {
    const int S = mantissas.size();
    vec<value_type> temp(S);
    double factor = *max_element(mantissas.begin(), mantissas.end());
    factor = numeric_limits<value_type>::max() / factor;
    for (int i=0; i != S; ++i) {
      temp[i] = (value_type)(mantissas[i] * factor);
    }
    SetData(1/factor, temp);
  }
     

  void SetData( const double multiplier,
                const vec<value_type> & mantissas ) {
    DestroyDynamicData();
    mUntrimmedLength = mLength = mantissas.size();
    mpMantissas = new value_type [ mLength ];
    for (unsigned int i = 0; i < mLength; i++)
    {
      mpMantissas[i] = mantissas[i];
    }
    mMultiplier = multiplier;
    mCapacity_SelfOwnedBit = TopBit32 | size();    
  }

  ///Remove flowsToTrim flows from the end of this flow vector.
  ///Note that the data are not lost: if we ever need to untrim, we can 
  ///get them back by calling UntrimEnd(). This is supported by the
  /// member mUntrimmedLength.
  /// It is safe to call this method repeatedly, no information will
  /// be lost.
  void TrimEnd(unsigned int flowsToTrim) { 
    AssertLe(flowsToTrim, size());
    resize( size() - flowsToTrim); 
  }

  ///Shorten this flow vector so the size is now newend (calls TrimEnd).
  ///Note that the data are not lost: if we ever need to untrim, we can 
  ///get them back by calling UntrimEnd(). This is supported by the
  /// member mUntrimmedLength.
  /// It is safe to call this method repeatedly, no information will
  /// be lost.
  void SetNewEnd(unsigned int newend) {
    AssertLe(newend, size());
    resize(newend);
  }

  ///Untrim the end to reveal all the flows originally recorded.
  void UntrimEnd() { 
    AssertGe(capacity(), longlong(mUntrimmedLength) );
    resize(mUntrimmedLength);
  }


  friend bool operator == ( const FlowVector & x, const FlowVector & y )
  {
    if ( x.mLength != y.mLength )
      return false;
    if ( x.mUntrimmedLength != y.mUntrimmedLength )
      return false;
    if ( x.mMultiplier != y.mMultiplier )
      return false;
    if ( x.capacity() != y.capacity() )
      return false;
    int complength = x.mUntrimmedLength * sizeof(value_type);
    if ( memcmp( x.mpMantissas, y.mpMantissas, complength) != 0 )
      return false;
    return true;
  }


  friend bool operator != ( const FlowVector & x, const FlowVector & y )
  {  return ! ( x == y );  }


  void Print( ostream & outstrm ) const;

  void Print( ostream & outstrm, int id ) const
  {
    outstrm << "Flow " << id << endl;
    Print(outstrm);
  }

  void Print( ostream & outstrm, String name ) const
  {
    outstrm << "Flow " << name << endl;
    Print(outstrm);
  }

  ///Print the contents as doubles separated by  separator
  void PrintDoubles( ostream & out, const String & separator = " ") const {
    for (int i = 0; i != isize(); ++i) {
      out << (*this)(i) << separator;
    }
    out << endl;
  }

  ///Make a string with gnuplot commands for plotting data.
  ///made static so it can be accessed by other classes if needed, because
  ///it's pretty generic.
  static String SetUpGnuPlot();

  ///Bargraph this flowvector using gnuplot, with the title provided.
  ///Draw horizontal lines at the levels indicated in lines

  void Graph(const String & title = "", 
	     const vec<float> & lines = vec<float>()) const;


  //--------------------------------------------------

  // The following methods are provided for compatibility with
  // the implementation of mastervec template in Feudal.
  //

  const char * StartOfStaticData() const
  {  return reinterpret_cast<const char *>( & mMultiplier );  }

  const int   SizeOfStaticData() const {  
    return sizeof(mMultiplier) + sizeof(mLength) + sizeof(mUntrimmedLength);
  }

  const value_type * StartOfDynamicData() const
  {  return mpMantissas;  }

  longlong               SizeOfDynamicData() const
  {  return mUntrimmedLength;  }

  longlong               ExtraSpaceForDynamicData() const
  {  return ExtraSpace();  }


  Bool SelfOwned() const
  {  return SelfOwnedBit() != 0;  }


  void SetExternal ( const char     * start_of_static_data,
                     value_type * pointer_to_dynamic_data,
                     longlong         size_of_dynamic_data,
                     longlong         extra_space_for_dynamic_data )
  {
    DestroyDynamicData();
    mpMantissas = pointer_to_dynamic_data;

    const double * doubleptr
      = reinterpret_cast<const double *>(start_of_static_data);
    mMultiplier = *doubleptr;
    doubleptr++;

    const unsigned int * uintptr 
      = reinterpret_cast<const unsigned int *>(doubleptr);
    mLength = *uintptr;
    ++uintptr;
    mUntrimmedLength = *uintptr; 

    mCapacity_SelfOwnedBit = size_of_dynamic_data 
      + extra_space_for_dynamic_data;

    ForceAssert( ! SelfOwned() );
  }


  void ShiftStartOfDynamicData( value_type * ptr1,
                                value_type * ptr2 )
  {  mpMantissas += ptr1 - ptr2;  }


  void Reinitialize()
  {
    DestroyDynamicData();
    DestroyStaticData();
    mCapacity_SelfOwnedBit = TopBit32;
  }


  void Blank()
  {
    mpMantissas = 0;
    DestroyStaticData();
    mCapacity_SelfOwnedBit = TopBit32;
  }

  //--------------------------------------------------

  // The following accessors should probably be made private.

  unsigned int SelfOwnedBit() const
  {  return mCapacity_SelfOwnedBit & TopBit32;  }

  unsigned int ExtraSpace() const
  {  return capacity() - mUntrimmedLength;  }

  void DestroyStaticData()
  {
    mLength = 0;
    mUntrimmedLength = 0;
    mMultiplier = 0;
  }

  void DestroyDynamicData()
  {
    if ( SelfOwned() && mpMantissas != 0 ) {
      delete [] mpMantissas;
      mpMantissas = 0;
    } else {
      mpMantissas = 0;
    }
  }


  //--------------------------------------------------
 private:

  /// capacity and self-ownership bit. 
  /// Note that capacity is measured in units of sizeof(value_type),
  /// i.e. the same units as size().
  unsigned int     mCapacity_SelfOwnedBit;

  /// pointer to the array of mantissas of the flow values, 
  value_type * mpMantissas;

  /// the common exponent of the flow values, stored as a multiplier
  double           mMultiplier;

  /// the number of flow values
  unsigned int     mLength;

  ///The original, untrimmed, length of the flow vector.
  unsigned int mUntrimmedLength;

};



inline
ostream & operator << ( ostream & out, const FlowVector & flowvec )
{
  flowvec.Print(out);
  return out;
}



typedef mastervec< FlowVector, FlowVector::value_type > vecFlowVector;


#endif

