// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowVector.cc

#include "454/flowdata/FlowVector.h"
#include "FeudalTemplate.h" // for instantiations, at end
#include "assembly/FeudalDataManagerTemplate.h" // for instantiations, at end
#include "454/flowdata/FlowVectorOps.h"
#include "system/System.h"

FlowVector & 
FlowVector::operator = ( const FlowVector & flowvec )
{
  ///Protect against self-assignment.
  if (this == &flowvec) return *this;

  ForceAssert( flowvec.mpMantissas != 0 );

  if ( (longlong)flowvec.size() > capacity() || mpMantissas == 0 ) {
      DestroyDynamicData();
      mCapacity_SelfOwnedBit = TopBit32 | flowvec.capacity();
      mpMantissas = new value_type [flowvec.capacity()];
  }
  
  mLength = flowvec.mLength;
  mUntrimmedLength = flowvec.mUntrimmedLength;
  mMultiplier = flowvec.mMultiplier;

  memcpy( mpMantissas, flowvec.mpMantissas,
	  flowvec.mUntrimmedLength * sizeof(value_type));

  return *this;
}

void FlowVector::SetFromBases(const FlowOrder & order,
      const basevector & bases, double noise) {
  vec<float> flows=FlowsFromBases(order, bases, noise);
  SetData(flows);
  }


void
FlowVector::Print( ostream & outstrm ) const
{
  outstrm << "  multiplier " << mMultiplier;
  outstrm.setf(ios::right);
  for (unsigned int i = 0; i < mLength; i++)
    {
      //output in multiples of four so the order is easy to follow
      if ( i % 12 == 0 )
	outstrm << endl << "  ";
      outstrm << setw(5) << mpMantissas[i] << ' ';
    }
  outstrm << endl;
}

String FlowVector::SetUpGnuPlot() {
  return "set data style boxes\n"
    "set autoscale\n"
    "set yrange [0:]\n"
    "set xrange [0:]\n"
    "set ytics 10000\n"
    "set xtics 8\n";
}

void FlowVector::Graph(const String & title, const vec<float> & lines) const {
  String fname = "FlowVectorGraph." + title + ".temp"; 
  Ofstream(os, fname);
  os << SetUpGnuPlot() << "plot ";
  for (int i=0; i != lines.isize(); ++i) {
    os << lines[i] << ", ";
  }
  os << " \"-\" using 1:2\n";
  for (int i=0; i != isize(); ++i) {
    os << i << " " << this->operator[](i) << endl;
  }
  os << "e" << endl;
  os.close();
  String command = "gnuplot -title " + title + " -persist " + fname;
  //+ "; rm -f " + fname;
  if (0 != system(command.c_str())) {
    FatalErr("gnuplot failed on " << title);
  }
}


// Instantiations of template classes associated with FlowVector

#ifdef __DECCXX_VER
#pragma define_template mastervec<FlowVector, FlowVector::value_type>
#else
INSTANTIATE_MASTERVEC( FlowVector, FlowVector::value_type )
#endif

template class FeudalDataManager<vecFlowVector,FlowVector>;
