// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowRead.h

// The class FlowRead manages all the information associated with
// a single flow in an assembly:
// name of the flow, the flow order, and the flow vector.


#ifndef FLOWDATA_FLOW
#define FLOWDATA_FLOW

#include "454/flowdata/FlowReadToken.h"
#include "454/flowdata/FlowVector.h"
#include "454/flowdata/FlowOrder.h"


// The class FlowRead is largely similar to the class Read.
// Most of the methods in Flow have analogues in Read,
// and they would even have parallel implementations.

class FlowRead
{
public:
  // Create an empty FlowRead
  FlowRead()
  { }

  ///Note ugly const_cast, needed because of the Token constructor.
  ///It should not have any consequences because all methods of
  ///FlowRead are const.
  FlowRead( const FlowDataBroker * pFlowDataBroker, const int id )
    : mToken( const_cast<FlowDataBroker *>(pFlowDataBroker), id )
  { }

  FlowRead( const FlowReadToken & flowReadToken )
    : mToken( flowReadToken )
  { }


  bool operator == ( const FlowRead & other ) const
  {  return ( mToken == other.mToken );  }

  bool operator != ( const FlowRead & other ) const
  {  return ! ( *this == other );  }

  bool operator < ( const FlowRead & other ) const
  {  return ( mToken < other.mToken );  }

  bool operator > ( const FlowRead & other ) const
  {  return ( mToken > other.mToken );  }

  bool IsValid() const
  {  return mToken.IsValid();  }

  int GetId() const
  {  return mToken.GetId();  }

  ///Graph this flow read with its own name as title
  ///and draw horizontal lines across at the levels indicated by lines.
  void Graph(vec<float> & lines) const {
    Graph(GetName(), lines);
  }

  ///Graph this flow read with a given title.
  ///and draw horizontal lines across at the levels indicated by lines.
  void Graph(const String & title, vec<float> & lines) const {
    GetFlowVector().Graph(title, lines);
  }
  

  String             GetName() const;
  const FlowOrder  & GetFlowOrder() const;
  const FlowVector & GetFlowVector() const;
  int                GetLength() const;


  // In the long run...
  //
  // basevector Basecall( threshold-params ) const


private:
  FlowDataBroker * GetMgrPtr() const
  {  return mToken.GetMgrPtr();  }

  FlowReadToken mToken;

};

String ToString(const FlowRead &r,
		bool includeName=true,
		bool includeXpos=false,
		bool includeYpos=false,
		bool includeFloats=true,
		bool includeIntegers=false,
		bool trimmed=true);

#endif
