// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowOrder.h

// The class FlowOrder manages the information of the order of flows
// associated with a single (or a collection of) FlowVector(s).
//

#ifndef FLOWDATA_FLOWORDER
#define FLOWDATA_FLOWORDER

#include "Vec.h"
class String;

class FlowOrder
{
public:
  // Create an empty FlowOrder
  FlowOrder()
  { }

  // Create a FlowOrder from the given list of characters
  // representing the flow types within a single cycle
  // (note: only the cycle, not the entire sequence of flows)
  FlowOrder( const vec<char> & cycle_list );

  // Create a FlowOrder from the given list of characters representing
  // the flow types within a single cycle (note: only the cycle, not
  // the entire sequence of flows).  If the string has a newline,
  // assume that it represents serialized FlowOrder as produced by
  // asString(), and read p-flow information as well.
  FlowOrder( const String & cycle_list );

  // mark an index as a "P" flow
  void MarkAsP( const int index );

  // given an index, returns the character representing the flow type
  char operator [] ( const int index ) const;

  bool operator == ( const FlowOrder & other ) const
  {
    return ( ( mCycleString == other.mCycleString ) &&
	     ( mPlist == other.mPlist ) );
  }

  ///Get a vector with all the positions of p flows
  const vec<int> & PPositions() const { return mPlist; }
 
  ///Get number of p flows
  int NumPPositions() const { return mPlist.size(); }

  ///Set positions of P flows to this vector.
  void SetPPositions (const vec<int> & newpos) {
    mPlist = newpos;
  }

  ///return the number of flows between two bases passed in as letters.
  int Offset(char start, char end) const {
    int spos = -1, epos = -1;
    const int S = mCycleString.size();
    for (int i=0; i != S; ++i) {
      if (tolower(mCycleString[i]) == tolower(start)) spos = i;
      if (tolower(mCycleString[i]) == tolower(end)) epos = i;
    }
    if (-1 == spos || -1 == epos) return -1;
    else return (epos + S - spos) % S;
  }

  ///Get cycle string
  String CycleString() const { return mCycleString; }
  
  // Returns the string output by <<.  This contains all the
  // information stored in the floworder and can be used as input to
  // the String constructor or provided to fromString().
  String asString() const;
  
  // Load object from a string, as for >>.
  void fromString(const String &in);
  
  // support I/O operator methods which are compatible with
  // the requirements of FeudalPlusDataManager
  friend
  istream & operator >> ( istream & instrm, FlowOrder & floworder ); 

private:
  // the string of standard flows in a cycle; e.g. "TACG"
  String   mCycleString;

  // the *sorted* list of indices of exceptional "P" flows
  vec<int> mPlist;
  
};

// The output operator, compatible with FeudalPlusDataManager 
ostream & operator << ( ostream & outstrm, const FlowOrder & floworder );

// The input operator, compatible with FeudalPlusDataManager 
istream & operator >> ( istream & instrm, FlowOrder & floworder );






#endif

