// Copyright (c) 2005 Broad Institute of MIT and Harvard
//

#ifndef FLOW_KEY_H
#define FLOW_KEY_H

/// This class contains and gives access to a flow key.
///
/// \class FlowKey
///
/// It can be constructed directly from a flow or from a String.
///
/// the static method GetKey does a keypass attempt on a flow.
///
/// Assignment and copy are OK, since all members are either copyable
/// objects or const references, and are probably small.

class FlowOrder;
class FlowVector;
class basevector;

#include "String.h"
#include "Vec.h"

#include <iosfwd>

class FlowKey {
  friend class TestFlowUtils;

 public:
  static const unsigned int FLOW_KEY_SIZE;
  static unsigned int MaxSizeInFlows() {
    return  1 + 4 * (FLOW_KEY_SIZE-1);
  }
  static const float BAD_FLOAT;

  ///Create an empty (Bad() == true) flow key
  FlowKey();

  ///Create a key from a string.
  FlowKey( const FlowOrder &, const String &);

  ///Create a key from a flow.
  ///Note that if the flow is ugly, the key will be Bad().
  FlowKey( const FlowOrder &, const FlowVector &);

  ///It is possible to construct a bad key from a noisy FlowVector.
  ///We don't want to throw an exception in that case, so we just allow
  ///the user to check for bad keys.
  bool Bad() const { return mString.size() != FLOW_KEY_SIZE; }

  /// Number of positive flows in the key.
  int size() const { return mString.size(); }

  /// Total number of flows in the key.
  int SizeInFlows() const { return Nsize() + Psize(); }

  ///mean height of 1-peaks in key.
  double PosMean(const FlowVector &) const;

  ///mean height of 0-peaks in key.
  double NegMean(const FlowVector &) const;

  String ToString() const { return mString; }

  ///Create a string key from a flow, returns empty string if no key found.
  ///This is pretty lenient in terms of noise in the flow, but does require
  ///a clear separation between singlets and 0-flows.
  static String GetKey(const FlowOrder & order, const FlowVector & v);

  void TrimFrom (basevector & bases);

 private:
  String mString; ///< String representation of the key.
  vec<int> mPos;
  vec<int> mNeg;


  //============= These methods could possible be made public?=========//
  ///Size of the negative indices array
  int Nsize() const { return mNeg.size(); }

  ///Size of the negative indices array
  int Psize() const { return mPos.size(); }

  ///Position in the flow vector of the ith negative index.
  int Neg(int i) const { return mNeg[i]; }

  ///Position in the flow vector of the ith positive index.
  int Pos(int i) const { return mPos[i]; }


  ///Set the vectors from the string key.
  ///If the key is of the wrong length, make sure Bad() returns true and
  /// that Neg() and Pos() will fail.
  void SetInternals( const FlowOrder &, const String & key );

  ///Set this key as bad.
  void SetBad();
};

ostream & operator<<(ostream &, const FlowKey &);

inline bool operator<(const FlowKey & rhs, const FlowKey & lhs) {
  return rhs.ToString() < lhs.ToString();
}

inline bool operator==(const FlowKey & rhs, const FlowKey & lhs) {
  return rhs.ToString() == lhs.ToString();
}

inline bool operator!=(const FlowKey & rhs, const FlowKey & lhs) {
  return !(rhs==lhs);
}

/// Hacks for using FlowInfo
/////////////////////////////////////////////////////////////////////////////
///Create a key string from a float.
String KeyStringFromFloat(float k);

///Create a float from a key string.
float FloatFromKeyString(const String &s);

///Create a float from a key.
float AsFloat(const FlowKey & key);

///Create a FlowKey from a float, using the same FlowOrder for each key.
struct FloatToFlowKey : unary_function<float, FlowKey> {
  FlowOrder &ord_;
  FloatToFlowKey(FlowOrder &ord) : ord_(ord) { }
  FlowKey operator()(float x) { return FlowKey(ord_, KeyStringFromFloat(x));  }
};

#endif //FLOW_KEY_H
