#include "454/flowdata/FlowKey.h"
#include "454/flowdata/FlowOrder.h"
#include "454/flowdata/FlowVector.h"
#include "Basevector.h"

#include <iostream>

const unsigned int FlowKey::FLOW_KEY_SIZE=4;
const float FlowKey::BAD_FLOAT = -1.0;

FlowKey::FlowKey():
  mString(),
  mPos(),
  mNeg() {}

FlowKey::FlowKey( const FlowOrder & order, const String & key):
  mString(),
  mPos(),
  mNeg()
{
  SetInternals(order, key);
}

FlowKey::FlowKey( const FlowOrder & order, const FlowVector & fvec) {
  String key = GetKey(order, fvec);
  SetInternals(order, key);
}

 ///Create a key from a float. This is a hack for using FlowInfo
String KeyStringFromFloat(float k) {
  if (k == FlowKey::BAD_FLOAT) return "";

  int intKey(static_cast<int>(k));
  String strKey;
  strKey.resize(FlowKey::FLOW_KEY_SIZE);
  for (unsigned int i=0; i != FlowKey::FLOW_KEY_SIZE; ++i) {
    strKey[i] = as_base(intKey & 0x3);
    intKey = intKey >> 2;
  }
  return strKey;
}

float FloatFromKeyString(const String &s)
{
  if (s.empty()) return FlowKey::BAD_FLOAT;
  int intKey = 0;
  for (int i = s.size(); i != 0; --i) {
    intKey = intKey << 2;
    intKey |= as_char(s[i-1]);
  }
  return static_cast<float>(intKey);
}

float AsFloat(const FlowKey & key) {
  return FloatFromKeyString(key.ToString());
}

ostream & operator<<(ostream & os, const FlowKey & key) {
  os << key.ToString();
  return os;
}

///mean height of 0-peaks in key for this flow vector.
double FlowKey::NegMean(const FlowVector & fvec) const {
  AssertGt(Nsize(),0);
  double mean=0;
  for (int i = 0; i != Nsize(); ++i) {
    mean += fvec[Neg(i)];
  }
  mean /= Nsize();
  return mean;
}

///mean height of 0-peaks in key for this flow vector.
double FlowKey::PosMean(const FlowVector & fvec) const {
  AssertGt(Psize(),1);
  double mean=0;
  for (int i = 0; i != Psize()-1; ++i) {
    mean += fvec[Pos(i)];
  }
  mean /= Psize()-1;
  return mean;
}


void FlowKey::SetInternals(const FlowOrder & order, const String & key) {
  if (key.size() != FLOW_KEY_SIZE) { //make sure Bad() returns true.
    SetBad();
    return;
  }
  mString = key;
  int fpos = 0;
  for (unsigned int i = 0; i != FLOW_KEY_SIZE; ++i) {
    //mark 0 flows as such and move forward
    while (order[fpos] != key[i]) {
      if ( order[fpos] != 'P' ) {
        mNeg.push_back(fpos);
      }
      ++fpos;
    }
    //mark 1 flows.
    mPos.push_back(fpos);
    ++fpos;
  }
  sort(mPos.begin(), mPos.end());//needed so that we can ignore last position.
  sort(mNeg.begin(), mNeg.end()); //just for tidyness and symmetry.
  if (mPos.empty() || mNeg.empty()) {
    SetBad();
  }
}

void FlowKey::TrimFrom(basevector & bases) {
  if (bases.size() < static_cast<unsigned int>(size())) {
    bases.resize(0);
    return;
  }
  for (unsigned int i = 0; i != mString.size(); ++i) {
    if (bases[i] != as_char(mString[i])) {
      bases.resize(0);
      return;
    }
  }
  basevector temp;
  temp.SetToSubOf(bases, size(), bases.size() - size());
  bases.Swap(temp);
}
  

void FlowKey::SetBad() {
  mString.resize(0);
  mPos.clear();
  mNeg.clear();
}

String FlowKey::GetKey(const FlowOrder & order, const FlowVector & v) {
  String key;
  int fpos=-1;
  bool failed = false;
  const int CYCLE=4;

  // return a bad key if the flow vector is too short.
  if (v.size() < MaxSizeInFlows()) return String();
  

  //Get one cycle of flows, and deduce the height of 1 and 0 peaks.
  vec<FlowVector::value_type> cycle;
  while (cycle.isize() != CYCLE) {
    if (order[++fpos] =='P') continue;
    cycle.push_back(v[fpos]);
  }
  sort(cycle.begin(), cycle.end());
  if (cycle.back() < cycle.front() * 2) {
    return String(); //fail, no key can be found.
  }
  int low = (cycle.back() + cycle.front()) /2;
  int high = 2 * cycle.back() - low;

  //Now deduce the key
  fpos = -1;
  int lastHigh=v.size(); //used to detect three 0 flows in a row.
  while (key.size() != FLOW_KEY_SIZE) {
    if (order[++fpos] == 'P') continue;
    if (fpos == v.isize())  {
      return String(); //fail, no key can be found. 
    }
    if ( v[fpos] < low ) { //do nothing, control for three 0 flows in a row.
      if (fpos >= lastHigh + 3) return String();//fail
    }
    else if (v[fpos] > high  && key.size() != FLOW_KEY_SIZE-1) {
      return String(); //fail
    } 
    else {
      //we are within 1-peak boundaries or it's the last base in the key.
      key += order[fpos];
      lastHigh = fpos;
    }
  }
  return key;
}
