// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FlowDataBroker.h

// The class FlowDataBroker defines the corresponding manger object
// which can be used to manage a collection FlowRead objects.


#ifndef FLOWDATA_BROKER
#define FLOWDATA_BROKER

#include "454/flowdata/FlowRead.h"


class IdManager;
class NameManager;
class FlowVectorManager;
class PrefetchStrategy;
class FlowKey;



/// FlowDataBroker has characteristics similar to ReadDataManager and Assembly.
///
/// \class FlowDataBroker
/// Most of the methods in FlowDataBroker have analogues in ReadDataManager,
/// and they would even have parallel implementations.
///
/// Perhaps a better way of doing things is to have a common base class of
/// ObjDataManager from which both ReadDataManager and FlowDataBroker are 
/// derived.
///
/// This idea is left open for future exploration.


class FlowDataBroker
{
public:
  FlowDataBroker();

  /// Create a broker and point it at the given files
  FlowDataBroker( const String & strFlowbFile,
		  const String & strFlowNamesFile,
		  bool removePflows = true);

  FlowDataBroker( NameManager       * pFNM,
		   FlowVectorManager * pFVM,
		   IdManager * pIM );

  ~FlowDataBroker();


  //======= ENTITY CREATION ROUTINES =========

  FlowRead NewFlowRead( const String     & name,
			// const FlowOrder  & floworder,
			const FlowVector & flowvec );

  FlowRead  AddFlowRead( const FlowRead & theFlowRead );

  /// Replace an existing read with a new one.
  /// It must have the same name.
  void ReplaceFlowVector(const int id, const FlowVector & fvec);     

  //============= ACCESSOR METHODS ===============

  /// Number of flowreads in broker
  int size() const;

  int GetNumFlowReads() const { return size(); }

  FlowRead GetFlowRead ( const int id ) const;
  FlowRead GetFlowRead ( const String & name ) const;
  void GetAllFlowReads( vec<FlowRead> & vecFlowReads ) const;
  
  ///Return true if this flowread is usable.
  /// For now, that means that it must have enough
  /// valid flows to have any known four-letter
  /// key, i.e. 13 flows.
  bool Good(const int id) const;

  ///Return true if this flowread is usable with this key.
  /// For now, that means that it must have enough
  /// valid flows to have this particular key.
  /// The Broker knows what the flow order is, so it
  /// can take a key in as a string.
  bool Good(const int id, const String & keystr) const;

  ///Return true if this flowread is usable with this key.
  /// For now, that means that it must have enough
  /// valid flows to have this particular key.
  bool Good(const int id, const FlowKey & key) const;

  ///Name of flow read at index id
  String             GetName      ( const int id ) const;
  ///index of flowread with given name.  Returns -1 on failure.
  int GetIndex ( const String & name ) const;

  const FlowOrder  & GetFlowOrder ( const int id ) const;
  const FlowVector & GetFlowVector( const int id ) const;

  ///Length of vector for flow read at index id
  int                GetLength    ( const int id ) const;


  ///Remove all P flows from all structures managed by this.
  void RemovePFlows();

  void SetFlowOrder( const FlowOrder & floworder );

  const FlowOrder & GetFlowOrder() const;


  //============ I/O METHODS ===========================

  /// The caller of this method is still responsible for
  /// the management of this pointer.
  void SetPrefetchStrategy( PrefetchStrategy * pStrategy );

  /// Read in the data from the filesystem.
  /// Will destroy current contents of the broker.
  void ReadData( const String & strFlowbFile,
		 const String & strFlowNamesFile );

  /// Read in the data from the filesystem, add usual extensions to prefix.
  /// Will destroy current contents of the broker.
  void ReadData( const String & prefix ) {
    ReadData(prefix+".flowb", prefix + ".flownames");
  }

  ///Write out all data to flowb flowb_plus and flownames files.
  void WriteData( const String & strFlowbFile,
		  const String & strFlowNamesFile );

  /// Write out all data, adds the usual extensions to the prefix.
  void WriteData( const String & prefix ) {
    WriteData(prefix+".flowb", prefix + ".flownames");
  }

  /// Write all data as text, one line per flowread.  By default
  /// include numeric name but not x and y.  Space separated, includes
  /// P flows, so call RemovePFlows first if you do not want them.
  void WriteTextData( const String & filename,
		      bool includeName=true,
		      bool includeXpos=false,
		      bool includeYpos=false);

private:
  // To prevent the unintentional copying of brokers,
  // the assignment operator is private and undefined.
  FlowDataBroker & operator = ( const FlowDataBroker & );
  // To prevent the unintentional copying of brokers,
  // the copy constructor is private and undefined.
  FlowDataBroker( const FlowDataBroker & );

  IdManager * mpIdMgr;
  NameManager * mpFlowNameMgr;
  FlowVectorManager * mpFlowVectorMgr;
  //
  // NOTE: For now, there is only one FlowVectorManager handle maintained by
  // the FlowDataBroker, which means that only one flow order for
  // the flow vectors is supported, i.e. all the flow vectors are assumed
  // to have the same flow order.
  //
  // One could envision a future situation in which flow vectors of
  // different flow orders are simultaneously present and hence need to be
  // simultaneously managed by the FlowDataBroker.  In that case,
  // FlowDataBroker will have to maintain handles to several FlowVectorManagers,
  // each of which manages the flow vectors of a common flow order.
  // An additional problem will be to translate the integer id of a flow
  // into the corresponding FlowVectorManager to query.
  // All these are left for future development.

  bool CheckId ( const int id ) const;

};


#endif //FLOWDATA_BROKER
