/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

//
// FlowDataBroker.cc


#include "454/flowdata/FlowDataBroker.h"

#include "assembly/IdManager.h"
#include "assembly/NameManager.h"
#include "454/flowdata/FlowVectorManager.h"
#include "454/flowdata/FlowKey.h"

FlowDataBroker::FlowDataBroker()
  : mpIdMgr( new IdManager ),
    mpFlowNameMgr( new NameManager ),
    mpFlowVectorMgr( new FlowVectorManager )    
{
  ForceAssert( mpIdMgr );
  ForceAssert( mpFlowNameMgr );
  ForceAssert( mpFlowVectorMgr );
}



FlowDataBroker::FlowDataBroker( NameManager       * pFNM,
				  FlowVectorManager * pFVM,
				  IdManager         * pIM )
  : mpIdMgr( pIM ),
    mpFlowNameMgr( pFNM ),
    mpFlowVectorMgr( pFVM )
{
  ForceAssert( mpIdMgr );
  ForceAssert( mpFlowNameMgr );
  ForceAssert( mpFlowVectorMgr );
}

FlowDataBroker::FlowDataBroker( const String & strFlowbFile,
				const String & strFlowNamesFile,
				bool removePflows): 
  mpIdMgr( 0),
  mpFlowNameMgr( 0),
  mpFlowVectorMgr( 0)    
{
  //the managers are created inside ReadData.
  this->ReadData( strFlowbFile, strFlowNamesFile );
  if (removePflows) {
    this->RemovePFlows();
  }
  // We no longer save flowb's that include P flows.  This assert
  // ensures we don't mistakenly read old flowbs that had them.  
  ForceAssert( GetFlowOrder().NumPPositions()==0 );
}


FlowDataBroker::~FlowDataBroker()
{
  delete mpIdMgr;
  delete mpFlowNameMgr;
  delete mpFlowVectorMgr;
}


int
FlowDataBroker::size() const
{
  return mpIdMgr->GetMaxId() + 1;
}



FlowRead
FlowDataBroker::NewFlowRead( const String     & name,
			     const FlowVector & flowvec )
{
  ForceAssert( ! name.empty() );

  FlowRead existingFlowRead = this->GetFlowRead( name );
  if ( existingFlowRead.IsValid() ) {
    if ( existingFlowRead.GetFlowVector() != flowvec ) {
      cerr << "Attempted to add flowread with same name but different data." 
	   << endl;
      PRINT2_TO( cerr,existingFlowRead.GetFlowVector().size(), flowvec.size());
      ForceAssert( 0 == 1 );
      return FlowRead();
    }
    else
      return existingFlowRead;
  }
  else {
    const int id = mpIdMgr->GetNewId();

    mpFlowNameMgr->SetName( id, name );
    mpFlowVectorMgr->SetFlowVector( id, flowvec );

    mpFlowNameMgr->Verify( id );
    mpFlowVectorMgr->Verify( id );

    return GetFlowRead( id );
  }
}

FlowRead
FlowDataBroker::AddFlowRead( const FlowRead & theFlowRead )
{
  FlowRead existingFlowRead = this->GetFlowRead( theFlowRead.GetName() );

  if ( existingFlowRead.IsValid() ) {
    ForceAssert( theFlowRead.GetFlowOrder() == existingFlowRead.GetFlowOrder());
    ForceAssert( theFlowRead.GetFlowVector() 
		 == existingFlowRead.GetFlowVector() );      
    return existingFlowRead;
  }

  FlowRead newFlowRead = this->NewFlowRead( theFlowRead.GetName(),
					    // theFlowRead.GetFlowOrder(),
					    theFlowRead.GetFlowVector() );

  return newFlowRead;
}


void FlowDataBroker::ReplaceFlowVector( const int id,
				       const FlowVector & fvec )
{
  mpFlowNameMgr->Verify( id );
  mpFlowVectorMgr->Verify( id );
  mpFlowVectorMgr->SetFlowVector( id, fvec);
}



bool
FlowDataBroker::CheckId( const int id ) const
{
  ForceAssertGe( id, 0 );
  ForceAssertLe( id, mpIdMgr->GetMaxId() );
  return true;
}



FlowRead
FlowDataBroker::GetFlowRead( const int id ) const
{
  CheckId( id );
  return FlowRead( this, id );
}


FlowRead
FlowDataBroker::GetFlowRead( const String & name ) const 
{
  int id = mpFlowNameMgr->FindId( name );

  if ( id == -1 )
    return FlowRead();

  return this->GetFlowRead( id );

}

void
FlowDataBroker::GetAllFlowReads( vec<FlowRead> & vecFlowReads ) const 
{
  vecFlowReads.clear();
  vecFlowReads.resize( this->size() );

  for (unsigned int flowreadId = 0;
       flowreadId < vecFlowReads.size();
       flowreadId++) {
    vecFlowReads[ flowreadId ] = GetFlowRead( flowreadId );
  }
}

bool FlowDataBroker::Good (const int id) const {
  CheckId(id);
  return longlong(mpFlowVectorMgr->GetFlowVector( id ).size()) > 
  	FlowKey::MaxSizeInFlows();
}

bool FlowDataBroker::Good (const int id, const FlowKey & key) const {
  CheckId(id);
  return longlong(mpFlowVectorMgr->GetFlowVector( id ).size()) > 
  	key.SizeInFlows();
}

bool FlowDataBroker::Good (const int id, const String & keystr) const {
  FlowKey key(GetFlowOrder(), keystr);
  return Good(id, key);
}

String
FlowDataBroker::GetName( const int id ) const
{
  CheckId( id );
  return mpFlowNameMgr->GetName( id );
}

int
FlowDataBroker::GetIndex ( const String & name ) const
{
  return mpFlowNameMgr->FindId( name );
}

const FlowOrder &
FlowDataBroker::GetFlowOrder( const int id ) const
{
  return mpFlowVectorMgr->GetFlowOrder();
}


const FlowVector &
FlowDataBroker::GetFlowVector( const int id ) const
{
  CheckId( id );
  return mpFlowVectorMgr->GetFlowVector( id );
}


int
FlowDataBroker::GetLength( const int id ) const
{
  CheckId( id );
  return mpFlowVectorMgr->GetLength( id );
}


void
FlowDataBroker::SetFlowOrder( const FlowOrder & floworder )
{
  mpFlowVectorMgr->SetFlowOrder(floworder);
}

const FlowOrder & FlowDataBroker::GetFlowOrder( ) const {
  return mpFlowVectorMgr->GetFlowOrder();
}


void
FlowDataBroker::RemovePFlows( )
{
  mpFlowVectorMgr->RemovePFlows( );
}


void
FlowDataBroker::WriteData( const String & strFlowbFile,
			const String & strFlowNamesFile )
{
  bool bOverwrite = true;
  mpFlowNameMgr->Write( bOverwrite,
			strFlowNamesFile );

  mpFlowVectorMgr->Write( bOverwrite,
			  strFlowbFile );

}

void
FlowDataBroker::ReadData( const String & strFlowbFile,
			  const String & strFlowNamesFile )
{
  delete mpIdMgr;
  delete mpFlowNameMgr;
  delete mpFlowVectorMgr;

  mpIdMgr = new IdManager;
  mpFlowNameMgr = new NameManager( strFlowNamesFile );
  mpFlowVectorMgr = new FlowVectorManager( strFlowbFile );

  mpIdMgr->UseId( MastervecFileObjectCount(strFlowbFile) - 1 );
  PrefetchAllStrategy prefetchAll;
  SetPrefetchStrategy (&prefetchAll);
}

void
FlowDataBroker::SetPrefetchStrategy( PrefetchStrategy * pStrategy )
{
  mpFlowNameMgr->SetPrefetchStrategy( pStrategy );
  mpFlowVectorMgr->SetPrefetchStrategy( pStrategy );
}

void FlowDataBroker::WriteTextData( const String & filename, bool includeName,
				    bool includeXpos, bool includeYpos) {
  Ofstream(os, filename);
  PrefetchAllStrategy ps;
  SetPrefetchStrategy( &ps );
  const int N = size();
  String name;
  for (int i=0; i != N; ++i) 
    os << ToString(GetFlowRead(i), includeName, includeXpos, includeYpos)
       << endl;
}
