// Copyright (c) 2005 Broad Institute of MIT and Harvard
//
// FeudalPlusDataManager.h

// The class FeudalPlusDataManager defines the corresponding manger object
// which can be used to manage (1) a collection of Feudal objects, together
// with (2) an extra bit of information (whose type is  PlusT) common to
// all of the Feudal objects.
//
// For file I/O purposes, it is required that the PlusT class supports
// the implementation of the following operator methods:
//
//   ostream & operator << ( ostream & outstrm, const PlusT & plusobj );
//
//   istream & operator >> ( istream & instrm, PlusT & plusobj );
//
// These methods should moreover be inverses of each other, so that
// the file I/O is lossless.
//



#ifndef FLOWDATA_FEUDALPLUSDATAMANAGER
#define FLOWDATA_FEUDALPLUSDATAMANAGER


#include "454/flowdata/FlowVector.h"
#include "454/flowdata/FlowOrder.h"
#include "assembly/FeudalDataManager.h"


template <class MasterVecT, class SerfVecT, class PlusT>
class FeudalPlusDataManager
{
public:

  FeudalPlusDataManager()
    : mbPlusInitialized( false ),
      mFeudalDataMgr()
  { }

  FeudalPlusDataManager( const String & strFeudalFile )
    : mFeudalDataMgr( strFeudalFile )
  {
    const String strFeudalPlusFile = strFeudalFile + "_plus";
    Ifstream( plusdata_strm, strFeudalPlusFile );
    plusdata_strm >> mPlusData;
    plusdata_strm.close();
    mbPlusInitialized = true;
  }


  void LoadData( const vec<int> & vecIds );
  void LoadData( const int id );
  void LoadAllData();



  const SerfVecT & GetData( const int id )
  {
    return mFeudalDataMgr.GetData( id );
  }


  const PlusT & GetPlusData() const
  {
    ForceAssert( mbPlusInitialized );
    return mPlusData;
  }


  void SetData( const int id, const SerfVecT & data )
  {
    mFeudalDataMgr.SetData( id, data );
  }


  void SetPlusData( const PlusT & plusdata )
  {
    mPlusData = plusdata;
    mbPlusInitialized = true;
  }


  void Write( const bool bOverwrite,
	      const String & strFeudalFile,
	      const int lastId = - 1 );


  Bool IsLoaded( const int id )
  {
    return mFeudalDataMgr.IsLoaded( id );
  }
  

  int Size()
  {
    return mFeudalDataMgr.Size();
  }


private:
  bool  mbPlusInitialized;
  PlusT mPlusData;

  FeudalDataManager<MasterVecT,SerfVecT>  mFeudalDataMgr;

};


typedef FeudalPlusDataManager<vecFlowVector,FlowVector,FlowOrder> FlowValuesManager;

#endif
