// Copyright (c) 2005 Broad Institute of MIT and Harvard
//

#include "454/flowdata/AnalysisInfo.h"
#include "454/flowdata/FlowKey.h"
#include "Basevector.h"
#include "FastaFileset.h"
#include "Map.h"
#include <fstream>
#include <sstream>

// Very tolerant parser for analysis information files.
AnalysisInfo::AnalysisInfo( const String &filename)
{
  Ifstream (in, filename.c_str() );
  ReadFrom(in);
}

AnalysisInfo::AnalysisInfo( int dummy, const String & text)
{
  istringstream in( text.c_str() );
  ReadFrom(in);
}

void AnalysisInfo::ReadFrom( istream & in) {
  String line, name, value, refdir;
  while(1) {
    getline( in, line );
    if ( !in ) break;
    if ( line.Contains( "#", 0 ) ) continue;
    if ( line.size( ) == 0 ) continue;
    int pos = line.Position("=");
    if (pos<0) continue;
    name = line.substr(0, pos);
    value = line.substr(pos+1, line.size()-pos);
    DeleteLeadingWhiteSpace(name);
    DeleteTrailingWhiteSpace(name);
    DeleteLeadingWhiteSpace(value);
    DeleteTrailingWhiteSpace(value);
    if (name=="REFDIR")
      refdir = value;
    else if (name=="WELLSFILE")
      wellsfile_ = value;
    else if (name=="REGION")
      region_ = value;
    else if (name=="BARCODE")
      barcode_ = value;
    else if (name=="NUMREGIONS")
      numRegions_ = value;
    else if (name=="PLATESIZE")
      plateSize_ = value;
    else if (name=="KEYREF")
      update(exp_, value);
    else if (name=="TESTKEYREF")
      update(control_, value);
    else if (name=="RUNNAME")
      runname_ = value;
  }
  addRefdir(exp_, refdir);
  addRefdir(control_, refdir);
}

void AnalysisInfo::update(Map &M, String value)
{
  if (!value.Contains(","))
    return;
  String key = value.Before(","), ref = value.After(",");
  DeleteLeadingWhiteSpace(key);
  DeleteTrailingWhiteSpace(key);
  DeleteLeadingWhiteSpace(ref);
  DeleteTrailingWhiteSpace(ref);
  M[FloatFromKeyString(key)]=ref;
}

void AnalysisInfo::addRefdir(Map &M, String refdir) {
  for (Map::iterator it=M.begin(); it!=M.end(); ++it) {
    if (!it->second.empty()) {
      it->second = refdir + it->second;
    }
  }
}

ulonglong AnalysisInfo::wellsAvailable() const
{
  String plate = plateSize_;
  plate.ToLower();
  if (plate=="14x43") {
    if (numRegions_=="1")
      return 255000;
    else if (numRegions_=="4")
      return 50000;
  } else if (plate=="30x60") {
    if (numRegions_=="1")
      return 781500;
    else if (numRegions_=="8")
      return 50000;
  } else if (plate=="60x60" && numRegions_=="2") {
    return 755000;
  }
  return 0;
}

vec<String> AnalysisInfo::expKeys() const
{
  vec<String> result;
  for (Map::const_iterator it=exp_.begin(); it!=exp_.end(); ++it)
    result.push_back(KeyStringFromFloat(it->first));
  return result;
}

vec<String> AnalysisInfo::controlKeys() const
{
  vec<String> result;
  for (Map::const_iterator it=control_.begin(); it!=control_.end(); ++it)
    result.push_back(KeyStringFromFloat(it->first));
  return result;
}

String AnalysisInfo::GetReference(const String & key) const{
  return GetReference(FloatFromKeyString(key));
}

 
String AnalysisInfo::GetReference(float key) const {
  String fname;
  Map::const_iterator i;
  if ((i = exp_.find(key)) != exp_.end()) {
    fname = i->second;
  }
  else if ((i = control_.find(key)) != control_.end()) {
    fname = i->second;
  }
  else {
    cout << "exp ";
    cout << exp_;
    cout << "\ncontrol "; cout  << control_ << endl;
    FatalErr("No key " << key << " in the analinfo file " 
	     << barcode() << "." << region());
  }
  return fname;
}

