// Copyright (c) 2005 Broad Institute of MIT and Harvard
//

#include "CoreTools.h"
#include "String.h"
#include "STLExtensions.h"
#include "math/Functions.h"
#include "454/EvaluateFlow.h"

#include <numeric>
#include <cmath>

//define this to see what is classified where
//#define VERBOSE_EVALUATOR


const int EvaluateFlow::BAD_KEY=-2;
const int EvaluateFlow::BAD_MEANS=-1;
const int EvaluateFlow::MIN_DATA = 20;
const double EvaluateFlow::MEANS_CHANGE_THRESHOLD=0.1;



float EvaluateFlow::OverlapScore(const vec<normal_distribution> & dists) {
  AssertEq(dists.size(), 2);
  if (0 == dists[1].mu_) {
    return 0;
  } 
  else { //success, 
    return (dists[1].mu_ - dists[0].mu_ - dists[1].sigma_ - dists[0].sigma_) 
      / dists[1].mu_;
    //return the "distance" between the noise +stdv and the 1peak-stdev
    //normalized to the height of the 1-peak.
    //This could be improved to return a probability of non-overlap, but
    //for now this is monotonically related to that probability.
  }
}


/*step 1: take means and stdevs and place points in clusters
  interface: pass in the flow. Each cluster is a vec<int> of indices within
  the flow, so that the clusters are really a vec<vec<int>.
  CalcDistributions(const FlowRead & flow, vec<normal_distribution > meanstdevs,
  const vec<vec<int> > & clusters.
  step 2: recalculate clusters based on means and stdevs.
  RecomputeClusters(const FlowRead & flow, const vec<pair<double, double > & meanstdevs, vec<vec<int> > & clusters);
  
*/

double MahalanobisDistance(double point, double mean, double stdev) {
  //shortcut, really is sqrt(sum of distances, but since it's 1-D...
  return abs(point - mean)/stdev;
}

void GetDistances(double point, const vec<normal_distribution> & distributions, 
                  vec<double> & distances) {
  const int S = distributions.size();
  distances.resize(S);
  for (int i = 0; i != S; ++i) {
    distances[i] = MahalanobisDistance(point, distributions[i].mu_,
                                       distributions[i].sigma_);
  }
}

void EvaluateFlow::ComputeCutoffs( const vec<normal_distribution> & dist,
				   vec<double> & cutoffs) {
  const int S = dist.size();
  AssertGt(S,1);

  //Make sure all clusters are empty:
  cutoffs.resize(S);

  for (int i=0; i != S-1; ++i) {
    cutoffs[i] = OptimalCutoff(dist[i], dist[i+1]);
  }
  cutoffs[S-1] = dist[S-1].mu_ + (dist[S-1].mu_ - cutoffs[S-2]);
}

