// Copyright (c) 2005 Broad Institute of MIT and Harvard
//

/// This class generates a basevector from a flow.
///
/// \class BaseCaller
///
/// It uses the values from the key, does a few passes of 
/// braindead EM-like adjustment of the means, and happily calls away.
///    
/// The output includes the key, which can be cut off later if needed.
///
/// Assignment and copy are OK, since all members are either copyable
/// objects or const references.


class basevector;
class FlowVector;
class FlowOrder;
class FlowKey;
class normal_distribution;
class QuadraticFunction;
template<class T> class vec;

#include <functional>

class BaseCaller {

 public:
  static const int MAX_PASSES=10;///< number of passes of means adjustment.
  static const int MAX_BASES=700;///< max bases in a read.

  ///Constructor.
  BaseCaller(const FlowOrder & order, int maxPasses = MAX_PASSES);

  /// Find the mean and stdev of the 0-peak (noise) and 1-peak (singlets).
  vec<normal_distribution> FindDistributions(const FlowVector & fvec, 
                                             const FlowKey & key);

  /// Call bases on flow, put them in bases.
  /// \param fvec flow vector to call bases on
  /// \param bases called bases are returned in this parameter.
  /// \param correspondence correlates base position to flow position
   /// Calls FindDistributions and GetBasesWithDistributionsAndCutoffs.
  void GetBases(const FlowVector & fvec, basevector & bases, 
		vec<int> * correspondence = 0);

  /// Call bases on fvec, put them in bases.
  /// \param fvec flow vector to call bases on
  /// \param dist distributions of the 0-peak (noise) and 1-peak (singlets).
  /// \param bases called bases are returned in this parameter. If there
  /// are problems, the basevector will be returned empty.
  /// \param cutoffs cutoffs between peaks (0-peak, 1-peak, etc...) expressed
  /// as the ratio between the cutoff point and 1-peak average. If empty,
  /// we use hardcoded cutoffs.
  /// \param correspondence correlates base position to flow position
  void GetBasesWithDistributionsAndCutoffs
  (const FlowVector & fvec, basevector & bases,
   const vec<normal_distribution> dist,
   const vec<double> & cutoffs, 
		vec<int> * correspondence = 0);
  
  /// Call bases on flow using the cutoffs in the cutoffs vector.
  /// \param fvec flow vector to call bases on
  /// \param bases called bases are returned in this parameter. If there
  /// are problems, the basevector will be returned empty.
  /// \param cutoffs cutoffs between peaks (0-peak, 1-peak, etc...). 
  /// \param  useDoubleValue if true, cutoffs to be compared to the
  /// values in fvec as doubles, i.e. to fvec(i), else cutoffs to be
  /// compared to the values in fvec as unsigned shorts, i.e. to 
  /// fvec[i].
  /// \param correspondence correlates base position to flow position
  void GetBasesWithCutoffs(const FlowVector & fvec, basevector & bases, 
      const vec<double> & cutoffs,
      bool useDoubleValue = false, 
		vec<int> * correspondence = 0);

  /// Calculate individual cutoffs for each flow based on the noise level.
  /// Prerequisites: flow has been normalized so mean1=1 (straight-line
  /// normalization. The noise level calculated by FlowOverlap (i.e. mean0)
  /// is used to estimate the cutoffs.
  /// Equation used: cutoff between peak n and peak n+1 is a 
  /// linear function of the noise,
  /// cutoff = a(n) * mean0 + b(n), where a and b are QuadraticFunctions in n.
  void GetBasesWithNoiseDependentCutoffs
    (const FlowVector & fvec, basevector & bases,
     const QuadraticFunction & a, const QuadraticFunction & b,
     float mean0, vec<int> * correspondence = 0);

  /// Call bases on flow using trivial cutoffs(0.5, 1.5, 2.5, up to ncutoffs).
  /// \param correspondence correlates base position to flow position
  void GetBasesWithTrivialCutoffs(const FlowVector & fvec, 
				  basevector & bases, int ncutoffs=10, 
				  vec<int> * correspondence = 0);

  /// Call bases on flow using trivial cutoffs(0.5, 1.5, 2.5, up to ncutoffs).
  /// \param correspondence correlates base position to flow position
  /// The number of cutoffs is set so we do not lose any information, i.e.
  /// to the max double value in fvec + 1.
  void GetBasesWithTrivialCutoffs(const FlowVector & fvec, 
				  basevector & bases,
				  vec<int> * correspondence = 0);

 private:
  const FlowOrder & mOrder;
  int mMaxPasses;

  static const basevector * BAD_BASES;
  static basevector * SetBadBases(); 

  ///Very simple EM-type attempt to adjust the means to the actual data.
  vec<double> FindNewMeans(const FlowVector & flow, const vec<double> & means);

};

///used to calculate mean squared error using std::inner_product.
template <class T>
struct SquaredError: public binary_function<T, T, T> {
  T operator()(const T & first, const T& second) {
    return (first - second ) * (first - second);
  }
};






