/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/** LabelRandomGenerator will return random labels from a LabelTally object
    based on the label frequency distribution.
    The generator is initialized using a LabelTally object that contains a
    list of Labels and their frequency.
    \class LabelRandomGenerator
*/

#ifndef LABELRANDOMGENERATOR_H
#define LABELRANDOMGENERATOR_H

#include "system/Types.h"
#include "util/LabelTally.h"
#include "random/Random.h"


template <class T> class LabelRandomGenerator {

private:
  vec<int> m_cumFreqTable;
  vec<T> m_labelTable;
  int m_tallySum;
  
public:

  LabelRandomGenerator() {
    m_tallySum = 0;
  }
  
  /// Constructor using a LabelTally object
  LabelRandomGenerator(const LabelTally<T>& t) {
    t.GenerateCumulativeTables(m_cumFreqTable, m_labelTable);
    m_tallySum = t.Sum();
  }

  /// Constructor using name of file containing LabelRandomGenerator object data
  LabelRandomGenerator(string filename) {
    ifstream in;
    OpenIfstream( in, filename );
    in >> (*this);
  }

  int size() const {
    return m_cumFreqTable.size();
  }

  /// Returns a random label
  T GetRandomLabel() const {
    
    int index = (lower_bound(m_cumFreqTable.begin(), m_cumFreqTable.end(),
			     (randomx() % m_tallySum) + 1)
		 - m_cumFreqTable.begin());
    
    return m_labelTable[index];
  }
  
  /// Writes LabelRandomGenerator object to stream
  friend ostream& operator<< ( ostream& o, const LabelRandomGenerator<T> &r) {
    o << r.m_tallySum << "\n";
    o << r.m_cumFreqTable.size() << "\n";
    for (unsigned int j = 0; j < r.m_cumFreqTable.size(); ++j) {
      o << r.m_cumFreqTable[j] << " " << r.m_labelTable[j] << "\n";
    }
    o << flush;
    return o;
  }
  
  /// Reads LabelRandomGenerator object from stream
  friend istream& operator>> ( istream& i,  LabelRandomGenerator<T>& r ) {
    int length;
    i >> r.m_tallySum;
    i >> length;

    r.m_cumFreqTable.empty();
    r.m_labelTable.empty();
    r.m_cumFreqTable.reserve(length);
    r.m_labelTable.reserve(length);

    T label;
    int cumFreq;
    for ( int j = 0; j < length; j++ ) {
      i >> cumFreq >> label;
      r.m_cumFreqTable.push_back(cumFreq);
      r.m_labelTable.push_back(label);
    }

    return i;
  }

  
};


#endif
