/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include <signal.h>
#include <unistd.h>

#include "system/MemTracker.h"

#include "system/Assert.h"
#include "system/RunTime.h"
#include "String.h"
#include "system/SysIncludes.h"
#include "system/System.h"
#include "util/Kronos.h"

#ifdef __solaris
#define SA_NOMASK SA_NODEFER
#endif

void our_new_handler( )
{    cout << "\nCall to new failed, memory usage before call = "
          << MemUsage( ) << "k." << endl;
     cout << "\nHere is the output of top:\n" << endl;
#ifdef __alpha
     System( "top -b | head -15" );
#else
     System( "top -b -n 1 | tail -n +7 | sort -nr -k 10 | head -15" );
#endif
     cout << "Aborting." << endl;
     TracebackThisProcess( );    }

// ===============================================================================
//
//  SetLimits: Make sure that stacksize is always at least 100M.
//
//  On some systems, this may not have an effect.
//
// WARNING WARNING WARNING!!
// If stacksize is 8192 kbytes (or less), Arachne (or some other program)
// may crash in a totally inexplicable way, causing untold grief.  And I
// don't know the exact value which is needed.  So don't lower the limit set
// here unless you have a very compelling reason.
//
// ===============================================================================

const long Mega = 1024 * 1024 ; 

void SetLimits( bool strict )
{    rlimit mem;
     getrlimit( RLIMIT_STACK, &mem );   

     // The following test is here because in certain cases it is not good enough
     // to change the stack size after a main program has started.  It may seg
     // fault in a way that is very hard to trace.  DO NOT CHANGE THIS!

     if ( strict && mem.rlim_cur < 100 * 1000 * 1000 )
     {    cout << "In order to run, this program should have 100MB of stack space.\n"
               << "Please first run \"limit stacksize 100000\" or the equivalent "
               << "command on your system." << endl;
          exit(-1);    }

     mem.rlim_cur = max<long>( mem.rlim_cur, 100 * Mega ) ; 
     mem.rlim_max = max<long>( mem.rlim_max, 100 * Mega ) ; 
     if ( setrlimit( RLIMIT_STACK, &mem ) != 0 ) 
     {    cout << "In order to run, Arachne requires 100MB of stack space.\n"
               << "This amount was requested by Arachne, but denied.  Therefore\n"
               << "you need to find a different computer, or ask a system\n"
               << "administrator to reconfigure the one you are using now.\n"
               << endl;
          exit( -1 ) ;    }    }

// ===============================================================================
//
// NoDump: turn off core dumps.
//
// ===============================================================================

void NoDump( )
{    rlimit core;
     core.rlim_cur = 0;
     core.rlim_max = 0;
     setrlimit( RLIMIT_CORE, &core );    }

// ===============================================================================
//
// arachne_signal_handler: this is the code that is called when an interrupt
// is detected.
//
// ===============================================================================

static const char* SIGSEGV_msg = 
"Segmentation violation.  An attempt will be made to backtrace,\n"
"but the backtrace itself may fail.";
static const char* SIGFPE_msg = "Arithmetic exception.";
static const char* SIGINT_msg = "Interrupt received (perhaps a ctrl-c).  Stopping.";
static const char* SIGBUS_msg = "SIGBUS interrupt.";
static const char* SIGILL_msg = "Illegal instruction.  Something has gone badly wrong.  Stopping.";
static const char* SIGABRT_msg = "Abort.  Stopping.";
static const char* SIGTERM_msg = "Killed.  Stopping.";

void print_signal_handler_message( int signal_number )
{  
  cout << "\n" << Date() << ".  ";
  switch ( signal_number ) {
    case SIGSEGV: cout << SIGSEGV_msg << endl; break;
    case SIGFPE:  cout << SIGFPE_msg << endl;  break;
    case SIGINT:  cout << SIGINT_msg << endl;  break;
    case SIGBUS:  cout << SIGBUS_msg << endl;  break;
    case SIGILL:  cout << SIGILL_msg << endl;  break;
    case SIGABRT: cout << SIGABRT_msg << endl; break;
    case SIGTERM: cout << SIGTERM_msg << endl; break;
    default:      
      cout << "Unrecognized signal (" << signal_number << ") received.  Stopping." << endl;
  }
}

extern Bool interrupt_detected;

void arachne_signal_handler( int signal_number, Bool no_ctrlc )
{
  interrupt_detected = True;

  if ( signal_number != SIGUSR1 && signal_number != SIGUSR2 )
       print_signal_handler_message( signal_number );

  // For SIGINT/SIGTERM, try to kill any child processes
  if (signal_number == SIGINT || signal_number == SIGTERM) {
    ProcTree kronos;
    kill(kronos.my_descendants(), signal_number);
  }

  // We do not attempt to trace back on SIGTERM interrupts, which are the interrupts 
  // resulting when one kills a process (without specifying an interrupt type).
  // The reason for this is that our traceback process includes a stack trace,
  // and to generate the stack trace, we have to fork a process, which in turn
  // asks the operating system for memory.  Under certain circumstances,
  // this can wreak havoc.  Once such circumstance is the simultanous killing of 
  // multiple large memory processes.  We did this once and the operating system
  // did not cope well.
  //
  // However, if you do want to kill a process, and get a traceback, this is still
  // possible: use "kill -INT".

  if ( signal_number == SIGTERM ) _exit(1);
  if ( no_ctrlc && signal_number == SIGINT ) _exit(1);

  if ( signal_number == SIGUSR1 || signal_number == SIGUSR2 )
  { 
#ifdef __solaris   
    String tracefile = "/tmp/traceback_from_process_" + ToString( (int)getpid( ) );
#else
    String tracefile = "/tmp/traceback_from_process_" + ToString( getpid( ) );
#endif
       Ofstream( out, tracefile + "_initial" );
       TracebackThisProcess( out, False, signal_number == SIGUSR1 );
       out.close( );
       Rename( tracefile + "_initial", tracefile );    }
  else
  {    printf( "\nGenerating a backtrace...\n" );
       TracebackThisProcess( cout, True, False );
       //  something not right, ugly fix attempt
       printf( "%s\n%s\n", "Back from backtrace.  This should not have occurred, "
            "but probably you don't care.", "Exiting." );
       _exit(1);    }

  interrupt_detected = False;
}

void arachne_signal_handler_standard( int signal_number )
{    arachne_signal_handler( signal_number, False );    }

void arachne_signal_handler_no_ctrlc_traceback( int signal_number )
{    arachne_signal_handler( signal_number, True );    }

// ===============================================================================
//
// ArachneInterruptHandler: set up run-time interrupt catching.
//
// ===============================================================================

void ArachneInterruptHandler(ArachneSignalHandler* pSigFunc)
{
  // On an Alpha, if you want to catch an unaligned access, you have to first tell
  // the operating system that unaligned accesses are to generate SIGBUG signals.

  #ifdef __alpha
  if ( System( "uac p sigbus" ) != 0 )
  { 
      cout << "Unable to catch unaligned accesses." << endl;
      exit(1);
  }
  #endif

  //  set up the sigaction function by defining members
  struct sigaction act, oact;

  act.sa_handler = pSigFunc;
  act.sa_flags = 0;

  //  init mask to empty
  sigemptyset(&act.sa_mask);

  //  define signals to block
  sigaddset( &act.sa_mask, SIGSEGV );
  sigaddset( &act.sa_mask, SIGFPE  );
  sigaddset( &act.sa_mask, SIGINT  );
  sigaddset( &act.sa_mask, SIGBUS  );
  sigaddset( &act.sa_mask, SIGILL  );
  sigaddset( &act.sa_mask, SIGABRT  );
  sigaddset( &act.sa_mask, SIGTERM  );
  sigaddset( &act.sa_mask, SIGUSR1  );
  sigaddset( &act.sa_mask, SIGUSR2  );

  //  now send the signal to the handler
  if ( sigaction( SIGSEGV, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGSEGV signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGFPE, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGFPE signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGBUS, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGBUS signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGILL, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGILL signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGABRT, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGABRT signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGUSR1, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGUSR1 signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGUSR2, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGUSR2 signal." << endl;
    exit(1);
  }
  if ( sigaction( SIGTERM, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGTERM signal." << endl;
    exit(1);
  }
  act.sa_flags = SA_RESETHAND ^ SA_NOMASK;
  if ( sigaction( SIGINT, &act, &oact ) < 0 )
  {
    cout << "Error catching SIGINT signal." << endl;
    exit(1);
  }
}

// Helper for setting advisory locks, from Stevens.
int lock_reg(int fd, int cmd, int type, off_t offset, int whence, off_t len)
{
  struct flock lock;
  lock.l_type = type;
  lock.l_start = offset;
  lock.l_whence = whence;
  lock.l_len = len;
  return fcntl(fd, cmd, &lock);
}

// Place an advisory lock on the current executable file.  This will
// go away automatically when the process ends for any reason, as
// will the file descriptor that's needed to make the lock.  We use
// the lock to prevent make from overwriting the current executable,
// as that may have disastrous effects on the running process in
// certain circumstances.  
void readLockThisExe()
{
  // Note: While this works on LSF and wga*, it does not work on the
  // alphas.  There is no such file so the open() will fail and no
  // lock will be placed.
  const char filename[] = "/proc/self/exe";
  int ret;
  int fid = open(filename, O_RDONLY, 0);
  if (fid>0) {
    ret = lock_reg(fid, F_SETLK, F_RDLCK, 0, SEEK_SET, 0);
    // Note we ignore errors in setting the lock. 
  }
}



// ===============================================================================
//
// RunTime: provide for various run-time services: turn off core dumps, make sure
// stack size is big enough, catch some interrupts.
//
// ===============================================================================

void RunTime( int no_dump, ArachneSignalHandler* pSigFunc, bool strict )
{
  // Decouple C-style stdio and C++-style c{out,err,log} streams.

  ios::sync_with_stdio(false);

  // Turn off core dumps.

  if ( no_dump ) NoDump();

  // Set up to catch failures of new.

  std::set_new_handler(our_new_handler);

  // Make sure stack size is big enough.

  SetLimits(strict); // Don't even think about removing this.
                     // Stack size too small will cause hideously untraceable errors.

  // Prepare to catch interrupts.

  if ( pSigFunc ) ArachneInterruptHandler(pSigFunc);

  // Ensure that shell is compliant with XPG4.  The only known affect of this
  // is to prevent procbuf closures from producing spurious blank lines.

  putenv( "BIN_SH=xpg4" );

  // Register the memory tracker's atexit function.  This function will have
  // no effect if no modules are doing memory tracking.

  atexit( &check_memory_tracker_records );

  readLockThisExe();
}
