/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef PARSEDARGS
#define PARSEDARGS

/**
   Header: ParsedArgs.h

   Support for command-line argument parsing.

   This include file supports parsing of command-line arguments, according to the
   following scheme.  First, your "main program line" should be:
   
   > int main( int argc, char *argv[] )

   Second, to parse the arguments, do something like:
   
   > BeginCommandArguments;
   > CommandArgument_UnsignedInt(MAXT);
   > CommandArgument_String_OrDefault(TACO, "cheese");
   > EndCommandArguments;

   Here's what happens under various scenarios:

   (begin example)
   
   command line                   action
   
   a.out MAXT=5 TACO=beef         unsigned int MAXT = 5; String TACO = "beef";
   
   a.out MAXT=34                  unsigned int MAXT = 34; String TACO = "cheese";
   
   a.out TACO=broccoli            error because MAXT missing
   
   a.out MAXT=4 TACO=potato N=4   error because N is a superfluous argument

   (end example)

   If a variable name X is undefined on the command line, but an 
   environment variable 
   ARACHNE_X is defined, its value is used for X.  If X is undefined and ARACHNE_X
   is undefined in the environment, but FetchArgsFromFile is used, and X is 
   defined
   there, then we use that value.  Otherwise, if a default value is given, we use
   that.  Otherwise, an error is flagged.
   
   Special arguments of the form @@f, where f is a filename, cause extra arguments
   to be read from the file f.  Any line starting with # in this treated as a comment.
   
   @file
*/

// When I have to do a return (even though the return statement can't be
// reached), I return GARBAGE.
#define GARBAGE 0
#define GARBAGESTRING String("Unreachable")

#include <stdio.h>

#include <fstream>

#include "String.h"
#include "system/Types.h"
#include "Vec.h"
#include "TokenizeString.h"
#include "ParseSet.h"

class parsed_arg_help {

 public:
  parsed_arg_help( const String& name, 
		   const String& type, 
		   const String& abbreviation,
		   const String& default_value,
		   const String& documentation)
    : name_(name), 
      type_(type), 
      abbreviation_(abbreviation), 
      default_(default_value),
      doc_(documentation)
  { }
 
  const bool operator< ( const parsed_arg_help& rhs ) const
  {
    return name_ < rhs.name_;
  }
  
  friend ostream& operator<< ( ostream& out, const parsed_arg_help& the_arg );
  bool isRequired() const {
    return default_ == "<required>";
  }

 private:
  String name_;
  String type_;
  String abbreviation_;
  String default_;
  String doc_;
};

//namespace parsed_args_string {
  template<class T> inline String Stringize(T t) { return ToString(t); }
  template<> String inline Stringize<Bool>(Bool b) {
    return b ? "True" : "False";
  }
//};
    
///Contains all Arachne command-line parameters
class parsed_args {

  static const Bool INVALID_BOOL;
  static const int INVALID_INT;
  static const unsigned int INVALID_UINT;
  static const double INVALID_DOUBLE;
  static const String INVALID_STRING;

  //using parsed_args_string::Stringize;

 public:

  static Bool pretty_help;
  static String usage_format;
  static String param_format;
  static String header_format;
  static String info_format;
  static String doc_format;


  parsed_args( int argc, char *argv[], String make_date );

  void FetchArgsFromFile( const String& filename );

  ///Put a command-line parameter and its value into this object.
  ///This method assumes that the parameter and its value are in the
  /// no-whitespace form "PARAM=Value"
  bool ParseString( const String& string );

  /// Put a command-line parameter and its value into this object.
  /// This method handles the case where the value is separated from the
  /// = sign by whitespace (e.g. "Param= Value"). 
  /// It is useful to be able to do this so as to 
  /// use tab completion in the shell.
  bool ParseString( int argc, char **argv, int & i);

  Bool GetHelpOnly( );

  void SetHelpOnly(bool b) { get_help_ = b; }

  void ConversionFailure(int i);

  void NoValue(String n);

  String CheckForArgAndAbbreviation( String n, String abbr );


  template<class T> 
    void GetValue (String n, T & value, const String & abbr = "", 
	const String & deflt = "<required>");

  template<class T>
    void ProcessArgs(T & value, const String & name,
	const String & type, const String & abbr,
	const String & def, const String & doc="") {
      if ( GetHelpOnly() )
	AddArgHelp( name, type, abbr, def, doc );
      else
	GetValue( name, value, abbr, def );
    }

  template<class T>
    void ProcessArgs(T & value, const String & name,
	const String & type, const String & abbr,
	const char * def, const String & doc="") {
    ProcessArgs(value, name, type, abbr, String(def), doc);
  }

  template<class T, class S>
    void ProcessArgs(T & value, const String & name,
	const String & type, const String & abbr,
	S def, const String & doc="") {
      if ( GetHelpOnly() )
	AddArgHelp( name, type, abbr, Stringize(T(def)), doc );
      else
	GetValue( name, value, abbr, Stringize(T(def)) );
    }


  String GetStringValue( String n, String abbr = "", 
      String deflt = INVALID_STRING);

  Bool GetBoolValue(String n, String abbr = "", Bool deflt = INVALID_BOOL );

  int GetUnsignedIntValue( String n, String abbr = "", 
                           unsigned int deflt = INVALID_UINT);

  int GetIntValue( String n, String abbr = "", int deflt = INVALID_INT);

  double GetDoubleValue(String n, String abbr = "", 
			double deflt = INVALID_DOUBLE);
 
  void CheckForExtraArgs( );

  void AddArgHelp( const String& arg, 
                   const String& type, 
                   const String& abbreviation,
                   const String& default_value,
		   const String& documentation="");
     
  void PrintArgHelp( );

  String TheCommand( );

  void PrintTheCommandPretty( ostream& out, String make_date, 
       const String& prefix = "" );

  ///Log all commands to command logfiles separated by month (note default). 
  void LogTheCommand( String logfileBase );
  ///Log all commands to command logfiles separated by month to the default dir.
  void LogTheCommand( );

  /// Removes command-line parameter from object
  bool RemoveArg(String n, String abbr = "");

  void AddDocString(const char *doc) { doc_ = doc; }

  vec<String> GetArgNames() const { return name_; }

 private:

  void CheckEnv( const String& n, const String& abbr );

  String command_;
  vec<String> name_, def_;
  vec<Bool> used_;
  String orig_command_;

  Bool get_help_;
  vec<parsed_arg_help> arg_help_;

  const char * doc_;
};

#ifndef MAKE_DATE
#define MAKE_DATE
#endif

// Getting a string constant from a define is hard.
#define CSTRING_FROM_DEFINE(X) CSTRING_FROM_DEFINE_SUB(X)
#define CSTRING_FROM_DEFINE_SUB(X) #X


#define BeginCommandArguments \
     parsed_args command(argc, argv, CSTRING_FROM_DEFINE(MAKE_DATE) );

#define CommandDoc(STRING) \
     command.AddDocString(STRING)

#define PrintCommandPretty(OUT) \
     command.PrintTheCommandPretty( OUT, CSTRING_FROM_DEFINE(MAKE_DATE) );

#define FetchCommandArgumentsFromDefaultsFile( defaults_file ) \
     command.FetchArgsFromFile( defaults_file );

#include "system/ParsedArgsAuto.h"

#define CommandArgument_VecString(NAME) \
     vec<String> NAME; \
     String NAME ## _COLON_CONCATENATED_LIST;\
     if ( command.GetHelpOnly() ) \
       command.AddArgHelp( #NAME, ":-sep strings", "", "<required>" ); \
     else { \
       NAME ## _COLON_CONCATENATED_LIST = command.GetStringValue( #NAME );\
       Tokenize(NAME ## _COLON_CONCATENATED_LIST, NAME, ":");\
     }

#define CommandArgument_VecString_OrDefault(NAME, DEFAULT) \
     vec<String> NAME; \
     String NAME ## _COLON_CONCATENATED_LIST;\
     if ( command.GetHelpOnly() ) \
       command.AddArgHelp( #NAME, ":-sep strings", "", DEFAULT ); \
     else { \
       NAME ## _COLON_CONCATENATED_LIST = command.GetStringValue( #NAME, "", DEFAULT );\
       Tokenize(NAME ## _COLON_CONCATENATED_LIST, NAME, ":");\
     }

#define EndCommandArguments \
     if ( command.GetHelpOnly() ) \
     { \
       command.PrintArgHelp( ); \
       exit(0); \
     } \
     else \
       command.CheckForExtraArgs( );

#define EndCommandArgumentsAllowExtras \
     if ( command.GetHelpOnly() ) \
     { \
       command.PrintArgHelp( ); \
       exit(0); \
     }
     
///Use this macro if you want your program to run with no arguments.
///(instead of printing usage, that is). Mainly intended for unit tests.
#define BeginCommandArgumentsAcceptEmptyArgList \
BeginCommandArguments; if (argc == 1) { command.SetHelpOnly(false);} else{}

#endif
