/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

///\file Assert.cc
/// Assert.cc provides TracebackThisProcess and some of the guts of the
/// Assert suite of macros (defined in Assert.h).

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <string>
#include <strstream>

#include "String.h"
#include "system/System.h"
#include "system/Types.h"

#ifdef __alpha
     #include <excpt.h>
     #include <setjmp.h>
#endif

#ifdef __i386__
#ifndef __solaris
     #include <execinfo.h>
#endif
#endif

#if __ia64__ || __x86_64__
     #include <unwind.h>
#endif

/// ===========================================================================
///
/// ReturnAddress(i), where 0 <= i <= 100: get the return address.  The 
/// implementation given here only works for g++, and is only known to work on
/// Intel boxes.  (It does not work on an alpha.)
///
/// ===========================================================================

inline void* ReturnAddress(int i)
{
     #ifdef __GNUC__

	
          #define RETURN_ADDR(I) if ( i == I ) return __builtin_return_address(I);

          RETURN_ADDR(0) 
	  RETURN_ADDR(1) 
	  RETURN_ADDR(2) 
	  RETURN_ADDR(3) 
          RETURN_ADDR(4) 
	  RETURN_ADDR(5) 
	  RETURN_ADDR(6) 
	  RETURN_ADDR(7) 
          RETURN_ADDR(8) 
	  RETURN_ADDR(9) 
	  RETURN_ADDR(10) 
	  RETURN_ADDR(11) 
          RETURN_ADDR(12) 
	  RETURN_ADDR(13) 
	  RETURN_ADDR(14) 
	  RETURN_ADDR(15) 
	  RETURN_ADDR(16) 
          RETURN_ADDR(17) 
	  RETURN_ADDR(18) 
	  RETURN_ADDR(19) 
	  RETURN_ADDR(20) 
	  RETURN_ADDR(21) 
          RETURN_ADDR(22) 
	  RETURN_ADDR(23) 
	  RETURN_ADDR(24) 
	  RETURN_ADDR(25) 
	  RETURN_ADDR(26) 
          RETURN_ADDR(27) 
	  RETURN_ADDR(28) 
	  RETURN_ADDR(29) 
	  RETURN_ADDR(30) 
	  RETURN_ADDR(31) 
          RETURN_ADDR(32) 
	  RETURN_ADDR(33) 
	  RETURN_ADDR(34) 
	  RETURN_ADDR(35) 
	  RETURN_ADDR(36) 
          RETURN_ADDR(37) 
	  RETURN_ADDR(38) 
	  RETURN_ADDR(39) 
	  RETURN_ADDR(40) 
	  RETURN_ADDR(41) 
          RETURN_ADDR(42) 
	  RETURN_ADDR(43) 
	  RETURN_ADDR(44) 
	  RETURN_ADDR(45) 
	  RETURN_ADDR(46) 
          RETURN_ADDR(47) 
	  RETURN_ADDR(48) 
	  RETURN_ADDR(49) 
	  RETURN_ADDR(50) 
	  RETURN_ADDR(51) 
          RETURN_ADDR(52) 
	  RETURN_ADDR(53) 
	  RETURN_ADDR(54) 
	  RETURN_ADDR(55) 
	  RETURN_ADDR(56) 
          RETURN_ADDR(57) 
	  RETURN_ADDR(58) 
	  RETURN_ADDR(59) 
	  RETURN_ADDR(60) 
	  RETURN_ADDR(61) 
          RETURN_ADDR(62) 
	  RETURN_ADDR(63) 
	  RETURN_ADDR(64) 
	  RETURN_ADDR(65) 
	  RETURN_ADDR(66) 
          RETURN_ADDR(67) 
	  RETURN_ADDR(68) 
	  RETURN_ADDR(69) 
	  RETURN_ADDR(70) 
	  RETURN_ADDR(71) 
          RETURN_ADDR(72) 
	  RETURN_ADDR(73) 
	  RETURN_ADDR(74) 
	  RETURN_ADDR(75) 
	  RETURN_ADDR(76) 
          RETURN_ADDR(77) 
	  RETURN_ADDR(78) 
	  RETURN_ADDR(79) 
	  RETURN_ADDR(80) 
	  RETURN_ADDR(81) 
          RETURN_ADDR(82) 
	  RETURN_ADDR(83) 
	  RETURN_ADDR(84) 
	  RETURN_ADDR(85) 
	  RETURN_ADDR(86) 
          RETURN_ADDR(87) 
	  RETURN_ADDR(88) 
	  RETURN_ADDR(89) 
	  RETURN_ADDR(90) 
	  RETURN_ADDR(91) 
          RETURN_ADDR(92) 
	  RETURN_ADDR(93) 
	  RETURN_ADDR(94) 
	  RETURN_ADDR(95) 
	  RETURN_ADDR(96) 
          RETURN_ADDR(97) 
	  RETURN_ADDR(98) 
	  RETURN_ADDR(99) 
	  RETURN_ADDR(100) 
          cout << "Stack too deep.\n";
          exit(1);   

     #else
     
          cout << "ReturnAddress only works if you compile with g++." << endl;
          exit(1);
          return( cout << endl ) ; // returns (void*) 

     #endif

           }


/// ===========================================================================
///
/// unwindOneLevel: I don't know what this does.  Only defined for alpha.
///
/// ===========================================================================

#ifdef __alpha
     void unwindOneLevel(
          struct sigcontext *unwindSignalContext,
          void *             runtimeProcedurePtr)
     {    unwind(unwindSignalContext, (pdsc_crd *)runtimeProcedurePtr);    }
#endif


// ============================================================================
///
/// SimplifyFunctionName: Replace its argument, say GerbilSpit( int, int ), with
/// GerbilSplit( ... ).  Some exceptions are dealt with.
//
// ============================================================================

String SimplifyFunctionName( String f )
{    
     // If the function name is short, we may as well print it in its entirety:

     if ( f.size( ) < 60 ) return f;
     
     // Don't mess with operators, as their full name may contain useful
     // information:

     if ( f.Contains( "operator" ) ) return f;

     // I'm not sure what this case would be:

     if ( !f.Contains( "(" ) ) return f;

     // The main cases:

     if ( f.Contains( ")", -1 ) ) return f.Before( "(" ) + "( ... )";
     if ( f.Contains( ") const", -1 ) ) return f.Before( "(" ) + "( ... )";

     // Fallen through, not sure why:

     return f;    }


// ===========================================================================
///
/// IgnoreFunction: Ignore stack dump entries for certain functions.
///
// ===========================================================================

Bool IgnoreFunction( String function )
{    Bool first_print(True);
     if ( first_print && function == "??" ) return True;
     if ( function.Contains( "TracebackThisProcess", 0 ) ) return True;
     if ( function.Contains( "Assert", 0 ) ) return True;
     if ( function.Contains( "arachne_signal_handler", 0 ) ) return True;
     if ( function.Contains( "dump_stack", 0 ) ) return True;
     first_print = False;
     return False;    }


// ==========================================================================
///
/// PrintFrame: print a given return address, prettified.
///
// ==========================================================================

void PrintFrame( void* returnAddress, ostream& out )
{    static int count(0);
     if ( count == 0 ) out << "[raw stack entries =" << flush;
     if ( count > 0 && count % 3 == 0 ) out << "\n                     ";
     else out << " ";

     // The following foolishness has the affect of getting returnAddress
     // printed in a 14-character wide field.  The direct approach didn't work.

     strstream hex_stream;
     hex_stream << hex << returnAddress << endl;
     String sp_addr;
     hex_stream >> sp_addr;
     out << setw(18) << sp_addr << flush;
     ++count;    }

Bool interrupt_detected(False);

// ===========================================================================
///
/// PrintStack: given as input the stack addresses, print a stack dump, exit.
/// This only works with g++.
//
// ===========================================================================

void PrintStack( const vector<void*>& sp_addresses, String command, 
     ostream& out, Bool exit_when_done = True, Bool minimal = False )
{    if ( !minimal) out << "]\n" << endl; // closure for calls to PrintFrame
     temp_file tempfile( "/tmp/temp_PrintStack_XXXXXXX" );
     signal( SIGINT, SIG_DFL );
     char* backtracer = getenv( "BACKTRACER" );
     if ( ! backtracer ) backtracer = "addr2line";

     String addr2line_command( backtracer );
     addr2line_command += " -i -e " + command + " -f -s -C ";

     for ( unsigned int i = 0; i < sp_addresses.size(); ++i )
     {    strstream hex_stream;
          #if __ia64 || __x86_64
               hex_stream << hex << (void *)(((char *)sp_addresses[i])-1) << endl;
          #else
               hex_stream << hex << sp_addresses[i] << endl;
          #endif
          String sp_addr;
          hex_stream >> sp_addr;
          addr2line_command += sp_addr + " ";    }
     addr2line_command += String("> ") + tempfile;
     if ( System( addr2line_command ) != 0 )
     {    out << "Call to " << backtracer << " failed." << endl;
          out << "Perhaps an interrupt was received, or perhaps "
               << backtracer << " could not be found." << endl << endl;
          _exit(1);    }
     ifstream tempstream( tempfile.c_str( ) );
     int depth = 0;
     String function, where;
     for ( unsigned int i = 0; i < sp_addresses.size(); ++i )
     {    if ( !tempstream ) break;
          getline( tempstream, function );
          getline( tempstream, where );
          if ( !IgnoreFunction(function) )
          {    out << depth++ << ". ";
               if ( function == "??" ) out << "??" << endl ;
               else out << SimplifyFunctionName(function) << ", in " << where 
                    << endl ;    }
          if ( function == "main" ) break;    }
     remove( tempfile.c_str( ) );
     if ( !minimal ) out << endl;
     if (exit_when_done)
     {    if (interrupt_detected) _exit(1);
          else exit(1);    }    }

// =========================================================================
///
/// dump_stack: generate a human-readable printout of the stack, exit.
/// This is only known to work on Alpha and Intel platforms, and only works under
/// g++.
//
// =========================================================================

  #if __ia64__ || __x86_64__ 

struct ia64_unwind_struct 
{
  static const int max_frames = 101;
  int num_frames;
  void * stacklist[max_frames];
};

_Unwind_Reason_Code ia64_unwind_callback(struct _Unwind_Context *info, void *arg)
{
  unsigned long ip;
  struct ia64_unwind_struct *unwind_data = (struct ia64_unwind_struct *)arg;

  if ( unwind_data->num_frames == unwind_data->max_frames )
    return _URC_END_OF_STACK;
  
  ip = _Unwind_GetIP( info );
  unwind_data->stacklist[unwind_data->num_frames++] = (void *)ip;

  return _URC_NO_REASON;
}

  #endif

void dump_stack( String command, ostream& out, Bool exit_when_done = True,
     Bool minimal = False )
{    if ( !minimal ) out << "\nDump of stack:" << endl << endl;
     vector<void*> sp_addresses;

#ifdef __solaris
     cout << "Sorry, stack dump does not work on Solaris " << endl;
     
     exit(1);
#else //__solaris
     #ifndef __GNUC__
          cout << "Sorry, stack dump only works if code was "
               << "compiled with g++." << endl;
          exit(1);
     #endif

     #ifdef __i386__

          void* return_addresses[101];
          int nback = backtrace( return_addresses, 101 );
	  for ( int j = 0; j < nback - 1; j++ )
          {    if ( !minimal ) PrintFrame( ReturnAddress(j), out );
               sp_addresses.push_back( ReturnAddress(j) );   }

     #endif

  #if __ia64__ || __x86_64__
          
          struct ia64_unwind_struct unwind_data;
          unwind_data.num_frames = 0;

          _Unwind_Backtrace( &ia64_unwind_callback, &unwind_data );

          for ( int depth = 0; depth < unwind_data.num_frames; ++depth )
          {
            if ( !minimal ) PrintFrame( unwind_data.stacklist[ depth ], out );
            sp_addresses.push_back( unwind_data.stacklist[ depth ] );
          }

  #endif

     #ifdef __alpha

          #define RETURNADDRREG (26)
          #define FAULTING_ADDRESS sc_traparg_a0

          // Get current execution context.

          jmp_buf context;
          setjmp(context);

          // Set the initial context for the unwind.

          struct sigcontext* signalContextPtr = (struct sigcontext *)context;
          struct sigcontext unwindSignalContext = *signalContextPtr;
  
          // Discard the frame for dump_stack() and TracebackThisProcess().

          int numLevelsToDiscard = 2;
          for ( int level = 0; level < numLevelsToDiscard; level++ )
          {    unsigned long int programCounter = unwindSignalContext.sc_pc;
               void* runTimeProcedurePtr = find_rpd(programCounter);
               unwindOneLevel(&unwindSignalContext, runTimeProcedurePtr);    }

          // Pick out the return address and program counter.

          unsigned long returnAddress  = unwindSignalContext.sc_regs[RETURNADDRREG];
          unsigned long programCounter = unwindSignalContext.sc_pc;
  
          // This is the address that caused the fault when we tried to access

          unsigned long faultingAddress = signalContextPtr->FAULTING_ADDRESS;
  
          // Special cases for bogus program counter values. If the program
          // counter is zero or the fault occurred when we were trying to
          // fetch an instruction (because the program counter itself was bad)
          // then we cannot unwind the stack.

          if (programCounter == 0) 
               out << "\nPC is zero - stack trace not available.\n";
          else if (programCounter == faultingAddress) 
               out << "\nbad PC (" << faultingAddress 
                    << ") - stack trace not available.\n";
  
          else 
          {    unsigned int sameSpCount = 0;

               // Loop through all the stack frames.

               unsigned long stackpointer = 0;
               void *runTimeProcedurePtr;

               while  ((returnAddress != 0) && (programCounter != 0)) 
               {
                    // Get the run time procedure descriptor for this frame.

                    runTimeProcedurePtr = find_rpd(programCounter);
      
                    if ( !minimal ) PrintFrame( (void*) returnAddress, out );
                    sp_addresses.push_back( (void*) returnAddress );

                    // Unwind one level.

                    unwindOneLevel(&unwindSignalContext, runTimeProcedurePtr);
                    returnAddress = unwindSignalContext.sc_regs[RETURNADDRREG];
                    programCounter = unwindSignalContext.sc_pc;
      
                    if ((unsigned int) unwindSignalContext.sc_sp <= stackpointer) 
	            {    if ( ++sameSpCount == 10 ) break;    }
                    else 
                    {    sameSpCount  = 0;
	                 stackpointer = unwindSignalContext.sc_sp;    }    }    }
     #endif

     PrintStack( sp_addresses, command, out, exit_when_done, minimal );    

#endif //__solaris
}

// ===========================================================================
/// \fn TracebackThisProcess
/// TracebackThisProcess: generate a human-readable stack dump for this process.
/// This only works if g++ was used, and is only known to work on Alpha and 
/// Intel boxes.
///
/// Sometimes gdb will produce a more informative stack dump.  Somehow it does
/// a better job than addr2line does of converting addresses into
/// function names/filenames/line numbers.  If you want to see the gdb stack 
/// dump 
/// instead of what we would otherwise give, set the global variable
/// use_gdb_for_tracebacks to True.  This is done by ParsedArgs.cc, when the
/// command-line option GDB=True is given.
///
// ==========================================================================

Bool use_gdb_for_tracebacks(False);

void TracebackThisProcess( ostream& out, Bool exit_when_done, Bool minimal )
{    
     // If called by gdb or emacs, crash.

     String mom = command_name_of_process( getppid( ) );
     if ( mom == "gdb" || mom == "emacs" )
     {    out << "called by gdb or emacs; crashing...\n" << flush;
          abort( );    }

     // Get the id of this process.

     int pid = getpid( );

     // Find the executable that was used, or a link to it.  On Linux systems,
     // we check to make sure that it was not deleted or overwritten, and if it 
     // was moved, we give a link that points to the moved file.  Otherwise,
     // we don't do this.

     String exe;
          
     #ifdef __linux

          exe = "/proc/" + ToString(pid) + "/exe";
          char* buf = new char[500];
          if ( readlink( exe.c_str( ), buf, 500 ) < 0 )
          {    out << "Attempt to link to executable failed.  Weird." << endl;
               exit(1);    }
          String buf_string(buf);
          if ( buf_string.Contains( "(deleted)" ) )
          {    out << "Traceback is impossible because the original executable "
                    << "no longer exists.  Bummer." << endl << endl;
               exit(1);    }
     #else

          exe = command_name_of_process(pid);

     #endif

     // On Alpha and Intel boxes, we use a built-in stack dump.  Otherwise,
     // call gdb.  None of this will work if you did not compile with g++.

     #if __alpha || __i386__ || __ia64__ || __x86_64__
          if ( !use_gdb_for_tracebacks) 
               dump_stack( exe, out, exit_when_done, minimal );
          if ( !exit_when_done ) return;
     #endif

     out << "\nInvoking gdb to backtrack...\n";
     out << "(If process stops, you may have to foreground (fg) it.)" << endl;
     Echo( "attach " + ToString(pid), "bt_file" );
     Echo( "bt",  "bt_file" );
     Echo( "quit",  "bt_file" );
     if ( System( "gdb -batch -x bt_file -q " + exe ) != 0 )
          out << "Bummer.  My call to gdb failed." << endl;
     Remove( "bt_file" );
     if (exit_when_done) exit(-1);    }


// ==========================================================================
//
// The rest of this file consists of assert stuff.
//
// ==========================================================================

#define AssertBodyMACRO( expr_string, function_name, line_num, file_name )   \
     cout << "\nAt " << Date( ) << ",\nAssert " << expr_string               \
          << " failed,\nin " <<                                              \
          SimplifyFunctionName(function_name) << ",\nline " << line_num      \
          << " of file " << file_name << "." << endl;                        \
     TracebackThisProcess( );

#define Assert_MACRO_CORE( assert_name, expr1, expr2, expr1_string,          \
     expr2_string, function_name, line_num, file_name, op )                  \
     cout << "\nAt " << Date( )                                              \
          << ",\n" << #assert_name << " " << expr1_string << " " << op       \
          << " " << expr2_string << " failed,\nin "                          \
          << SimplifyFunctionName(function_name) << ",\nline " << line_num   \
          << " of file " << file_name << ".\n";                              \
     cout << "Actually, " << expr1_string << " = " << expr1 << " and "       \
          << expr2_string << " = " << expr2 << "." << endl;                  \
     TracebackThisProcess( );

#define AssertEqBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertEq, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, "==" )

#define AssertNeBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertNe, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, "!=" )

#define AssertLtBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertLt, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, "<" )

#define AssertGtBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertGt, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, ">" )

#define AssertLeBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertLe, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, "<=" )

#define AssertGeBodyMACRO( expr1, expr2, expr1_string, expr2_string,         \
     function_name, line_num, file_name )                                    \
     Assert_MACRO_CORE( AssertGe, expr1, expr2, expr1_string, expr2_string,  \
          function_name, line_num, file_name, ">=" )


// FAST_ASSERT versions

void AssertBody( const char* expr_string, const char* function_name, 
		 const int line_num, const char* file_name )
{    AssertBodyMACRO( expr_string, function_name, line_num, file_name );    }

void AssertEqBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertEqBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertEqBody( const string& expr1, const string& expr2, 
		   const char* expr1_string, const char* expr2_string,
		   const char* function_name, const int line_num, 
                   const char* file_name )
{    AssertEqBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertNeBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertNeBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertNeBody( const string& expr1, const string& expr2, 
		   const char* expr1_string, const char* expr2_string,
		   const char* function_name, const int line_num, 
                   const char* file_name )
{    AssertNeBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertLtBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertLtBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertGtBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertGtBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertLeBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertLeBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }

void AssertGeBody( longlong expr1, longlong expr2, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name )
{    AssertGeBodyMACRO( expr1, expr2, expr1_string, expr2_string,
          function_name, line_num, file_name );    }


// DEFAULT versions

#ifndef DEFAULT_ASSERT

// If DEFAULT_ASSERT is not defined, we're using some other Assert
// method, which, in turn, means we haven't defined these classes yet.

#include "system/AssertStructures.h"

#endif

void AssertBody( const assert_structure* this_assert )
{    AssertBodyMACRO( this_assert->expr_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertEqBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertEqBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertEqBody( const string& expr1, const string& expr2, 
     const assert_structure2* this_assert )
{    AssertEqBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertNeBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertNeBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertNeBody( const string& expr1, const string& expr2, 
     const assert_structure2* this_assert )
{    AssertNeBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertLtBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertLtBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertGtBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertGtBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertLeBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertLeBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }

void AssertGeBody( longlong expr1, longlong expr2, 
                   const assert_structure2* this_assert )
{    AssertGeBodyMACRO( expr1, expr2, this_assert->expr1_string, 
          this_assert->expr2_string, this_assert->function_name, 
          this_assert->line, this_assert->filename );    }
