// Copyright (c) 2004 Broad Institute/Massachusetts Institute of Technology

// An attempt to visualize what's going on during the walking of an insert

#ifndef PATHS_WALKPLOT
#define PATHS_WALKPLOT

#include "CoreTools.h"
#include "Basevector.h"
#include "paths/KmerBaseBroker.h"
#include "paths/KmerPath.h"
#include "paths/PathWalk.h"

class WalkPlotter {

public:

  enum WalkType 
    { DEAD_END, CLOSED, HEAP_EXPLOSION, OPEN, TRUTH };
    // NOTE that these must take values 0,1,2,..., since they are used
    // as indices into the following arrays.  Yes, I suppose we could
    // change these to map<WalkType,char*>, if you want to be pedantic.
  static const char *TypeName[];
  static const char *TypeColor[];
  static const char *TypeLightColor[];

  // constructor-destructor:
  WalkPlotter( KmerBaseBroker *kbb ) : 
    mp_kbb       ( kbb ),
    I_own_my_kbb ( false ),
    K            ( kbb->GetK() ),
    draw_gaps    ( true ),
    max_path_count( 10000 ),
    path_offset  ( 0 ),
    zoom_level   ( 0 )
  { setup_weights(); }

  WalkPlotter( String run_dir, int k ) : 
    mp_kbb       ( new KmerBaseBroker(run_dir, k) ),
    I_own_my_kbb ( true ),
    K            ( k ),
    draw_gaps    ( true ),
    max_path_count( 10000 ),
    path_offset  ( 0 ),
    zoom_level   ( 0 )
  { setup_weights(); }

  ~WalkPlotter( ) {
    if( I_own_my_kbb ) delete mp_kbb;
  }

  void Add( const KmerPath& new_path, WalkType type ) {
    paths_to_plot.push_back( new_path );
    path_types.push_back( type );
  }

  void AddClosure( const KmerPath& new_path ) {
    Add( new_path, CLOSED );
  }

  // This path is printed with type TRUTH, and is printed *last*,
  // so it isn't covered up by anything else.
  // Remove the truth by setting it to the empty KmerPath
  void SetTruth( const KmerPath& truth ) {
    truth_path = truth;
  }

  void Plot() const;

  void Clear() {
    paths_to_plot.clear();
    path_types.clear();
    truth_path.Clear();
  }
  
  void DrawGaps( bool draw ) 
    { draw_gaps = draw; }

  void MaxPathCount( unsigned int max )
    { max_path_count = max; }

  // How many vertical pixels should we offset successive paths?
  void PathOffset( int offset ) 
    { path_offset = offset; }

  // How zoomed in should gv be? (0 = 1.0, -5 = out 10x, 5 = in 10x )
  void ZoomLevel( int zoom ) 
    { 
      ForceAssertGe( zoom, -7 );
      ForceAssertLe( zoom, 7 );
      zoom_level = zoom; 
    }

private:
  // internal methods:
  void setup_weights();
  float kmer_to_real( basevector kmer ) const;
  float kmer_to_real( longlong k ) const 
  { return kmer_to_real( mp_kbb->Bases(k) ); }

  // data members:
  KmerBaseBroker* mp_kbb;
  bool I_own_my_kbb;
  int K;
  vec<double> base_weight;
  mutable vec<KmerPath> paths_to_plot;
  mutable vec<WalkType> path_types;
  bool draw_gaps;  // draw lines to point out gaps?
  unsigned int max_path_count;
  int  path_offset; // vertical pixels between path starts 
  int  zoom_level;  // how zoomed in should gv be?
  KmerPath truth_path;  // The One True Kmer Path: plot this last.
};









#endif
