// Copyright (c) 2004 Broad Institute/Massachusetts Institute of Technology

// An attempt to visualize what's going on during the walking of an insert

#include "paths/WalkPlot.h"
#include <map>
#include <set>


// Weights are roughly e^(-x^2), with the kmer stretching from -2 to 2.
void WalkPlotter::setup_weights() {
  base_weight.resize(K);
  double tot=0;

  for(int i=0; i<K; i++)
    tot += base_weight[i] = exp(-(2*K-1-4*i)*(2*K-1-4*i)/((double)K*K));
  // Then normalize so all 3s (Ts) will yeild a kmer weight of 1:
  tot *= 3;
  for(int i=0; i<K; i++)
    base_weight[i] /= tot;
}

// A C G T
const int base_lookup[4] = { 0, 1, 2, 3 };

// A T C G
//const int base_lookup[4] = { 0, 2, 3, 1 };

// Converts a length-K basevector to a value in [0,1] using above weights
// The class declaration overloads this so it can take a kmer number instead.
float WalkPlotter::kmer_to_real( basevector kmer ) const {
  if( (int)kmer.size() != K ) return 0.0;
  double val=0.0;  // accumulate in high precision
  for(int i=0; i<K; i++)
    val += base_weight[i] * base_lookup[kmer[i]];
  return val;
}


// Names and colors for the types:
const char* WalkPlotter::TypeName[] = 
  { "DEAD_END", "CLOSED", "HEAP_EXPLOSION", "OPEN", "TRUTH" };
const char* WalkPlotter::TypeColor[] = 
  { "0 setgray", "0 1 0 setrgbcolor", "1 0 0 setrgbcolor", 
    "0 0 1 setrgbcolor", "1 .5 0 setrgbcolor"};
const char* WalkPlotter::TypeLightColor[] = 
  { ".8 setgray", ".8 1 .8 setrgbcolor", "1 .8 .8 setrgbcolor", 
    ".8 .8 1 setrgbcolor", ".1 .9 .8 setrgbcolor" };


// A struct for Plot to track what's already been done for an interval
struct Done {
  String  name;   // name of the PS routine that plots this
  float   y0;     // y value for the first kmer of the segment
  float   y1;     // y value for the last kmer of the segment
  set< pair<int,WalkPlotter::WalkType> > drawn;  // what's already been plotted.
};

  
// Plot the collection of walks
void WalkPlotter::Plot() const {

  // If we know the truth, that gets plotted last (so on top).
  if( ! truth_path.IsEmpty() ) {
    paths_to_plot.push_back( truth_path );
    path_types.push_back( TRUTH );
  }


  vec<KmerPath>::const_iterator pathp;

  // The basic unit of the picture will be the kmer.
  // The x-axis will run from 0 to max_path_len:
  int max_path_len = 0;
  for(pathp = paths_to_plot.begin(); pathp != paths_to_plot.end(); pathp++)
    max_path_len = max( max_path_len, (pathp->MinLength()+pathp->MaxLength())/2 );

  // The y-axis will indicate the value in [0,1] associated with the kmer.
  // We fix the aspect ratio by choosing some arbitrary multiplier.
  int y_axis_scale = 500;

  // Each KmerPathInterval will become a PostScript defined command i####.
  int interval_count = 0;
  map< KmerPathInterval, Done > whats_done;



  // Create the postscript file and write header
  temp_file psfile("/tmp/WalkPlot.eps_XXXXXX");
  ofstream ps( psfile.c_str() );

  ps << "%!PS-Adobe-3.0 EPSF-3.0\n"
     << "%%BoundingBox: -10 " 
     <<    min<int>(0, paths_to_plot.size() * path_offset) << " "
     <<    max_path_len+10 << " " 
     <<    y_axis_scale + max<int>( 0, paths_to_plot.size() * path_offset) << "\n"
     << "/r { 1 exch rlineto } def\n"
     << "/dot { -.3 0 rmoveto .6 0 rlineto } def" << endl;

  vec< pair< pair<int,float>,pair<int,float> > > gap_lines;

  String comment;
  uint num_plotted=0;

  // Loop over each path
  for( uint p=0; p<paths_to_plot.size(); p++ ) {

    bool wrote_something = (path_offset != 0);

    if( p >= max_path_count && 
	path_types[p] != CLOSED && path_types[p] != TRUTH )
      continue;

    if ( p > 0 && path_offset != 0 )
      ps << "0 " << path_offset << " translate\n";

    const KmerPath& path = paths_to_plot[p];
    if( path.NSegments() == 0 ) continue;

    comment = String("% path ") + ToString(p+1) + " of "
      + ToString(paths_to_plot.size()) + "\n";
//     ps << "% path " << p+1 << " of " << paths_to_plot.size() << "\n";

    int x=0;  // x coordinate = kmer # in path
    bool newpath = true, joinpath = false;  // state for line joining
    float yjoin=-1;  // if joinpath=true, the y coord we should join from

    // Loop over each segment of each path
    for( int seg=0; seg < path.NSegments(); seg++ ) {
      if( path.isGap(seg) ) {
	// Hmm, should we draw lines for gaps?
	// Keep playing with what looks good.
	if( draw_gaps ) {
	  gap_lines.push_back( make_pair(
	    make_pair(x-1,kmer_to_real(path.Stop(seg-1)) * y_axis_scale),
	    make_pair(x+path.Minimum(seg)+path.Stretch(seg)/2,
		      kmer_to_real(path.Start(seg+1)) * y_axis_scale)
	    ) );
	}
	newpath = true;
	joinpath = false;
	x += path.Minimum(seg)+path.Stretch(seg)/2;
	continue;
      }

      Done& done = whats_done[path.Segment(seg)];

      if( done.name.empty() ) {  // define a new postscript routine
	if( path.Length(seg) == 1 ) { // draw a pre-defined /dot
	  done.name = "dot";
	  done.y0 = done.y1 = kmer_to_real(path.Start(seg)) * y_axis_scale;
	}
	else {
	  if( ! wrote_something ) {
	    wrote_something = true;
	    ps << comment;
	  }
	  done.name = "i" + ToString(++interval_count);
	  longlong kmer = path.Start(seg), lastkmer = path.Stop(seg);
	  done.y0 = kmer_to_real(kmer) * y_axis_scale;
	  float yold = done.y0, ynew;
	  ps << "/" << done.name << " { % " << path.Segment(seg) << "\n";
	  for( kmer++; kmer <= lastkmer; kmer++ ) {
	    ynew = kmer_to_real(kmer) * y_axis_scale;
	    ps << round(100.0*(ynew - yold))/100.0 << " r ";
	    yold = ynew;
	  }
	  ps << "} def" << endl;
	  done.y1 = yold;
	}
      }

      if( done.drawn.insert(make_pair(x,path_types[p])).second == false && 
          path_offset == 0 ) {
	// if this has been drawn here already, don't redraw it
	newpath = false;
	joinpath = true;
	yjoin = done.y1;
      } else {  // we need to draw it
	if( ! wrote_something ) {
	  wrote_something = true;
	  ps << comment;
	}
	if ( newpath )
	  ps << x << " " << done.y0 << " moveto " << done.name << "\n";
	else if ( joinpath )
	  ps << x-1 << " " << yjoin << " moveto "
	     << x << " " << done.y0 << " lineto " << done.name << "\n";
	else
	  ps << x << " " << done.y0 << " lineto " << done.name << "\n";
	newpath = joinpath = false;
      }
      x += path.Length(seg);
    }

    // If we're going to print lines for gaps, do it first,
    // so they are under the present path:
    if( !gap_lines.empty() ) {
      // We don't cache drawing gaps, so these may be repeated
      if( ! wrote_something ) {
	wrote_something = true;
	ps << comment;
      }
      ps << "gsave newpath % begin lines for gaps\n";
      for(uint i=0; i<gap_lines.size(); i++) {
	ps << gap_lines[i].first.first << " "
	   << gap_lines[i].first.second << " moveto "
	   << gap_lines[i].second.first << " "
	   << gap_lines[i].second.second << " lineto\n";
      }
      ps << TypeLightColor[path_types[p]]
	 << " .25 setlinewidth stroke grestore % end lines for gaps\n";
      gap_lines.clear();
    }

    // color and stroke it
    if( wrote_something ) {
      ps << TypeColor[path_types[p]] << " % " 
	 << TypeName[path_types[p]] << "\n"
	 << "stroke" << endl;
      num_plotted++;
    }
  } // Done looping over paths

  if( ! truth_path.IsEmpty() ) {
    paths_to_plot.pop_back();
    path_types.pop_back();
  }

  if( num_plotted > max_path_count ) {
    cout << "Over " << max_path_count << " paths to plot; truncating" << endl;
    ps << "% Over " << max_path_count << " paths to plot; truncating" << endl;
  }
  
  ps.close();

  // Display the postscript file -- no antialiasing, for speed
  int exitcode = Csh( "gv -scale " + ToString( zoom_level ) + " -noantialias " + psfile
		     + " ; rm " + psfile );
  // Using Csh means ^C will generate a fatal error to the caller.
  // This isn't the best behavior, but it's better than nothing.

  // It's always possible we'll get here somehow...
  if ( exitcode != 0 ) {
    if( IsRegularFile(psfile) ) Remove(psfile);
    cout << "gv exited abnormally; press a key to continue..." << flush;
    cin.get();
    cout << endl;
  }

}
