/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef UNIPATH_H
#define UNIPATH_H

// This define Unipath(...).  See the documentation in the main program
// Unipather.cc.  This file defines a callable module.  Output is:
// - unipaths (a vecKmerPath);
// - unipathsdb (a vec<TAGGED_RPINT>, which is some sort of *_tagged_rpint).
// - unipaths_file (optional file of unipaths);
// If unipaths_file is unspecified, then it is not generated.  It is also used as a 
// scratch file, and if specified, performance may be better.

#include "CoreTools.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerPath.h"
template<class TAGGED_RPINT>
void Unipath( 
     /* inputs: */  const vecKmerPath& paths, const vecKmerPath& paths_rc,
                    const vec<TAGGED_RPINT>& pathsdb, 
     /* outputs: */ vecKmerPath& unipaths, vec<TAGGED_RPINT>& unipathsdb, 
                    Bool log_progress = False, const String& unipaths_file = "",
                    bool verbose = false );

// The implementation for Unipath is in Unipath.cc, and it must be
// explicitly instaintiated there for each TAGGED_RPINT type.

// UnipathDestructive is just like Unipath, except that it
// destroys its arguments when they are no longer needed, 
// to fit large jobs in memory.
template<class TAGGED_RPINT>
void UnipathDestructive( 
     /* inputs: */  vecKmerPath& paths, vecKmerPath& paths_rc,
                    vec<TAGGED_RPINT>& pathsdb, 
     /* outputs: */ vecKmerPath& unipaths, vec<TAGGED_RPINT>& unipathsdb, 
                    Bool log_progress = False, const String& unipaths_file = "",
                    bool verbose = false );

// DecomposePath: given a KmerPath x, break it up into unipaths, and express x
// in terms of those.

void DecomposePath( const KmerPath& x, Bool brief = False );

// BuildUnipathAdjacencyGraph.  If unipaths have already been computed, build the
// graph whose nodes are the unipaths and whose edges are defined by adjacent kmers.

template<class TAGGED_RPINT>
void BuildUnipathAdjacencyGraph(
     /* inputs: */ const vecKmerPath& paths, 
                   const vecKmerPath& paths_rc,
                   const vec<TAGGED_RPINT>& pathsdb, 
                   const vecKmerPath& unipaths,
                   const vec<TAGGED_RPINT>& unipathsdb, 
     /* output: */ digraph& A );

// BuildUnipathAdjacencyHyperKmerPath.  Take the output of 
// BuildUnipathAdjacencyGraph, and turn it into a HyperKmerPath (changing vertices
// to edges).

void BuildUnipathAdjacencyHyperKmerPath(
     int K, const digraph& A, const vecKmerPath& unipaths, HyperKmerPath& h );

// UnipathInvolution.  Define map that takes a unipath to its reverse complement.

template<class TAGGED_RPINT>
void UnipathInvolution( const vecKmerPath& unipaths, 
     const vec<TAGGED_RPINT>& unipathsdb, vec<int>& to_rc );

void PrintInColumns( const vec<String>& s );




///////////////////////////////////////////////////////////////////////
//
// Implementations for functions templatized over TAGGED_RPINT classes
//
///////////////////////////////////////////////////////////////////////

template<class TAGGED_RPINT>
void BuildUnipathAdjacencyGraph( const vecKmerPath& paths, 
     const vecKmerPath& paths_rc, const vec<TAGGED_RPINT>& pathsdb, 
     const vecKmerPath& unipaths, const vec<TAGGED_RPINT>& unipathsdb, 
     digraph& A )
{    int nuni = unipaths.size( );
     A.Clear( );
     A.ToMutable( ).resize(nuni), A.FromMutable( ).resize(nuni);
     vec<longlong> places, before, after;
     for ( int i = 0; i < nuni; i++ )
     {    const KmerPath& u = unipaths[i];
          longlong left = u.FirstSegment( ).Start( );
          longlong right = u.LastSegment( ).Stop( );
          Contains( pathsdb, left, places );
          before.clear( ), after.clear( );
          for ( int j = 0; j < places.isize( ); j++ )
          {    const TAGGED_RPINT& t = pathsdb[ places[j] ];
               int id = t.PathId( );
               const KmerPath& p = ( id >= 0 ? paths[id] : paths_rc[-id-1] );
               int seg = t.PathPos( ), pos = left - t.Start( );
               if ( pos == 0 && seg == 0 ) continue;
               if ( pos > 0 ) --pos;
               else
               {    --seg;
                    pos = p.Length(seg) - 1;    }
               before.push_back( p.Segment(seg).Start( ) + pos );    }
          Contains( pathsdb, right, places );
          for ( int j = 0; j < places.isize( ); j++ )
          {    const TAGGED_RPINT& t = pathsdb[ places[j] ];
               int id = t.PathId( );
               const KmerPath& p = ( id >= 0 ? paths[id] : paths_rc[-id-1] );
               int seg = t.PathPos( ), pos = right - t.Start( );
               if ( pos == p.Length(seg) - 1 && seg == p.NSegments( ) - 1 ) continue;
               if ( pos < p.Length(seg) - 1 ) ++pos;
               else
               {    ++seg;
                    pos = 0;    }
               after.push_back( p.Segment(seg).Start( ) + pos );    }
          UniqueSort(before), UniqueSort(after);
          ForceAssertLe( before.size( ), 4 );
          ForceAssertLe( after.size( ), 4 );
          for ( int j = 0; j < before.isize( ); j++ )
          {    Contains( unipathsdb, before[j], places );
               if ( places.size( ) != 1 ) continue;
               A.ToMutable( )[i].push_back( 
                    unipathsdb[ places[0] ].PathId( ) );    }
          for ( int j = 0; j < after.isize( ); j++ )
          {    Contains( unipathsdb, after[j], places );
               if ( places.size( ) != 1 ) continue;
               A.FromMutable( )[i].push_back( 
                    unipathsdb[ places[0] ].PathId( ) );    }    }    
     for ( int i = 0; i < nuni; i++ )
     {    Sort( A.FromMutable(i) );
          Sort( A.ToMutable(i) );    }    }

template<class TAGGED_RPINT>
void UnipathInvolution( const vecKmerPath& unipaths, 
     const vec<TAGGED_RPINT>& unipathsdb, vec<int>& to_rc )
{    to_rc.resize( unipaths.size( ) );
     for ( int i = 0; i < unipaths.size( ); i++ )
     {    KmerPath u = unipaths[i];
          u.Reverse( );
          static vec<longlong> locs;
          Contains( unipathsdb, u.Segment(0), locs );
          to_rc[i] = unipathsdb[ locs[0] ].PathId( );    }    }




#endif
