// Copyright (c) 2005 Broad Institute/Massachusetts Institute of Technology

#include "paths/SubMuxGraphBuilder.h"
#include "paths/BigPicture.h"
#include "TaskTimer.h"

void SubMuxGraphBuilder::BuildFromPaths( const vecKmerPath& pathsFw, 
					 const vecKmerPath& pathsRc,
                                         const int partition ) {
  DeleteStuff();

  TaskTimer timer;
  if( m_verbose ) timer.Start();

  // Sanity-check data and record read lengths.
  ForceAssertEq( pathsFw.size(), pathsRc.size() );
  m_readLengthsInKmers = vec<int>(pathsFw.size());

  for( int i=0; i < pathsFw.size(); i++ )
    m_readLengthsInKmers[i] = pathsFw[i].MinLength();
  int len;
  for( int i=0; i < pathsRc.size(); i++ )
    if( (len = pathsRc[i].MinLength()) != 0 ) {
      if( m_readLengthsInKmers[i] == 0 )
	m_readLengthsInKmers[i] = len;
      else if( m_readLengthsInKmers[i] != len )
	// Error out: Fw and Rc paths disagree
	ForceAssertEq( pathsFw[i].MinLength(), 
		       pathsRc[i].MinLength() );
    }

  if( m_verbose ) {
    timer.Stop();
    cout << "path lengths: " << timer << endl;
    timer.Reset();
    timer.Start();
  }

  // Build the needed objects:
  mp_pathsDb = new KmerPathDatabase( pathsFw, pathsRc );
  
  if( m_verbose ) {
    timer.Stop();
    cout << "path database: " << timer << endl;
    timer.Reset();
    timer.Start();
  }

  MuxFinder finder( pathsFw, pathsRc, *mp_pathsDb, &cout );
  finder.StudyReads();
  finder.SetMinNumKmers( m_minNumKmers );
  finder.SetMinOverlap( m_minOverlap );
  mp_muxGraph = new MuxGraph(pathsFw.size());
  finder.FindMuxes( *mp_muxGraph, partition );

  if( m_verbose ) {
    timer.Stop();
    cout << "FindMuxes: " << timer << endl;
    timer.Reset();
    timer.Start();
  }

  mp_subList = new SubsumptionList( pathsFw.size() );
  const int chunkSize = 100000;
  ostream* nolog = 0;
  FindSubsumedReads( *mp_subList, pathsFw, pathsRc, *mp_pathsDb, 
                     m_minNumKmers, chunkSize, &cout, partition );

  if( m_verbose ) {
    timer.Stop();
    cout << "Find subsumptions: " << timer << endl;
  }

  mp_searcher = new KmerPathMuxSearcher( mp_muxGraph,
					 mp_fillDb,
					 mp_subList,
					 &m_readLengthsInKmers );
}

void SubMuxGraphBuilder::BuildFromSomePaths( vecKmerPath& pathsFw, 
					     vecKmerPath& pathsRc,
					     const vec<int>& subsetFw,
					     const vec<int>& subsetRc ) {
  TaskTimer timer;
  if( m_verbose ) timer.Start();

  // Make sub-base-vectors with only those reads.

  vecKmerPath pathsFw_restricted(pathsFw.size());
  vecKmerPath pathsRc_restricted(pathsRc.size());

  // The commented-out lines use SetExternal to make the subsets
  // without copying any of the path data, just the pointer to it.
  // This is (1) dangerous, because the copy can modify the original,
  // and (2) unavailable, since KmerPath inherits privately from
  // serfvec, so methods like StartOfDynamicData() aren't allowed.
  // As it happens, even with the copy-free version this takes much
  // too long, so it's not worth mucking about with Feudal to make
  // this possible (even if inadvisable) in general.  If you want
  // to play with it temporarily, switch to the commented-out version,
  // and make KmerPath inherit publicly

//   for( vec<int>::const_iterator i = subsetFw.begin(); 
//        i != subsetFw.end(); i++ ) {
//       KmerPath& read = pathsFw[*i];
//       pathsFw_restricted[*i].SetExternal( read.StartOfStaticData(), 
// 					  read.StartOfDynamicData(),
// 					  read.SizeOfDynamicData(), 0 );
//   }
//   for( vec<int>::const_iterator i = subsetRc.begin(); 
//        i != subsetRc.end(); i++ ) {
//       KmerPath& read = pathsRc[*i];
//       pathsRc_restricted[*i].SetExternal( read.StartOfStaticData(), 
// 					  read.StartOfDynamicData(),
// 					  read.SizeOfDynamicData(), 0 );
//   }


  // Instead, let's just dumb-copy everything:

  for( vec<int>::const_iterator i = subsetFw.begin(); i != subsetFw.end(); i++ )
    pathsFw_restricted[*i] = pathsFw[*i];
  for( vec<int>::const_iterator i = subsetRc.begin(); i != subsetRc.end(); i++ )
    pathsRc_restricted[*i] = pathsRc[*i];

  if( m_verbose ) {
    timer.Stop();
    cout << "take subsets: " << timer << endl;
  }

  // Now call the normal BuildFromPaths
  this->BuildFromPaths( pathsFw_restricted, pathsRc_restricted );
}

  

void SubMuxGraphBuilder::BigPictureDot( const String& filename, const int partition ) const
{
  PrintBigPictureDot( filename, *mp_muxGraph, *mp_subList, partition );
}
