/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// ReadsToPathsCore.  See documentation in ReadsToPaths.cc.
// See also ReadsToPathsCoreX.cc, which is very similar to this file.

#ifndef FORCE_DEBUG
    #define NDEBUG
#endif

#include <map>

#include "pairwise_aligners/BalancedMutmerGraph.h"
#include "Basevector.h"
#include "Bitvector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "FeudalMimic.h"
#include "math/Functions.h"
#include "KmerRecord.h"
#include "pairwise_aligners/Mutmer.h"
#include "Qualvector.h"
#include "SortKmers.h"
#include "paths/KmerPath.h"
#include "paths/ReadsToPathsCore.h"

template<class POS> class mhit {

     public:

     mhit( ) { }
     mhit( int id1, int id2, Bool rc, POS pos1, POS pos2, POS len )
          : id1_(id1), id2_(id2), rc_(rc), pos1_(pos1), pos2_(pos2), len_(len) { }

     void Set( int id1, int id2, Bool rc, POS pos1, POS pos2, POS len )
     {    id1_ = id1;
          id2_ = id2;
          rc_ = rc;
          pos1_ = pos1;
          pos2_ = pos2;
          len_ = len;    }

     int Id1( ) const { return id1_; }
     int Id2( ) const { return id2_; }
     POS pos1( ) const { return pos1_; }
     POS pos2( ) const { return pos2_; }
     POS Len( ) const { return len_; }
     Bool Rc2( ) const { return rc_; }

     void SetId1( int id1 ) { id1_ = id1; }
     void SetId2( int id2 ) { id2_ = id2; }
     void SetPos1( int pos1 ) { pos1_ = pos1; }
     void SetPos2( int pos2 ) { pos2_ = pos2; }

     friend Bool operator<( const mhit& h1, const mhit& h2 )
     {    if ( h1.Id2( ) < h2.Id2( ) ) return True;
          if ( h1.Id2( ) > h2.Id2( ) ) return False;
          return h1.Len( ) > h2.Len( );    }
          
     private:

     int id1_, id2_;
     Bool rc_;
     POS pos1_, pos2_;
     POS len_;

};

template<int K, int I> 
     inline void GetNextKmerPairPaths( vector< kmer_record<K,I> >& R, int S, 
     int& read_id1, int& read_id2, int& pos1, int& pos2, vecbitvector& chosen,
     vec<longlong>& multiplicity )
{    static int i, j, r2, xpos1;
     if ( read_id1 >= 0 ) goto after_return;
     for ( i = 0; i < S; i++ )
     {    
          // Get batch of k-mer records corresponding to one k-mer.

          int l;
          for ( j = i+1; j < S; j++ )
          {    for ( l = (K+3)/4 - 1; l >= 0; l-- )
                    if ( R[j].Bytes( )[l] != R[i].Bytes( )[l] ) break;
               if ( l >= 0 ) break;    }
          sort( R.begin( ) + i, R.begin( ) + j, kmer_record<K,I>::id_cmp_pos );
          read_id1 = R[i].GetId( );
          pos1 = R[i].GetPos( );
          ++multiplicity[ Min( 1000, j-i ) ];

          // Mark canonical placement of k-mer.

          xpos1 = pos1;
          if ( pos1 < 0 ) xpos1 = -pos1;
          --xpos1;
          chosen[read_id1].Set( xpos1, True );

          // Go through pairs.

          for ( r2 = i+1; r2 < j; r2++ )
          {    read_id2 = R[r2].GetId( );
               pos2 = R[r2].GetPos( );
               return;
               after_return: continue;    }
          i = j - 1;    }    
     read_id1 = -1;    }

template<int I, int k, int BLOCKS_PER_NODE, class POS> void MakeAlignsPaths(
     const vecbasevector& EE, String aligns_file, ostream& log, 
     longlong& COUNT, vecbitvector& chosen, String head, Bool paths_only,
     ostream& out )
{
     int N = EE.size( );
     mutmer_graph<I, BLOCKS_PER_NODE> M(N);

     vec<int> rid(N);
     for ( int i = 0; i < N; i++ )
          rid[i] = i;
     longlong S_init = 0;
     for ( int l = 0; l < EE.size( ); l++ )
          if ( EE[l].size( ) >= k ) S_init += EE[l].size( ) - k;
     S_init += S_init/4;
     S_init /= 33;

     if ( S_init > 3000000000u )
          FatalErr( "MakeAlignsPaths: the value of S is " << S_init
               << ".  This is dangerously large, because it represents\nthe "
               << "size of a vector which is to be stored in an unsigned int,\n"
               << "and this vector could grow. "
               << "You may need to rewrite some code to get it to work." );

     unsigned int S = S_init, total_S = 0;

     vec< kmer_record<k,I> > R(S);
     vec<longlong> multiplicity( 1001, 0 );
     out << "pass " << flush;
     for ( int pass = 0; pass < 100; pass++ )
     {    dummy<100> d100;
          SortKmers( d100, EE, rid, pass, R, S );
          ForceAssertGe( R.size( ), S );
          total_S += S;
          int read_id1(-1), read_id2, pos1(-1), pos2;
          while(1)
          {    
               // Get next k-mer pair.

               GetNextKmerPairPaths<k,I>( R, S, read_id1, read_id2, pos1, pos2,
                    chosen, multiplicity );
               if ( read_id1 < 0 ) break;

               // Merge into mutmer graph.

               M.MergeKmer( pos1, pos2, k, read_id1, read_id2, 
                    EE[read_id1], EE[read_id2], True );    }    
          if ( pass == 97 ) out << "1";
          else if ( pass % 10 == 8 ) out << (pass/10 + 1) % 10;
          else if ( pass % 10 == 9 ) out << "0 ";
          else out << ".";
          flush(out);    }
     out << "\n" << flush;
     if ( !paths_only )
          BinaryWrite2( head + ".paths.mult.k" + ToString(k), multiplicity );

     if ( total_S == 0 )
     {    out << "MakeAlignsPaths: I've been passed reads of length";
          for ( int l = 0; l < EE.size( ); l++ )
               out << " " << EE[l].size( );
          out << "\nI can't work with this data.\n";
          ForceAssert( 0 == 1 );    }

     // Generate file containing mutmers:

     PipeOstream( aligns_out, aligns_file.Before( ".gz" ) );
     vec< mutmer_read_id<I> > mid( Max( M.Counts( ) ) );
     int j, id2_last = -1;
     for ( int l = 0; l < N; l++ ) 
     {    int count = M.All(l, mid);
          sort( mid.begin( ), mid.begin( ) + count );
          for ( int i = 0; i < count; i++ ) 
          {    for ( j = i+1; j < count; j++ ) 
               {    if ( mid[i].ReadIdRc( ) != mid[j].ReadIdRc( ) ) break;    }
               Bool RC = mid[i].Rc( );
               int id1 = l, id2 = mid[i].ReadId( );
               COUNT += j - i;
               for ( int r = 0; r < j - i; r++ ) 
               {    int pos1, pos2, len, e;
	            mid[i+r].Unpack( pos1, pos2, len, e );
                    static mhit<POS> h;
                    if ( id1 < id2 ) h.Set( id1, id2, RC, pos1, pos2, len );
                    else if ( !RC ) h.Set( id2, id1, RC, pos2, pos1, len );
                    else
                    {    int n1 = EE[id1].size( ), n2 = EE[id2].size( );
                         h.Set( id2, id1, RC, n2-pos2-len, n1-pos1-len, len );    }
                    BinWrite( aligns_out, h );    }
               i = j - 1;    }    }    }

template<class POS> void ReadsToPathsMainPart( longlong COUNT, 
     const String& mutmers_file, const int N, const vec< pair<int,int> >& q40id,
     const vec<int>& broken, const vec<int>& broken_back, const int longread,
     const unsigned int& K, vecbasevector& EE, const String& head,
     vecbitvector& chosen, vecbitvector& chosen_rc, const String& HEAD,
     ostream& clogr, map<longlong,longlong>& palindromic_id, longlong npalindromes,
     const Bool VERBOSE, const Bool PATHS_ONLY, const Bool BIGREADS,
     const int blocks_per_node, ostream& out );

void ReadsToPathsCore( const String& PRE, const String& DATA, const String& RUN,
     const String& HEAD, const unsigned int& K, const Bool& VERBOSE,
     const Bool& BREAK_LONG, const String& GENOME_SIZE, 
     const Bool& USE_QUALITY_SCORES, const Bool& PATHS_ONLY, ostream& out )
{
     // Check command-line arguments, set up paths, set up logging.

     ForceAssertSupportedK( K );
     if ( HEAD == "" ) ForceAssert( DATA != "" && RUN != "" );
     if ( HEAD != "" ) ForceAssert( DATA == "" && RUN == "" && GENOME_SIZE != "" );
     String predata = PRE + "/" + DATA;
     String run_dir = PRE + "/" + DATA + "/" + RUN;
     String head = ( HEAD == "" ? run_dir + "/reads" : HEAD );
     String logpath = run_dir + "/ReadsToPaths.log";
     Ofstream( logstream, ( HEAD == "" ? logpath : "/dev/null" ) );
     ostream& clogr = ( HEAD == "" ? logstream : out );

     // Set up tracking of palindromes.

     longlong npalindromes = 0;
     map<longlong,longlong> palindromic_id;

     // Bring in the reads.  If there are reads of length >= 1024, break them in 
     // two to save memory later.  However, if there are reads of length ~2000 or 
     // larger, don't do anything.  
  
     const int longread = 1024;
     vecbasevector EE( head + ".fastb" );
     int N = EE.size( );

     Bool short_too_small = False;
     for ( int i = 0; i < N; i++ )
          if ( EE[i].size( ) > SHRT_MAX ) short_too_small = True;
     vec<int> broken( N, -1 ), broken_back;
     Bool BIGREADS = False;
     if ( !BREAK_LONG )
     {    for ( int i = 0; i < N; i++ )
               if ( (int) EE[i].size( ) >= 1024 ) BIGREADS = True;    }
     else
     {    for ( int i = 0; i < N; i++ )
               if ( (int) EE[i].size( ) >= 2048 - (int) K ) BIGREADS = True;    }
     if (BIGREADS) clogr << "using BIGREADS" << endl;
     if ( !BIGREADS )
     {    int nbroken = 0, extra_bases = 0;
          for ( int i = 0; i < N; i++ )
          {    if ( (int) EE[i].size( ) >= longread )
               {    ++nbroken;
                    extra_bases 
                         += (int) EE[i].size( ) - longread + (int) K;    }    }
          EE.Reserve( 
               EE.rawsize( ) + extra_bases/16 + nbroken, EE.size( ) + nbroken );
          for ( int i = 0; i < N; i++ )
          {    if ( (int) EE[i].size( ) >= longread )
               {    broken[i] = EE.size( );
                    static basevector part1, part2;
                    part1.SetToSubOf( EE[i], 0, longread - 1 );
                    part2.SetToSubOf( EE[i], longread - (int) K, 
                         (int) EE[i].size( ) - ( longread - (int) K ) );
                    broken_back.push_back(i);
                    EE[i] = part1;
                    EE.push_back(part2);    }    }    }
     int NPLUS = EE.size( );

     // Place reads in descending order by number of quality 40+ bases.  For now
     // we ignore the splitting of reads which occurred in the previous step.  The
     // vector q40id remembers the new order, which is reversed later.

     vec< pair<int,int> > q40id(N);
     if (USE_QUALITY_SCORES)
     {    vecqualvector Qpart;
          Qpart.Reserve( 1000000, 1000 );
          for ( int x = 0; x < N; x += 1000 )
          {    Qpart.clear( );
               int y = Min( x + 1000, N );
               Qpart.ReadRange( head + ".qualb", x, y );
               for ( int i = x; i < y; i++ )
               {    int count = 0;
                    for ( int j = 0; j < Qpart[i-x].size( ); j++ )
                         if ( Qpart[i-x][j] >= 40 ) ++count;
                    q40id[i] = make_pair( count, i );    }    }    }
     else
     {    for ( int i = 0; i < N; i++ )
               q40id[i] = make_pair( EE[i].size( ), i );    }
     ReverseSort(q40id);
     {    vec<int> new_order(N), new_map(N);
          for ( int i = 0; i < N; i++ )
               new_order[i] = q40id[i].second;
          for ( int i = 0; i < N; i++ )
               new_map[ new_order[i] ] = i;
          for ( int i = 0; i < N; i++ )
          {    if ( i == new_order[i] ) continue;
               swap( EE[i], EE[ new_order[i] ] );
               int saved_new_order_i = new_order[i];
               swap( new_order[i], new_order[ new_map[i] ] );
               swap( new_map[i], new_map[saved_new_order_i] );    }    }

     // Remove old output file.

     String mutmers_file;
     if ( HEAD == "" ) mutmers_file = run_dir + "/mutmers.k" + ToString(K) + ".gz";
     else mutmers_file = HEAD + ".mutmers.k" + ToString(K) + ".gz";
     Remove(mutmers_file);

     // Get genome size.

     longlong genome_size;
     if ( GENOME_SIZE == "" ) 
          genome_size = StringOfFile( predata + "/genome.size", 1 ).Int( );
     else genome_size = GENOME_SIZE.Int( );
     ForceAssert( genome_size > 0 );

     // Go through one pass.

     int blocks_per_node = 50;
     longlong kmercount = 0;
     for ( int i = 0; i < NPLUS; i++ )
       if ( EE[i].size() >= K )
         kmercount += EE[i].size() - K + 1;

     float coverage = float(kmercount) / float(genome_size);
     // PRINT3( kmercount, genome_size, coverage );
     if ( coverage < 10.0 ) blocks_per_node = int(floor(2.5 * coverage));
     // PRINT( blocks_per_node );

     longlong COUNT = 0;
     vecbitvector chosen, chosen_rc;
     Mimic( EE, chosen);

     if ( !short_too_small )
     {    ReadsToPathsMainPart<short>( COUNT, mutmers_file, N, q40id, broken, 
               broken_back, longread, K, EE, head, chosen, chosen_rc, HEAD, clogr,
               palindromic_id, npalindromes, VERBOSE, PATHS_ONLY, BIGREADS,
               blocks_per_node, out );    }
     else
     {    ReadsToPathsMainPart<int>( COUNT, mutmers_file, N, q40id, broken, 
               broken_back, longread, K, EE, head, chosen, chosen_rc, HEAD, clogr,
               palindromic_id, npalindromes, VERBOSE, PATHS_ONLY, BIGREADS,
               blocks_per_node, out );    }    }

template<class POS> void ReadsToPathsMainPart( longlong COUNT, 
     const String& mutmers_file, const int N, const vec< pair<int,int> >& q40id,
     const vec<int>& broken, const vec<int>& broken_back, const int longread,
     const unsigned int& K, vecbasevector& EE, const String& head,
     vecbitvector& chosen, vecbitvector& chosen_rc, const String& HEAD,
     ostream& clogr, map<longlong,longlong>& palindromic_id, longlong npalindromes,
     const Bool VERBOSE, const Bool PATHS_ONLY, const Bool BIGREADS,
     const int blocks_per_node, ostream& out )
{
     #define CALL_MAKE_ALIGNS( A, B, C )                                          \
     {    MakeAlignsPaths<A, B, C, POS>( EE, mutmers_file, clogr, COUNT, chosen,  \
              head, PATHS_ONLY, out );    }

     #define CASE(_K)                                                             \
     {    if ( !BIGREADS )                                                        \
          {    if ( blocks_per_node <= 12 ) CALL_MAKE_ALIGNS( 1, _K, 6 )          \
               else if ( blocks_per_node <= 24 ) CALL_MAKE_ALIGNS( 1, _K, 12 )    \
               else CALL_MAKE_ALIGNS( 1, _K, 50 );    }                           \
          else                                                                    \
          {    if ( blocks_per_node <= 12 ) CALL_MAKE_ALIGNS( 2, _K, 6 )          \
               else if ( blocks_per_node <= 24 ) CALL_MAKE_ALIGNS( 2, _K, 12 )    \
               else CALL_MAKE_ALIGNS( 2, _K, 50 );    }    }

     DISPATCH_ON_K(K, CASE);

     out << "\n" << COUNT << " mutmers generated\n";
     out << Date( ) << ": generated " << mutmers_file << ", size = " 
          << setprecision(5)
          << float(FileSize(mutmers_file)) / 1000000.0 << " MB." << endl << endl;
     if ( HEAD == "" )
     {    clogr << "\n" << COUNT << " mutmers generated\n";
          clogr << Date( ) << ": generated " << mutmers_file << ", size = " 
               << setprecision(5) << float(FileSize(mutmers_file)) / 1000000.0 
               << " MB." << endl << endl;    }

     // Read mutmers back in, remove file, undo chopping and sorting of reads, 
     // sort mutmers.

     vec< mhit<POS> > mhits;
     mhits.reserve(COUNT);
     {    PipeIstream( hin, mutmers_file.Before( ".gz" ) );
          mhit<POS> mh;
          while(1)
          {    BinRead( hin, mh );
               if ( !hin ) break;
               mhits.push_back(mh);    }    }
     Remove(mutmers_file);
     {    vec<int> new_order(N);
          for ( int i = 0; i < N; i++ )
               new_order[i] = q40id[i].second;
          for ( int i = 0; i < mhits.isize( ); i++ )
          {    if ( mhits[i].Id1( ) < N )
                    mhits[i].SetId1( new_order[ mhits[i].Id1( ) ] );
               if ( mhits[i].Id2( ) < N )
                    mhits[i].SetId2( new_order[ mhits[i].Id2( ) ] );    }
          for ( int i = 0; i < mhits.isize( ); i++ )
          {    int id1 = mhits[i].Id1( ), id2 = mhits[i].Id2( );
               if ( id1 >= N )
               {    mhits[i].SetId1( broken_back[ id1 - N ] );
                    mhits[i].SetPos1( mhits[i].pos1( ) + longread - (int) K );    }
               if ( id2 < N && broken[id2] >= 0 && mhits[i].Rc2( ) )
               {    mhits[i].SetPos2( mhits[i].pos2( ) 
                         + (int) EE[ broken[id2] ].size( ) - (int) K + 1 );    }
               if ( id2 >= N )
               {    mhits[i].SetId2( broken_back[ id2 - N ] );
                    if ( !mhits[i].Rc2( ) )
                         mhits[i].SetPos2( 
                              mhits[i].pos2( ) + longread - (int) K );    }    }
          EE.clear( );
          EE.ReadAll( head + ".fastb" );
          vecbitvector chosen2;
          Mimic( EE, chosen2 );
          for ( int i = 0; i < N; i++ )
               chosen2[ new_order[i] ] = chosen[i];
          for ( int i = 0; i < N; i++ )
          {    if ( broken[i] >= 0 )
               {    int ip = broken[i];
                    // chosen2[i].resize( EE[i].size( ) ); // not supported now, so:
                    static bitvector b;
                    b.Setsize( EE[i].size( ) );
                    for ( int j = 0; j < (int) chosen2[i].size( ); j++ )
                         b.Set( j, chosen2[i][j] );
                    chosen2[i] = b;
                    for ( int j = 0; j < (int) chosen[ip].size( ); j++ )
                         chosen2[i].Set( 
                              longread + j - (int) K, chosen[ip][j] );    }    }
          chosen = chosen2;    }
     Sort(mhits);

     // If chosen[i](j) is set, set chosen_rc[i](j), unless the k-mer is
     // palindromic.

     Mimic( EE, chosen_rc );
     int iK = K;
     for ( int i = 0; i < chosen.size( ); i++ )
     {    for ( int j = 0; j < (int) chosen[i].size( ); j++ )
          {    if ( !chosen[i][j] ) continue;
               const basevector& b = EE[i];
               int x;
               for ( x = 0; x < (int) K/2; x++ )
                    if ( b[j+x] + b[j+iK-x-1] != 3 ) break;
               if ( x == (int) K/2 ) continue;
               chosen_rc[i].Set( j, True );    }    }

     // Number positions on the reads.

     longlong kx = 0;
     vec<longlong> start( EE.size( ) );
     for ( int i = 0; i < EE.size( ); i++ )
     {    start[i] = kx;
          kx += EE[i].size( );    }
     if ( kx > halfway_kmer )
     {    FatalErr( "Five byte limit on k-mer indices exceeded." );    }

     // Compute average basic path length.

     vec<int> lengths;
     for ( int i = 0; i < chosen.size( ); i++ )
     {    for ( int j = 0; j < (int) chosen[i].size( ); j++ )
          {    if ( !chosen[i][j] ) continue;
               int k;
               for ( k = j + 1; k < (int) chosen[i].size( ); k++ )
                    if ( !chosen[i][k] ) break;
               lengths.push_back( k - j );
               j = k - 1;    }    }
     for ( int i = 0; i < chosen_rc.size( ); i++ )
     {    for ( int j = 0; j < (int) chosen_rc[i].size( ); j++ )
          {    if ( !chosen_rc[i][j] ) continue;
               int k;
               for ( k = j + 1; k < (int) chosen_rc[i].size( ); k++ )
                    if ( !chosen_rc[i][k] ) break;
               lengths.push_back( k - j );
               j = k - 1;    }    }
     out << "average basic path length = " 
          << BigSum(lengths) / lengths.size( ) << endl;
     if ( HEAD == "" )
     {    clogr << "average basic path length = " 
               << BigSum(lengths) / lengths.size( ) << endl;    }

     // Build read paths for the reads themselves.

     vec<KmerPath> paths;
     paths.reserve(N);
     longlong nsegments = 0;
     int i = 0;
     for ( int id2 = 0; id2 < N; id2++ )
     {    int j;
          for ( j = i; j < mhits.isize( ); j++ )
               if ( mhits[j].Id2( ) != id2 ) break;
          static vec<longlong> index;
          int n2 = EE[id2].size( );
          index.resize_and_set( Max( 0, n2 - (int) K + 1 ), -1 );
          for ( int u = 0; u <= n2 - (int) K; u++ )
               if ( chosen[id2][u] ) index[u] = halfway_kmer + start[id2] + u + 1;

          for ( int k = i; k < j; k++ )
          {    int id1 = mhits[k].Id1( );
               int pos1 = mhits[k].pos1( ), pos2 = mhits[k].pos2( );
               int len = mhits[k].Len( );
               Bool rc2 = mhits[k].Rc2( );

               // Validate mutmer.

               #ifndef NDEBUG
               {    for ( int u = 0; u < len; u++ )
                    {    if ( !rc2 )
                         {    ForceAssertEq( EE[id1][pos1+u], EE[id2][pos2+u] );    }
                         else
                         {    ForceAssertEq( 3-EE[id1][pos1+u], 
                                   EE[id2][n2-pos2-u-1] );    }    }    }
               #endif

               // Find correspondence.

               if ( !rc2 )
               {    for ( int u = 0; u <= len - (int) K; u++ )
                    {    if ( !chosen[id1][pos1+u] || index[pos2+u] >= 0 ) continue;
                         index[pos2+u] 
                              = halfway_kmer + start[id1] + pos1 + u + 1;    }    }
               else
               {    for ( int u = (int) K - 1; u < len; u++ )
                    {    int up = u - (int) K + 1;
                         if ( !chosen[id1][ pos1 + up ] ) continue;
                         if ( index[n2-pos2-u-1] >= 0 ) continue;
                         if ( chosen_rc[id1][ pos1 + u - (int) K + 1 ] )
                              index[n2-pos2-u-1] 
                                   = halfway_kmer - start[id1] - pos1 - up;
                         else
                              index[n2-pos2-u-1]
                                   = halfway_kmer + start[id1] 
                                        + pos1 + up + 1;    }    }    }

          // Check for palindromes.

          for ( int u = 0; u < index.isize( ); u++ )
          {    const basevector& rd = EE[id2];
               int x;
               for ( x = 0; x < (int) K/2; x++ )
                    if ( rd[u+x] + rd[u+iK-x-1] != 3 ) break;
               if ( x == (int) K/2 )
               {    longlong pid = palindromic_id[ index[u] ];
                    if ( pid == 0 )
                    {    ForceAssertLt( npalindromes, max_palindromes );
                         pid = first_palindrome + npalindromes;
                         palindromic_id[ index[u] ] = pid;
                         ++npalindromes;    }
                    index[u] = pid;     }     }

          KmerPath p;
          if (VERBOSE) out << "\n" << id2 << ":";
          int sc = 0;
          for ( int u = 0; u <= n2 - (int) K; u++ )
          {    int v;
               for ( v = u + 1; v <= n2 - (int) K; v++ )
                    if ( index[v] != index[v-1] + 1 ) break;
               ++nsegments;
               if ( VERBOSE && ++sc == 6 )
               {    sc = 1;
                    out << "\n ";
                    String s = ToString(id2);
                    for ( int y = 0; y < (int) s.size( ); y++ )
                         out << " ";    }
               p.AddSegment( index[u], index[u] + v - u - 1 );
               if (VERBOSE) out << " " << index[u] << "(" << v - u << ")";
               u = v - 1;    }
          paths.push_back(p);
          if (VERBOSE) out << "\n";

          for ( int u = 0; u <= n2 - (int) K; u++ )
          {    if ( index[u] < 0 ) PRINT3_TO( out, id2, u, n2 );
               ForceAssert( index[u] >= 0 );    }

          i = j;    }

     // Write reads paths for reads.

     vecKmerPath pathsv;
     pathsv.Reserve( nsegments, N );
     for ( int i = 0; i < N; i++ )
          pathsv.push_back( paths[i] );
     pathsv.WriteAll( head + ".paths.k" + ToString(K) );
     if (PATHS_ONLY) return;

     // Build and write read paths for the reverse complements of the reads.

     vec<KmerPath> paths_rc;
     longlong nsegments_rc = 0;
     for ( int i = 0; i < N; i++ )
     {    static KmerPath p;
          p = paths[i];
          p.Reverse( );
          paths_rc.push_back(p);
          nsegments_rc += p.NSegments( );    }
     vecKmerPath pathsv_rc;
     pathsv_rc.Reserve( nsegments_rc, N );
     for ( int i = 0; i < N; i++ )
          pathsv_rc.push_back( paths_rc[i] );
     pathsv_rc.WriteAll( head + ".paths_rc.k" + ToString(K) );

     out << "\naverage number of segments per read path = "
          << setprecision(3) << float(nsegments)/float(N) << endl;
     clogr << "\naverage number of segments per read path = "
	   << setprecision(3) << float(nsegments)/float(N) << endl;
     out << npalindromes << " palindromic kmers seen " << endl;
     clogr << npalindromes << " palindromic kmers seen " << endl;

     // Build paths database.
     // At this point paths have no gaps in them, so nothing needs to be skipped.

     Destroy(paths), Destroy(paths_rc);
     vec<tagged_rpint> segs;
     segs.reserve( pathsv.size( ) + pathsv_rc.size( ) );
     for ( int i = 0; i < pathsv.size( ); i++ )
          pathsv[i].AppendToDatabase( segs, i );
     for ( int i = 0; i < pathsv_rc.size( ); i++ )
          pathsv_rc[i].AppendToDatabase( segs, -i-1 );
     Prepare(segs);
     BinaryWrite3( head + ".pathsdb.k" + ToString(K), segs );    }
