/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef PULL_APART_HKP_H
#define PULL_APART_KHP_H

#include "Basevector.h"
#include "ReadPairing.h"
#include "ReadLocation.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/SeqOnHyper.h"


/** Function: PullApartByInsertSize
    
    Attempts to pull apart overcollapsed vertices and edges, if the links across
    them can partition the edges into distinct genomic clusters.
    
    This function will first use the smallest insert size pairs until no more
    changes can be made. Then it will try using the both the smallest and
    2nd smallest insert size pairs, and will continue in this fashion until
    it has tried all insert sizes and can make no more changes.
    
    Returns an updated HyperKmerPath and HyperBaseVector, plus an updated
    set of alignments, alignment index and pair places.

    Calls <PullApartFull>
*/
void PullApartByInsertSize( HyperKmerPath& h, HyperBasevector& hb,
			    vec<CompressedSeqOnHyper>& csaligns,
			    vec<vec<int> >& csaligns_index,
			    vec< IndexPair >& pair_places,
			    const vec<int>& insert_sizes,
			    const vec<read_pairing>& pairs, 
			    const vec<int>& pairs_index,
			    const vec<tagged_rpint>& unipathsxdb,
			    const vec<int>& predicted_copyno,
			    const vecbasevector& reads, 
			    const KmerBaseBroker* kbb, 
			    const String& sub_dir,
			    bool pull_verbose, bool SHOW_PAIR_ALIGNS );



/** Function: PullApartByFull

    Attempts to pull apart overcollapsed vertices and edges, if the links across
    them can partition the edges into distinct genomic clusters.

    For inserts up to the specified size (or all inserts if MAX_SHORT_INSERT_SIZE
    is set to -1), cleans up repeatedly until the graph is stable.
    
    Returns an updated HyperKmerPath and HyperBaseVector, plus an updated
    set of alignments and alignment index. Also returns updated pair places - but
    only up to the insert size specified.

    Calls <PullApartHKP>
*/
int PullApartFull(// Non-const things we change (or update):
		  HyperKmerPath& h,
		  HyperBasevector& hb,
		  vec<CompressedSeqOnHyper>& csaligns, 
		  vec<vec<int> >& csaligns_index, 
		  vec< pair<int,int> >& pair_places,
		  // Other things needed for PullApartVertices (const):
		  const int MAX_SHORT_INSERT_SEP,
		  const vec<read_pairing>& pairs, 
		  const vec<int>& pairs_index,
		  // Needed for MaxEdgeCopyNumber (const):
		  const vec<tagged_rpint>& unipathsxdb,
		  const vec<int>& predicted_copyno,
		  // Other things needed for AlignPairsToHyper (const):
		  const vecbasevector& reads, 
		  const KmerBaseBroker* kbb, 
		  const String& sub_dir,
		  // flags:
		  bool verbose, bool SHOW_PAIR_ALIGNS );


/** Function: PullApartHKP

    Attempt to pull apart overcollapsed vertices in the HyperKmerPath.

    If examination of read pairs makes it clear that we can partition
    the edges incident to a vertex v into multiple classes such that
    the true genome path must always enter and exit v along edges in
    the same class, then pull v apart into one vertex for each class.
    
    Returns true if it made any changes to the graph.
*/
bool PullApartHKP( HyperKmerPath& hkp, const int max_insert_sep,
		   const vec<CompressedSeqOnHyper>& csaligns,
		   const vec< IndexPair >& pair_places,
		   const vec<read_pairing>& pairs, 
		   const vec<int>& pairs_index,
		   const vec<vrtx_t>& to_left_vertex,  // left=from, right=to
		   const vec<vrtx_t>& to_right_vertex,
		   const vec<int>& max_edge_copyno,
		   bool verbose );

#endif
