/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "Equiv.h"
#include "paths/AlignSeqsToHyper.h"
#include "paths/AlignPairsToHyper.h"
#include "paths/GlobalCleanAnnex.h"
#include "paths/PullApartHyperKmerPath.h"


/* Function: PullApartByInsertSize
   Attempts to pull apart overcollapsed vertices and edges, if the links across
   them can partition the edges into distinct genomic clusters.
*/
void PullApartByInsertSize( HyperKmerPath& h, HyperBasevector& hb,
			    vec<CompressedSeqOnHyper>& csaligns,
			    vec<vec<int> >& csaligns_index,
			    vec< IndexPair >& pair_places,
			    const vec<int>& insert_sizes,
			    const vec<read_pairing>& pairs, 
			    const vec<int>& pairs_index,
			    const vec<tagged_rpint>& unipathsxdb,
			    const vec<int>& predicted_copyno,
			    const vecbasevector& reads, 
			    const KmerBaseBroker* kbb, 
			    const String& sub_dir,
			    bool pull_verbose, bool SHOW_PAIR_ALIGNS ) {

  double vclock = WallClockTime( );
  vec<int> pull_rounds( insert_sizes.size(), 0 );
  int tot_good_rounds = 0;
  bool another_pass = true;
  while( another_pass ) {
    another_pass = false;
    int cur_good_rounds;
    for( int i=0; i<insert_sizes.isize(); i++ ) {
      // Find pair placements
      FindPairPlacements( h, hb, csaligns, csaligns_index, reads.size(), pairs,
			  pair_places, SHOW_PAIR_ALIGNS, sub_dir, insert_sizes[i] );
      // Pull apart using inserts up to specified size
      tot_good_rounds += cur_good_rounds =
	PullApartFull( h, hb, csaligns, csaligns_index, pair_places, insert_sizes[i], 
		       pairs, pairs_index, unipathsxdb, predicted_copyno,
		       reads, kbb, sub_dir, pull_verbose, SHOW_PAIR_ALIGNS );
      pull_rounds[i] += cur_good_rounds;
      if( cur_good_rounds > 0 && i > 0 )
	another_pass = true;
    }
  }
  cout << TimeSince(vclock) << " used in pull-apart/rebuild loop. "
       << tot_good_rounds << " rounds effective. ";
  for(int i=0; i<pull_rounds.isize(); i++)
    cout << (i?"+":"(") << pull_rounds[i];
  cout << " by max insert size)" << endl;
}


/* Function: PullApartByFull
   Attempts to pull apart overcollapsed vertices and edges, if the links across
   them can partition the edges into distinct genomic clusters.
*/
int PullApartFull(// Non-const things we change (or update):
		  HyperKmerPath& h,
		  HyperBasevector& hb,
		  vec<CompressedSeqOnHyper>& csaligns, 
		  vec<vec<int> >& csaligns_index, 
		  vec< pair<int,int> >& pair_places,
		  // Other things needed for PullApartVertices (const):
		  const int MAX_SHORT_INSERT_SEP,
		  const vec<read_pairing>& pairs, 
		  const vec<int>& pairs_index,
		  // Needed for MaxEdgeCopyNumber (const):
		  const vec<tagged_rpint>& unipathsxdb,
		  const vec<int>& predicted_copyno,
		  // Other things needed for AlignPairsToHyper (const):
		  const vecbasevector& reads, 
		  const KmerBaseBroker* kbb, 
		  const String& sub_dir,
		  // flags:
		  bool verbose, bool SHOW_PAIR_ALIGNS ) {

  bool did_something = true;
  int successes = 0;
  vec<vrtx_t> to_right_vertex, to_left_vertex;
  vec<int> max_edge_copyno;
  double pull_clock, rebuild_clock;

  while( did_something ) {
    
    pull_clock = WallClockTime();
    h.ToRight(to_right_vertex), h.ToLeft(to_left_vertex);
    MaxEdgeCopyNumber( h, unipathsxdb, predicted_copyno, max_edge_copyno );
    did_something = PullApartHKP( h, MAX_SHORT_INSERT_SEP, 
				  csaligns, pair_places, pairs, pairs_index,
				  to_left_vertex, to_right_vertex, 
				  max_edge_copyno,
				  verbose );
    cout << "  " << TimeSince(pull_clock) << " used pulling vertices apart with ";
    if( MAX_SHORT_INSERT_SEP == -1 )
      cout << "all inserts" << endl;
    else
      cout << "inserts under " << MAX_SHORT_INSERT_SEP << " bases" << endl;

    // Clean up and rebuild, if pulling apart did anything
    if( did_something ) {
      successes++;
      if(verbose)
	cout << "  PullApartVertices successful (round " << successes
	     << ", cleaning up and trying again)" << endl;

      rebuild_clock = WallClockTime();
      h.RemoveUnneededVertices( );
      h.ReduceLoops( );
      h.CompressEdgeObjects( );
      h.RemoveDeadEdgeObjects( );
      h.RemoveEdgelessVertices( );

      // Rebuild hb and alignments.

      hb = HyperBasevector( h, *kbb );
      AlignPairsToHyper( h, hb, reads, pairs, sub_dir, csaligns, csaligns_index,
			 pair_places, SHOW_PAIR_ALIGNS, MAX_SHORT_INSERT_SEP );
      cout << "  " << TimeSince(rebuild_clock) 
	   << " used rebuilding alignments" << endl;
    }
  }
  return successes;
}


// Helper functions for PullApartHKP.


int MinInEdge( int vx, const HyperKmerPath& hkp ) {
  if( hkp.To(vx).empty() ) return -1;
  int ans = hkp.EdgeObjectByIndexTo(vx,0).KmerCount();
  for(int i=1; i<hkp.ToSize(vx); i++)
    ans = Min(ans, hkp.EdgeObjectByIndexTo(vx,i).KmerCount());
  return ans;
}

int MinOutEdge( int vx, const HyperKmerPath& hkp ) {
  if( hkp.From(vx).empty() ) return -1;
  int ans = hkp.EdgeObjectByIndexFrom(vx,0).KmerCount();
  for(int i=1; i<hkp.FromSize(vx); i++)
    ans = Min(ans, hkp.EdgeObjectByIndexFrom(vx,i).KmerCount());
  return ans;
}

// Only look at vertices where min(in_edge_len) + min(out_edge_len) > min_diameter
// and with at least two in-edges and at least two out-edges.
void VxsToPull( const HyperKmerPath& hkp, int min_diameter,
		vec<Bool>& pull_vx ) {
  pull_vx.resize( hkp.N() );
  for( int vx = 0; vx < hkp.N(); vx++ )
    pull_vx[vx] = ( hkp.ToSize(vx)>1 && hkp.FromSize(vx)>1
		    && MinInEdge(vx,hkp) + MinOutEdge(vx,hkp) > min_diameter );
}

// Only look at edges   ===x---y===   where there is a unique edge out of x
// and a unique edge into y, >1 edge into x and out of y, and where you expect
// partners of some read on each === edge to hit another.
//
// We record this edge by setting pull_edge[x]=y; else pull_edge[x]=-1;
void EdgesToPull( const HyperKmerPath& hkp, int max_edge_len,
		  int min_diameter, vec<vrtx_t>& pull_edge ) {
  pull_edge.resize( hkp.N(), -1 );
  for( vrtx_t vx = 0; vx < hkp.N(); vx++ ) {
    if( hkp.ToSize(vx)<2 || hkp.FromSize(vx)!=1 ) continue;
    vrtx_t vy = hkp.From(vx)[0];
    if( hkp.FromSize(vy)<2 || hkp.ToSize(vy)!=1 ) continue;
    int edge_len = MinOutEdge(vx,hkp);
    if( edge_len > max_edge_len ) continue;
    int min_in = MinInEdge(vx,hkp);
    int min_out = MinOutEdge(vy,hkp);
    if( min_in + edge_len + min_out < min_diameter ) continue;
    pull_edge[vx] = vy;
  }
}


void UniquelyPlacedReads( const vec<CompressedSeqOnHyper>& csaligns,
			  const vec< IndexPair >& pair_places,
			  int num_reads, vec<Bool>& unique ) {
  vec<int> read_placement_count( num_reads, 0 );
  for( vec< IndexPair >::const_iterator pp = pair_places.begin();
       pp != pair_places.end(); pp++ ) {
    read_placement_count[ csaligns[pp->first].Id1() ]++;
    read_placement_count[ csaligns[pp->second].Id1() ]++;
  }
  
  unique.resize( num_reads );
  for(int i=0; i<num_reads; i++)
    unique[i] = (read_placement_count[i]==1);
}


// Change connection counts into an equivalence relation:
// edges are joined if more than one read pair say to do so.
// Probably we should be using variations on this mechanism
// all the time, instead of the current strange hybrid where 
// some equiv rel's are built on the fly.
//
// This is morally dicey: if in-edge A is connected to 
// out-edge B once and out-edge C twice, are we really
// confident that B is right?  I'm worried this might
// introduce mistakes.

bool EquivNotTooFew( const map<pair<int,int>,int>& counts,
		     int to_vx_degree, int from_vx_degree,
		     equiv_rel& rel,
		     int too_few = 1 ) {
  rel.Initialize( to_vx_degree + from_vx_degree );
  for( map<pair<int,int>,int>::const_iterator mi = counts.begin();
       mi != counts.end(); mi++ )
    if( mi->second > too_few )
      rel.Join( mi->first.first, to_vx_degree + mi->first.second );

  return ( rel.OrbitCount() > 1 && ! rel.Singletons() );
}


// Ignoring low-frequency links is much more justifiable if all
// the edges involved occur only once in the true genome.

int MaxCopynoIn( const HyperKmerPath& hkp, int vx,
		 const vec<int>& max_edge_copyno ) {
  int ans = -1;
  for(int i=0; i<hkp.ToSize(vx); i++)
    ans = Max(ans, max_edge_copyno[ hkp.EdgeObjectIndexByIndexTo(vx,i) ]);
  return ans;
}

int MaxCopynoOut( const HyperKmerPath& hkp, int vx,
		  const vec<int>& max_edge_copyno ) {
  int ans = -1;
  for(int i=0; i<hkp.FromSize(vx); i++)
    ans = Max(ans, max_edge_copyno[ hkp.EdgeObjectIndexByIndexFrom(vx,i) ]);
  return ans;
}

// currently unused:
int EdgeMaxCopyno( const HyperKmerPath& hkp, int vx, int vy, 
		   const vec<int>& max_edge_copyno ) {
  return( Max( MaxCopynoIn(hkp,vx,max_edge_copyno),
	       MaxCopynoOut(hkp,vy,max_edge_copyno) ) );
}  



// Change connection counts into an equivalence relation:
// if some connecetions occur more than squelch_ratio times 
// as often as the next rarest, ignore the infrequent ones.
// Best if EdgeMaxCopyno==1 (or at least is low).
bool EquivSquelch( const map<pair<int,int>,int>& counts,
		   int to_vx_degree, int from_vx_degree,
		   equiv_rel& rel, int squelch_ratio ) {
  // Check whether there's a cutoff with appropriate squelch ratio
  vec<int> count_values;
  for( map<pair<int,int>,int>::const_iterator mi = counts.begin();
       mi != counts.end(); mi++ )
    count_values.push_back(mi->second);
  Sort(count_values);
  
  int too_few = -1;
  for(int i=0; i<count_values.isize()-1; i++)
    if(  count_values[i] * squelch_ratio <= count_values[i+1] ) {
      too_few = count_values[i];
      break;
    }
  if(too_few == -1)
    return false;

  return(EquivNotTooFew( counts, to_vx_degree, from_vx_degree, rel, too_few ));
}


/* Function: PrintPair
   Print info about pair that spans an overcollapsed vertex/edge.
*/
void PrintPair( const HyperKmerPath& hkp, 
                const edge_t edge_id, 
                const CompressedSeqOnHyper& left, 
                const CompressedSeqOnHyper& right, 
                const vrtx_t left_vx, 
                const vrtx_t right_vx, 
                const edge_t left_edge, 
                const edge_t right_edge, 
                const vec<read_pairing>& pairs, 
                const vec<int>& pairs_index,
                const bool unique = false ) {
  // Edges always overlap by K-1.
  int middle_dist = -(hkp.K()-1); 
  if( edge_id >= 0 ) {
    middle_dist += hkp.EdgeObject( edge_id ).KmerCount();
    cout << "At edge " << left_vx << "--" << BaseAlpha(edge_id) << "--" << right_vx;
  }
  else
    cout << "At vertex " << left_vx;
  int left_dist = hkp.EdgeObject(left_edge).KmerCount() + hkp.K() - 1 - left.Part0().Pos2();
  int right_dist = right.Part0().pos2();
  double total_sep = left_dist + middle_dist + right_dist;
  double exp_sep = pairs[pairs_index[left.Id1()]].sep;
  double exp_dev = pairs[pairs_index[left.Id1()]].sd;
  double stretch = (total_sep - exp_sep ) / exp_dev;
  cout << ", in-edge " 
       << BaseAlpha(left_edge)
       << " and out-edge " 
       << BaseAlpha(right_edge)
       << " joined by " << left.Id1() << (left.Rc1() ? "rc" : "fw") 
       << " at " <<  left_dist << " on " << BaseAlpha(left_edge) << " / "
       << right.Id1() << (right.Rc1() ? "rc" : "fw") 
       << " at " << right_dist << " on " << BaseAlpha(right_edge) 
       << " (stretch=" << stretch << ")";
  if ( unique )
    cout << " (unique placement)";
  cout << endl;
}


/* Function: PullApartHKP
   Attempt to pull apart overcollapsed vertices in the HyperKmerPath.
*/
bool PullApartHKP( HyperKmerPath& hkp, const int max_insert_sep,
		   const vec<CompressedSeqOnHyper>& csaligns,
		   const vec< IndexPair >& pair_places,
		   const vec<read_pairing>& pairs, 
		   const vec<int>& pairs_index,
		   const vec<vrtx_t>& to_left_vertex,  // left=from, right=to
		   const vec<vrtx_t>& to_right_vertex,
		   const vec<int>& max_edge_copyno,
		   bool verbose ) {

  // minimum, maximum believable lengths for the inserts we're going to use
  int sd_mult = 4;
  int min_stretched_sep = INT_MAX, max_stretched_sep = 0, max_compressed_sep=0;
  //  int min_diameter = 0;
  for( vec<read_pairing>::const_iterator p = pairs.begin(); p != pairs.end(); p++ )
    if( max_insert_sep == -1  ||  p->sep <= max_insert_sep ) {
      min_stretched_sep = Min( min_stretched_sep, p->sep - sd_mult * p->sd );
      max_stretched_sep = Max( max_stretched_sep, p->sep + sd_mult * p->sd );
      max_compressed_sep = Max( max_compressed_sep, p->sep - sd_mult * p->sd );
    }
  if( max_stretched_sep == 0 ) return false;

 
  // Which vertices are we going to try to pull apart?
  vec<Bool> pull_vx;
  VxsToPull( hkp, max_stretched_sep, pull_vx );

  // Which edges are we going to try to pull apart?
  // pull_edge[x]=y means pull the edge from vertex x to vertex y.
  vec<vrtx_t> pull_edge;
  EdgesToPull( hkp, max_stretched_sep /* -hkp.K()+1 */, max_compressed_sep, pull_edge );

  int numVerticesToPull = 0;
  for ( unsigned int i = 0; i < pull_vx.size(); ++i )
    if ( pull_vx[i] ) ++numVerticesToPull;

  int numEdgesToPull = 0;
  for ( unsigned int i = 0; i < pull_edge.size(); ++i )
    if ( pull_edge[i] >= 0 ) ++numEdgesToPull;

  if ( verbose ) 
    cout << "Attempting to pull apart " << numVerticesToPull << " vertices and "
         << numEdgesToPull << " edges." << endl;

  // Which reads are uniquely placed?
  vec<Bool> uniquely_placed;
  UniquelyPlacedReads( csaligns, pair_places, pairs_index.size(), uniquely_placed );

  // For each vertex or edge we look at, we create two equivalence relations 
  // among its edges, one from all pairs, the other from uniquely placed pairs.
  vec<equiv_rel> related_all( hkp.N() );
  vec<equiv_rel> related_unique( hkp.N() );

  // We also count how many links there are between each in and out edge.
  // This means the equivalence relation is redundant, we could just create
  // it from the counts when needed.
  vec< map<pair<int,int>,int> > related_unique_counts( hkp.N() );
  vec< map<pair<int,int>,int> > related_all_counts( hkp.N() );

  for(int vx=0; vx < hkp.N(); vx++ ) {
    if( pull_vx[vx] ) {
      ForceAssert( pull_edge[vx] == -1 );
      related_all[vx].Initialize( hkp.ToSize(vx) + hkp.FromSize(vx) );
      related_unique[vx].Initialize( hkp.ToSize(vx) + hkp.FromSize(vx) );
    }
    if( pull_edge[vx] != -1 ) {
      ForceAssert( ! pull_vx[vx] );
      related_all[vx].Initialize( hkp.ToSize(vx) + hkp.FromSize(pull_edge[vx]) );
      related_unique[vx].Initialize( hkp.ToSize(vx) + hkp.FromSize(pull_edge[vx]) );
    }
  }

  // An in-edge and an out-edge are declared equivalent if the
  // end reads of a (short enough) read pair land on them.
  for( vec< IndexPair >::const_iterator pp = pair_places.begin();
       pp != pair_places.end(); pp++ ) {

    const CompressedSeqOnHyper& left = csaligns[pp->first];
    const CompressedSeqOnHyper& right = csaligns[pp->second];
    if( ! left.SingleEdge() || ! right.SingleEdge() )
      continue;

    const read_pairing& the_pair = pairs[pairs_index[left.Id1()]];
    if( the_pair.sep + sd_mult*the_pair.sd > max_stretched_sep )
      continue;

    int left_edge = left.Part0().Id2();
    int right_edge = right.Part0().Id2();
    vrtx_t left_vx = to_right_vertex[left_edge];
    vrtx_t right_vx = to_left_vertex[right_edge];

    bool avoids_vertex = false;
    if( left_edge == right_edge ) {
      // Careful in loops!  pairs might not cross the vertex.
      int sepdiff = right.Part0().pos2() - left.Part0().Pos2() - the_pair.sep;
      avoids_vertex = Abs(sepdiff) <= 2 * sd_mult * the_pair.sd;
    }

    if( (right_vx == left_vx  && pull_vx[right_vx]  &&  ! avoids_vertex)
	||
	(pull_edge[left_vx] == right_vx) ) {
      // Declare those edges equivalent.
      // This code handles both the vx-pulling and edge-pulling cases.
      // For verbose output, track which case we're in:
      bool edge_case = (pull_edge[left_vx] == right_vx);

      int to_index = hkp.EdgeObjectIndexToToIndex( left_vx, left_edge );
      int from_index = hkp.EdgeObjectIndexToFromIndex( right_vx, right_edge );
      
      related_all_counts[left_vx][make_pair(to_index,from_index)]++;

      int left_dist = 0, right_dist = 0;
      double stretch = 0.;

      bool new_join_all =
	related_all[left_vx].Join( to_index, hkp.ToSize(left_vx)+from_index );
      if( verbose && new_join_all ) {
        edge_t edge_id = ( edge_case ? hkp.EdgeObjectIndexByIndexFrom(left_vx,0) : -1 );
        PrintPair( hkp, edge_id, left, right, left_vx, right_vx, left_edge, right_edge, pairs, pairs_index );
      }
	
      if( uniquely_placed[ left.Id1() ] ) {
	ForceAssert( uniquely_placed[ right.Id1() ] );

	related_unique_counts[left_vx][make_pair(to_index,from_index)]++;
	
	bool new_join_unique =
	  related_unique[left_vx].Join( to_index, hkp.ToSize(left_vx)+from_index );
	if( verbose && new_join_unique ) {
          edge_t edge_id = ( edge_case ? hkp.EdgeObjectIndexByIndexFrom(left_vx,0) : -1 );
          PrintPair( hkp, edge_id, left, right, left_vx, right_vx, left_edge, right_edge, pairs, pairs_index, true );
	}
      }
    }
  }

  if( verbose ) { // maybe should be verbosity>1?
    for(int vx=0; vx < hkp.N(); vx++) {

      if( pull_vx[vx] ) {
	cout << "At vertex " << vx;
	cout << ",  in-edges ";
	for(int i=0; i<hkp.ToSize(vx); i++)
	  cout << hkp.EdgeObjectIndexByIndexTo(vx,i) << "(m<=" 
	       << max_edge_copyno[hkp.EdgeObjectIndexByIndexTo(vx,i) ] << ") ";
	cout << " out-edges ";
	for(int i=0; i<hkp.FromSize(vx); i++)
	  cout << hkp.EdgeObjectIndexByIndexFrom(vx,i) << "(m<=" 
	       << max_edge_copyno[hkp.EdgeObjectIndexByIndexFrom(vx,i) ] << ") ";
	cout << "\n";
	
	cout << "     All links ok:  ";
	for( map<pair<int,int>,int>::iterator i = related_all_counts[vx].begin();
	     i != related_all_counts[vx].end(); i++ ) {
	  int e1 = hkp.EdgeObjectIndexByIndexTo(vx,i->first.first);
	  int e2 = hkp.EdgeObjectIndexByIndexFrom(vx,i->first.second);
	  cout << e1<< "->" << e2 << " x" << i->second << "   ";
	}
	cout << "\n";
	cout << "     Uniques only:  ";
	for( map<pair<int,int>,int>::iterator i = related_unique_counts[vx].begin();
	     i != related_unique_counts[vx].end(); i++ ) {
	  int e1 = hkp.EdgeObjectIndexByIndexTo(vx,i->first.first);
	  int e2 = hkp.EdgeObjectIndexByIndexFrom(vx,i->first.second);
	  cout << e1<< "->" << e2 << " x" << i->second << "   ";
	}
	cout << "\n";
      }

      if( pull_edge[vx] != -1 ) {
	vrtx_t vy = pull_edge[vx];
        edge_t edge_id = hkp.EdgeObjectIndexByIndexFrom(vx,0);
	cout << "At edge " << vx << "--" << BaseAlpha(edge_id) << "--" << vy;
	cout << ",  in-edges ";
	for(int i=0; i<hkp.ToSize(vx); i++)
	  cout << hkp.EdgeObjectIndexByIndexTo(vx,i) << "(m<=" 
	       << max_edge_copyno[hkp.EdgeObjectIndexByIndexTo(vx,i) ] << ") ";
	cout << " out-edges ";
	for(int i=0; i<hkp.FromSize(vy); i++)
	  cout << hkp.EdgeObjectIndexByIndexFrom(vy,i) << "(m<=" 
	       << max_edge_copyno[hkp.EdgeObjectIndexByIndexFrom(vy,i) ] << ") ";
	cout << "\n";
	cout << "     All links ok:  ";
	for( map<pair<int,int>,int>::iterator i = related_all_counts[vx].begin();
	     i != related_all_counts[vx].end(); i++ ) {
	  int e1 = hkp.EdgeObjectIndexByIndexTo(vx,i->first.first);
	  int e2 = hkp.EdgeObjectIndexByIndexFrom(vy,i->first.second);
	  cout << e1<< "->" << e2 << " x" << i->second << "   ";
	}
	cout << "\n";
	cout << "     Uniques only:  ";
	for( map<pair<int,int>,int>::iterator i = related_unique_counts[vx].begin();
	     i != related_unique_counts[vx].end(); i++ ) {
	  int e1 = hkp.EdgeObjectIndexByIndexTo(vx,i->first.first);
	  int e2 = hkp.EdgeObjectIndexByIndexFrom(vy,i->first.second);
	  cout << e1<< "->" << e2 << " x" << i->second << "   ";
	}
	cout << "\n";
      }
    }
    cout << flush;
  }


  // Modifications we will make to the graph:
  // The list <edge_id,new_vx> of edge endpoints to move
  vec< pair<int,vrtx_t> > new_from_ends, new_to_ends;
  // The list <old_edge_id, new_left_vx> of edgees to duplicate.
  // It goes from newly-created edges new_left_vx to new_left_vx + 1.
  vec< pair<int,vrtx_t> > edges_to_copy;
  // The list of vertices to delete, along with all incident edges.
  // No need to put edgeless vertices on this list, but we use it to
  // clean up the edges that were duplicated in edge-pulling.
  vec<vrtx_t> vxs_to_delete;


  int vx_pull_count = 0, edge_pull_count = 0;

  int next_new_vx = hkp.N(), orbits;
  vec<vrtx_t> orbit_reps;


  for(vrtx_t vx=0; vx < hkp.N(); vx++ ) {
    if( ! pull_vx[vx] && pull_edge[vx]==-1 ) continue;

    bool edge_case = (pull_edge[vx] != -1);
    vrtx_t vy = (edge_case ? pull_edge[vx] : vx);
    int to_size = hkp.ToSize(vx), from_size = hkp.FromSize(vy);

    equiv_rel on_the_fly;
    equiv_rel *rel_p = NULL;
    int all_squelch_ratio = 10, unique_squelch_ratio = 5;  // Shouldn't be hard-coded

    if( related_unique[vx].OrbitCount() > 1 && ! related_unique[vx].Singletons() )
      rel_p = &related_unique[vx];
    else if( related_all[vx].OrbitCount() > 1 && ! related_all[vx].Singletons() )
      rel_p = &related_all[vx];
    else if( MaxCopynoIn(hkp, vx, max_edge_copyno) == 1
             || MaxCopynoOut(hkp, vy,max_edge_copyno) == 1 ) {
      if( EquivSquelch(related_unique_counts[vx], 
                       to_size, from_size, on_the_fly, unique_squelch_ratio) )
        rel_p = &on_the_fly;
      else if( EquivSquelch(related_all_counts[vx], 
                            to_size, from_size, on_the_fly, all_squelch_ratio) )
        rel_p = &on_the_fly;
//     else if( EquivNotTooFew( related_unique_counts[vx], to_size, from_size, on_the_fly ) )
//       rel_p = &on_the_fly;
//     else if( EquivNotTooFew( related_all_counts[vx], to_size, from_size, on_the_fly ) )
//       rel_p = &on_the_fly;
    }

    if( rel_p == NULL )
      continue;
    const equiv_rel& rel = *rel_p;

    rel.OrbitRepsAlt( orbit_reps );

    if(verbose) {
      if( edge_case )
	cout << "    Partitioning edges around edge " << vx << "---" << vy
	     << " and replacing it with " << orbit_reps.size()
	     << " new edges" << endl;
      else
	cout << "    Partitioning edges around vertex " << vx
	     << " and replacing it with " << orbit_reps.size()
	     << " new vertices" << endl;
    }

    for(int j=0; j<to_size; j++) {
      int edge_id = hkp.EdgeObjectIndexByIndexTo(vx,j);
      int new_vx = next_new_vx 
	+ (edge_case?2:1)*BinPosition( orbit_reps, rel.ClassId(j) );
      ForceAssert(new_vx >= next_new_vx);
      new_to_ends.push_back(make_pair( edge_id, new_vx ));
    }
    for(int j=0; j<from_size; j++) {
      int edge_id = hkp.EdgeObjectIndexByIndexFrom(vy,j);
      int new_vx = next_new_vx
	+ (edge_case?2:1)*BinPosition( orbit_reps, rel.ClassId(j+to_size) )
	+ (edge_case?1:0);
      ForceAssert(new_vx >= next_new_vx);
      new_from_ends.push_back(make_pair( edge_id, new_vx ));
    }

    if( edge_case ) {
      int edge_to_copy = hkp.EdgeObjectIndexByIndexFrom(vx,0);
      for(int i=0; i<orbit_reps.isize(); i++)
	edges_to_copy.push_back(make_pair( edge_to_copy, next_new_vx + 2*i ));
      vxs_to_delete.push_back(vx);  // to delete the oft-copied outgoing edge 0
    }

    next_new_vx += (edge_case?2:1) * orbit_reps.size();

    if( edge_case ) edge_pull_count++; else vx_pull_count++;

  }

  if( new_from_ends.empty() && new_to_ends.empty() )
    return false;

  // Finally, make the modifications to the HyperKmerPath
  hkp.AddVertices( next_new_vx - hkp.N() );
  vec< pair<int,vrtx_t> >::const_iterator ev;
  // Copy edges:
  for( ev = edges_to_copy.begin(); ev != edges_to_copy.end(); ev++ )
    hkp.AddEdge( ev->second, ev->second + 1, hkp.EdgeObject(ev->first) );
  // Move From vertices:
  for( ev = new_from_ends.begin(); ev != new_from_ends.end(); ev++ )
    hkp.GiveEdgeNewFromVx( ev->first, to_left_vertex[ev->first], ev->second );
  // Move To vertices:
  for( ev = new_to_ends.begin(); ev != new_to_ends.end(); ev++ )
    hkp.GiveEdgeNewToVx( ev->first, to_right_vertex[ev->first], ev->second );
  // Delete vertices: (to clean up after edge pulling)
  UniqueSort(vxs_to_delete);
  for( int i=0; i<vxs_to_delete.isize(); i++)
    hkp.DeleteEdgesAtVertex( vxs_to_delete[i] );

  cout << "Pulled apart " << vx_pull_count 
       << (vx_pull_count==1 ? " vertex" : " vertices")
       << " and " << edge_pull_count
       << (edge_pull_count==1 ? " edge" : " edges") << "."
       << endl;

  return true;
}
