/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "CoreTools.h"
#include "math/Functions.h"
#include "paths/PairedPair.h"
#include "paths/PoorlyCoveredPairs.h"
#include "paths/UniqueExtend.h"
#include "random/NormalRandom.h"
#include "random/PoissonTail.h"

void DetectPoorlyCoveredPairs( 

     const HyperKmerPath& h,           // the HyperKmerPath
     vec<pp_pair>& ppp,                // the pairs
     vec<int>& pppmult,                // their multiplicities
     vec<Bool>& ppp_trusted,
     const vec<Bool>& unique_edge,     // which edges are believed to be unique
     const vec<Bool>& hstart_known,
     const vec<int>& hstart,
     const int NHOOD_RADIUS_INTERNAL,
     const double mean_read_length,    // for the short-insert pairs, in kmers
     const double coverage,            // by the short-insert pairs
     const double mean_insert_length,  // of the short-insert pairs, in kmers
     const double insert_dev,          // insert length deviation
     const int sim_cov,                // coverage level used in simulation
     Bool verbose
     
          )

{    
     // Set up indices.

     vec<int> to_right_vertex( h.EdgeObjectCount( ) ); 
     for ( int i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.To(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexTo( i, j );
               to_right_vertex[e] = i;    }    }
     vec<int> to_left_vertex( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.From(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( i, j );
               to_left_vertex[e] = i;    }    }
     vec<int> elen( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          elen[i] = h.EdgeLength(i);

     // Define edges in graph that are actually in use.

     vec<Bool> edges_to_use( h.EdgeObjectCount( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          for ( int j = 0; j < p.LeftSize( ); j++ )
               edges_to_use[ p.Left(j) ] = True;
          for ( int j = 0; j < p.RightSize( ); j++ )
               edges_to_use[ p.Right(j) ] = True;    }

     // Go through the pairs.

     int read_length = int( round(mean_read_length) );
     vec<Bool> to_delete( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          if (verbose) cout << "\nanalyzing [" << i << "] " << p << "\n";
          const pp_read &r1 = p.Left( ), &r2 = p.Right( );
          int r1len = 0, r2len = 0;
          for ( int j = 0; j < r1.isize( ); j++ )
               r1len += elen[ r1[j] ];
          for ( int j = 0; j < r2.isize( ); j++ )
               r2len += elen[ r2[j] ];

          // Carry out the simulation.   Note that we're allowing reads to go off
          // the end at sources and sinks, which doesn't make sense.

          vec< pair<pp_read, pp_read> > range;
          for ( int sc = 0; sc < sim_cov; sc++ )
          {    
               // start1 is the start of the first read on r1

               int first_start1 = Max( -read_length + 1, 
                    int( floor( r1len + p.Gap( ) 
                    - mean_insert_length - 4.0 * insert_dev - 4.0 - p.Dev( ) ) ) );
               for ( int start1 = first_start1; start1 <= r1len - 1; start1++ )
               {    int insert_length = int( round( 
                         mean_insert_length + FastNormal( ) * insert_dev ) );
                    int p_sep = int( round( p.Gap( ) + FastNormal( ) * p.Dev( ) ) );
                    int stop1 = start1 + read_length;

                    // Figure out which edges of r1 are touched.  The range of 
                    // touched edges is [first_touched1, last_touched1], where 
                    // first_touched1 = -1 means that the read is off the end 
                    // on the left, and first_touched = r1.size( ) means that 
                    // the read is off the end on the right.

                    int first_touched1 = -1, last_touched1 = -1;
                    int pos1 = 0;
                    for ( int m = 0; m <= r1.isize( ); m++ )
                    {    if ( start1 < pos1 ) 
                         {    first_touched1 = m-1;
                              break;    }
                         pos1 += elen[ r1[m] ];    }
                    pos1 = r1len;
                    for ( int m = r1.isize( ); m >= 0; m-- )
                    {    if ( stop1 > pos1 )
                         {    last_touched1 = m;
                              break;    }
                         pos1 -= elen[ r1[m-1] ];    }

                    // Now extend uniquely.

                    for ( int u = first_touched1; u >= 1; u-- )
                    {    if ( !h.To( to_left_vertex[ r1[u] ] ).solo( ) ) break;
                         --first_touched1;    }
                    for ( int u = last_touched1; u < r1.isize( ) - 1; u++ )
                    {    if ( !h.From( to_right_vertex[ r1[u] ] ).solo( ) ) break;
                         ++last_touched1;    }
     
                    // start2 is the predicted start of the second read on r2.

                    int start2 = start1 + insert_length 
                         - read_length - r1len - p_sep;
                    int stop2 = start2 + read_length;

                    // Figure out which edges of r2 are touched.  If none are
                    // touched, bail.  The range of touched edges is
                    // [first_touched2, last_touched2], where first_touched2 = -1 
                    // means that the read is off the end on the left, and 
                    // first_touched2 = r2.size( ) means that the read is off the 
                    // end on the right.

                    if ( stop2 <= 0 || start2 >= r2len ) continue;
                    int first_touched2 = -1, last_touched2 = -1;
                    int pos2 = 0;
                    for ( int m = 0; m <= r2.isize( ); m++ )
                    {    if ( start2 < pos2 ) 
                         {    first_touched2 = m-1;
                              break;    }
                         pos2 += elen[ r2[m] ];    }
                    pos2 = r2len;
                    for ( int m = r2.isize( ); m >= 0; m-- )
                    {    if ( stop2 > pos2 )
                         {    last_touched2 = m;
                              break;    }
                         pos2 -= elen[ r2[m-1] ];    }

                    // Forget the cases that are off the end - they don't seem to
                    // be useful.

                    if ( first_touched1 < 0 || last_touched1 == r1.isize( ) )
                         continue;
                    if ( first_touched2 < 0 || last_touched2 == r2.isize( ) )
                         continue;

                    // Now extend uniquely.

                    static pp_read left, right;
                    left.clear( ), right.clear( );
                    for ( int u = first_touched1; u <= last_touched1; u++ )
                         left.push_back( p.Left(u) );
                    for ( int u = first_touched2; u <= last_touched2; u++ )
                         right.push_back( p.Right(u) );
                    int left_add, right_add;
                    UniqueExtend( left, left_add, right_add, False, False, h, 
                         edges_to_use, to_left_vertex, to_right_vertex, False,
                         hstart_known, hstart, NHOOD_RADIUS_INTERNAL, elen );
                    // UniqueExtend( left, left_add, right_add, True, True, h, 
                    //      edges_to_use, to_left_vertex, to_right_vertex, True,
                    //      hstart_known, hstart, NHOOD_RADIUS_INTERNAL, elen );
                    UniqueExtend( right, left_add, right_add, False, False, h, 
                         edges_to_use, to_left_vertex, to_right_vertex, False,
                         hstart_known, hstart, NHOOD_RADIUS_INTERNAL, elen );
                    // UniqueExtend( right, left_add, right_add, True, True, h, 
                    //      edges_to_use, to_left_vertex, to_right_vertex, True,
                    //      hstart_known, hstart, NHOOD_RADIUS_INTERNAL, elen );

                    // Record the edge info.

                    range.push_back( make_pair( left, right ) );    }    }

          // Collate results of simulation.

          Sort(range);
          for ( int j = 0; j < range.isize( ); j++ )
          {    int u;
               for ( u = j + 1; u < range.isize( ); u++ )
                    if ( range[u] != range[j] ) break;
               double mrl = mean_read_length + h.K( ) - 1;
               int mult = int( round( double(u-j) * 
                    ( coverage / mrl ) / double(sim_cov) ) );
               if (verbose)
               {    cout << "[" << mult << "] "
                         << range[j].first << " --> " << range[j].second;    }
               int count = 0;
               for ( int m = 0; m < ppp.isize( ); m++ )
               {    if ( ppp[m].Left( ) == range[j].first 
                         && ppp[m].Right( ) == range[j].second )
                    {    count += pppmult[m];    }    }
               double prob = PoissonTail( mult, count );
               if (verbose) cout << " [see " << count << ", p = " << prob << "]";    
               if ( prob < 0.001 ) 
               {    if (verbose) cout << " IMPROBABLE!";
                    if ( ppp_trusted[i] && prob >= 0.0001 )
                    {    if (verbose) cout << " (trusted, not deleting)";    }
                    else to_delete[i] = True;    }
               if (verbose) cout << "\n";
               j = u - 1;    }    }
     EraseIf( ppp, to_delete );
     EraseIf( pppmult, to_delete );
     EraseIf( ppp_trusted, to_delete );    }
