// Copyright (c) 2005 Broad Institute/Massachusetts Institute of Technology

#ifndef PATHS_PILEREADS_H
#define PATHS_PILEREADS_H

#include "Vec.h"

#include "paths/KmerPath.h"
#include "paths/PathEmbedding.h"
#include "paths/KmerBaseBroker.h"
#include "paths/SuperBaseVector.h"

#include <list.h>  // PileReadsOnPath needs a list<KmerPath> subpath_storage

/// A ReadOnPath encapsulates the notion of a particular read embedded
/// into a path, possibly hanging off the end.
//
//  Philosophically, we'd like to hold a read_id and an Embedding.
//  However, we also want to handle the case where a part of a read
//  embeds, and the rest of the read hangs off the end.  So we
//  actually hold *two* embeddings: one of the sub_path into the read,
//  and one of the sub_path into the embedding.  If there's nothing
//  hanging off the end, the first embedding will just be the identity.
//  Identity embeddings Push &c trivially, so the cost is nonzero but small.
//
//  The two embeddings ought to always have the same sub.

struct ReadOnPath {
  ReadOnPath(const PathEmbedding& into_read, 
	     const PathEmbedding& into_path,
	     int id) 
    : intoRead(into_read), intoPath(into_path), readId(id) { };
  PathEmbedding intoRead;
  PathEmbedding intoPath;
  int readId;  // -id-1 means rc of read id.
};

inline std::ostream & operator<<(std::ostream &out, const ReadOnPath &p)
{
  out << "Embed of read # " << p.readId << "\n";
  if (p.intoRead.IsIdentity()) {
    out << p.intoPath;
  } else {
    out << "(" << p.intoRead << ", " << p.intoPath << ")\n";
  }
  return out;
}


class ReadsOnPathPiler {
public:
  ReadsOnPathPiler( const vecKmerPath &Paths,
		    const vecKmerPath &Paths_rc,
		    const vec<tagged_rpint> &PathsDB ) :
    paths(Paths), paths_rc(Paths_rc), pathsDB(PathsDB) { }

  /// Gives back a vector of all embeddings of reads onto the path

  void PileReadsOnPath( const KmerPath& path,
			vec<ReadOnPath>& pile,
			list<KmerPath>& subpath_storage,
			bool allow_overhang = true ) const;

private:
  const vecKmerPath &paths;
  const vecKmerPath &paths_rc;
  const vec<tagged_rpint> &pathsDB;

};

/// Convert a vec<ReadOnPath>, eg returned by PileReadsOnPath,
/// to a vec<ReadOnSuperBaseVector>, to go along with the 
/// SuperBaseVector of the path the reads were piled on.

void ConvertEmbeddingsToLocs( const vec<ReadOnPath>& ROPs,
			      vec<ReadOnSuperBaseVector>& SBVlocs,
			      const KmerBaseBroker* kbb );

#endif
