// Copyright (c) 2005 Broad Institute/Massachusetts Institute of Technology

#ifndef PATHS_PATHWALKHEAP_H
#define PATHS_PATHWALKHEAP_H

#include "paths/PathWalk.h"
#include "paths/WalkPlot.h"

// An abstract interface for a heap of PathWalks.

class AbstractPathWalkHeap
{
 public:
  virtual 
  ~AbstractPathWalkHeap()
  {}

  virtual 
  void Push( const PathWalk &newWalk ) = 0;
  
  virtual
  void Pop() = 0;

  virtual
  void Clear() = 0;
  
  virtual
  unsigned int Size() const = 0;
  
  virtual
  const PathWalk& GetTop() const = 0;
  
  class Visitor
  {
   public:
    virtual 
    ~Visitor() {}

    virtual 
    void Visit( const PathWalk& walk ) = 0;
  };

  // Accept() calls vistor.Visit() on each PathWalk in the heap
  // (currently in undefined order!).
  virtual
  void Accept( Visitor& visitor ) const = 0;

  class Filter
  {
   public:
    virtual
    ~Filter() {}

    virtual
    bool Matches( const PathWalk& walk ) const = 0;
  };

  // RemoveIf() removes PathWalks where filter.Matches() returns true.
  virtual 
  void RemoveIf( const Filter& filter ) = 0;
};



// A set-based concrete subclass of AbstractPathWalkHeap. 

// The Comparator class should be a functor class that accepts two
// PathWalks are returns true if the first PathWalk should be extended
// before the second.

template <class Comparator>
class SetBasedPathWalkHeap
  : public AbstractPathWalkHeap
{
 private:
  set<PathWalk,Comparator> m_heap;

 public:
  virtual
  ~SetBasedPathWalkHeap() {}

  virtual 
  void Push( const PathWalk &newWalk )
  {
    PathWalkSetInsert( m_heap, newWalk );
  }

  virtual
  void Pop()
  {
    if ( m_heap.empty() )
    {
      cout << "Tried to pop an empty heap!" << endl;
      TracebackThisProcess();
    }
    else
      m_heap.erase( --(m_heap.end()) );
  }

  virtual
  void Clear()
  {
    m_heap.clear();
  }

  virtual
  unsigned int Size() const
  {
    if ( m_heap.empty() )
      return 0;
    else
      return m_heap.size();
  }

  virtual
  const PathWalk& GetTop() const
  {
    return *(m_heap.rbegin());
  }
    
  virtual
  void Accept( Visitor& visitor ) const
  {
    for ( typename set<PathWalk,Comparator>::const_iterator iter = m_heap.begin();
          iter != m_heap.end(); ++iter )
      visitor.Visit( *iter );
  }

  virtual
  void RemoveIf( const Filter& filter )
  {
    typename set<PathWalk,Comparator>::iterator pathIter, deadIter;
    pathIter = m_heap.begin();
    while( pathIter != m_heap.end() )
      if ( filter.Matches( *pathIter ) )
      {
        deadIter = pathIter;
        ++pathIter;
        if ( pathIter == m_heap.end() )
        {
          m_heap.erase( deadIter );
          pathIter = m_heap.end();
        }
        else
          m_heap.erase( deadIter );
      }
      else
        ++pathIter;
  }
};


// A visitor that extracts paths from a heap.

class HeapPathExtractor
  : public AbstractPathWalkHeap::Visitor
{
 public:
  HeapPathExtractor( set<KmerPath>* destination )
    : AbstractPathWalkHeap::Visitor(),
      mp_dest( destination ),
      m_copyClosedPaths( true ),
      m_copyOpenPaths( true )
  {}

  virtual
  ~HeapPathExtractor() {}

  void SetCopyClosedPaths( const bool value ) { m_copyClosedPaths = value; }
  void SetCopyOpenPaths( const bool value ) { m_copyOpenPaths = value; }

  virtual
  void Visit( const PathWalk& walk )
  {
    if ( walk.ContainsCloser() ) {
      if ( m_copyClosedPaths )
        mp_dest->insert( walk.GetPath() );
    }
    else {
      if ( m_copyOpenPaths )
        mp_dest->insert( walk.GetPath() );
    }
  }

 private:
  set<KmerPath>* mp_dest;
  bool m_copyClosedPaths;
  bool m_copyOpenPaths;
};

    
// A visitor that plots the walks on a heap.

class HeapPlotVisitor
  : public AbstractPathWalkHeap::Visitor
{
 public:
  HeapPlotVisitor( WalkPlotter* p_plotter )
    : AbstractPathWalkHeap::Visitor(),
      mp_plotter( p_plotter ),
      m_plotClosed( true ),
      m_closedType( WalkPlotter::CLOSED ),
      m_plotOpen( true ),
      m_openType( WalkPlotter::OPEN )
  {}

  virtual
  ~HeapPlotVisitor() {}

  void SetPlotClosedPaths( bool plot ) { m_plotClosed = plot; }
  void SetClosedType( WalkPlotter::WalkType type ) { m_closedType = type; }

  void SetPlotOpenPaths( bool plot ) { m_plotOpen = plot; }
  void SetOpenType( WalkPlotter::WalkType type ) { m_openType = type; }

  virtual 
  void Visit( const PathWalk& walk )
  {
    if ( walk.ContainsCloser() ) {
      if ( m_plotClosed )
        mp_plotter->Add( walk.GetPath(), m_closedType );
    }
    else {
      if ( m_plotOpen )
        mp_plotter->Add( walk.GetPath(), m_openType );
    }
  }

 private:
  WalkPlotter* mp_plotter;
  bool m_plotClosed;
  WalkPlotter::WalkType m_closedType;
  bool m_plotOpen;
  WalkPlotter::WalkType m_openType;
};


// A filter that returns true on open paths.

class OpenPathWalkFilter
  : public AbstractPathWalkHeap::Filter
{
 public:
  virtual
  ~OpenPathWalkFilter() {}

  virtual
  bool Matches( const PathWalk& walk ) const
  { 
    return ( ! walk.ContainsCloser() );
  }
};

#endif
