/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "CoreTools.h"
#include "math/Functions.h"
#include "math/HoInterval.h"
#include "graph/Digraph.h"
#include "paths/PairedPair.h"
#include "paths/PairedWalk.h"

class walkleafplus {

     public:

     walkleafplus( ) { }
     walkleafplus( int e, int v, int w, int eprev, int plen, Bool open ) 
          : e(e), v(v), w(w), eprev(eprev), plen(plen), open(open) { }

     int e;      // edge id (in the HyperKmerPath, not the walkleafplus graph)
     int v;      // vertex id at source of edge
     int w;      // vertex id at target of edge
     int eprev;  // previous edge (or -1 for root)
     int plen;   // length of path through e from root, in kmers
     Bool open;  // if open for further extension

     // delete?
     friend Bool operator<( const walkleafplus& l1, const walkleafplus& l2 )
     {    return l1.plen < l2.plen;    }

};

class placed_read {

     public:

     placed_read( ) { }
     placed_read( const int read_id, const int start_on_graph, 
          const int stop_on_graph, const int start_on_read, const int align_length,
          const int plen ) : read_id_(read_id), start_on_graph_(start_on_graph), 
          stop_on_graph_(stop_on_graph), start_on_read_(start_on_read), 
          align_length_(align_length), plen_(plen) { }

     int ReadId( ) const { return read_id_; }
     int PairId( ) const { return read_id_/2; }
     Bool Left( ) const { return read_id_ % 2 == 0; }
     Bool Right( ) const { return read_id_ % 2 == 1; }
     int StartOnGraph( ) const { return start_on_graph_; }
     int StopOnGraph( ) const { return stop_on_graph_; }
     int StartOnRead( ) const { return start_on_read_; }
     int StopOnRead( ) const { return start_on_read_ + align_length_; }
     int AlignLength( ) const { return align_length_; }
     int RootToStop( ) const { return plen_; }

     Bool Dead( ) const { return read_id_ < 0; }
     Bool Alive( ) const { return read_id_ >= 0; }
     void Kill( ) { read_id_ = -1; }

     void SetStopOnGraph( int s ) { stop_on_graph_ = s; }
     void SetAlignLength( int l ) { align_length_ = l; }
     void SetRootToStop( int d ) { plen_ = d; }

     private:

     int read_id_;        // in [ 0, 2*ppp.size )
     int start_on_graph_; // vertex id in graph
     int stop_on_graph_;  // vertex id in graph
     int start_on_read_;  // in [ 0, read.size )
     int align_length_;   // number of aligned positions
     int plen_;           // length of path from root through read, in kmers

};

void PairedWalkRight( const vec<pp_pair> ppp, const vec<Bool>& to_walk,
     const vec<pp_closure>& templates, const int max_walk, const vec<int>& L, 
     const double dmult, const vec< vec<pp_closure> >& known_closures,
     vec< vec<pp_closure> >& closures, vec<pp_read>& chunks, 
     const int max_opens, Bool create_closures_if_fail, vec<Bool>& fail, 
     int verbosity, int max_nodes )
{    
     // Sanity checks, etc.

     double clock = 0.0;
     if ( verbosity >= 1 ) clock = WallClockTime( );
     if ( verbosity >= 2 ) cout << "Entering PairedWalkRight\n";
     ForceAssert( to_walk.empty( ) ^ templates.empty( ) );
     if ( to_walk.nonempty( ) ) ForceAssertEq( ppp.size( ), to_walk.size( ) );
     int N = ( to_walk.nonempty( ) ? ppp.size( ) : templates.size( ) );
     if ( to_walk.nonempty( ) ) fail.resize_and_set( N, False );
     closures.clear_and_resize( ppp.size( ) );
     vec< vec<int> > chunk_starts( L.size( ) );
     for ( int i = 0; i < chunks.isize( ); i++ )
          chunk_starts[ chunks[i][0] ].push_back(i);

     // Index reads by where they start.

     vec< vec<int> > starts( L.size( ) );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    for ( int pass = 0; pass < 2; pass++ )
          {    const pp_read& r = ( pass == 0 ? ppp[i].Left( ) : ppp[i].Right( ) );
               int read_id = 2*i + pass;
               starts[ r[0] ].push_back(read_id);    }    }

     // Get gap bounds.

     vec<int> lo_d( ppp.size( ) ), hi_d( ppp.size( ) );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          lo_d[i] = int( ceil( p.Gap( ) - dmult * p.Dev( ) ) );
          hi_d[i] = int( floor( p.Gap( ) + dmult * p.Dev( ) ) );    }

     // For pairs with negative gaps, precompute the possible overlaps.

     vec< vec<int> > overlaps( ppp.size( ) );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          if ( lo_d[i] < 0 )
          {    static vec<int> offsets;
               GetOverlaps( p.Left( ), p.Right( ), offsets );
               for ( int j = 0; j < offsets.isize( ); j++ )
               {    int o = offsets[j];
                    int overlap = 0;
                    for ( int k = Max( 0, o ); k < p.LeftSize( ); k++ )
                         overlap += L[ p.Left(k) ];
                    for ( int k = 0; k < -o; k++ )
                         overlap += L[ p.Right(k) ];
                    double offby = Abs( double(overlap) + p.Gap( ) );
                    if ( offby <= dmult * p.Dev( ) )
                         overlaps[i].push_back(o);    }    }    }

     // Go through the pairs or templates.

     for ( int pi = 0; pi < N; pi++ )
     {    if ( to_walk.nonempty( ) && !to_walk[pi] ) continue;
          double this_clock = 0.0;
          if ( verbosity >= 2 ) 
          {    this_clock = WallClockTime( );
               cout << "\nTrying [" << pi << "] ";
               if ( to_walk.nonempty( ) ) cout << ppp[pi] << "\n";
               else cout << templates[pi] << "\n";    }
          pp_pair p;
          if ( to_walk.nonempty( ) ) p = ppp[pi];
          else p = pp_pair( templates[pi], pp_read( ), 0.0, 0.0 );
          int max_opens_seen = 0;

          // Find all read placements on p.Left( ).  Right reads are not placed
          // unless their left partners are placed consistently.  Also define initial
          // "right reachback read placement": first, any initially defined right 
          // read placement is a right reachback read placement; thence, 
          // recursively, any read placement that aligns to a right reachback read 
          // placement and extends it to the right is itself one.  We keep track of 
          // them by the vertices that they pass through.

          static vec<placed_read> places;
          static vec< vec<int> > places_by_read_id, places_by_graph_stop;
          places_by_read_id.clear_and_resize( 2 * ppp.size( ) );
          places_by_graph_stop.clear_and_resize( p.LeftSize( ) + 1 );
          places.clear( );
          static vec< vec<int> > right_reachback;
          right_reachback.clear_and_resize( p.LeftSize( ) + 1 );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    for ( int pass = 0; pass < 2; pass++ )
               {    const pp_read& r 
                         = ( pass == 0 ? ppp[i].Left( ) : ppp[i].Right( ) );
                    int read_id = 2*i + pass;
                    static vec<int> offsets;
                    GetOverlaps( p.Left( ), r, offsets );
                    for ( int j = 0; j < offsets.isize( ); j++ )
                    {    int o = offsets[j];
                         int start_on_read = Max( -o, 0 );
                         int stop_on_read = Min( r.isize( ), p.LeftSize( ) - o );
                         int start_on_graph = Max( o, 0 );
                         int align_length = stop_on_read - start_on_read;
                         int stop_on_graph = start_on_graph + align_length;

                         // For right reads, make sure that left read is placed
                         // consistently.

                         if ( pass == 1 )
                         {    Bool validated = False;
                              const vec<int>& lp = places_by_read_id[read_id-1]; 
                              int right_offset = o;
                              for ( int u = 0; u < lp.isize( ); u++ )
                              {    const placed_read& left = places[ lp[u] ];
                                   int left_offset = left.StartOnGraph( );
                                   if ( left_offset == 0 )
                                        left_offset = -left.StartOnRead( );
                                   if ( left.StopOnGraph( ) <= start_on_graph )
                                   {    int sep = 0;
                                        for ( int k = left.StopOnGraph( );
                                             k < start_on_graph; k++ )
                                        {    sep += L[ p.Left(k) ];    }
                                        if ( lo_d[i] <= sep && sep <= hi_d[i] )
                                             validated = True;    }
                                   else
                                   {    for ( int k = 0; k < overlaps[i].isize( );
                                             k++ )
                                        {    int lr_offset = overlaps[i][k];
                                             if ( left_offset + lr_offset
                                                  == right_offset )
                                             {    validated = True;
                                                  break;    }    }    }
                                   if (validated) break;    }
                              if ( !validated ) continue;    
                              for ( int k = start_on_graph; k <= stop_on_graph; k++ )
                              {    right_reachback[k].push_back( 
                                        places.size( ) );    }    }

                         // Record read placement.

                         places_by_read_id[read_id].push_back( places.size( ) );
                         places_by_graph_stop[stop_on_graph].push_back(
                              places.size( ) );
                         int root_to_stop = 0;
                         for ( int k = 0; k < stop_on_graph; k++ )
                              root_to_stop += L[ p.Left(k) ];
                         places.push( read_id, start_on_graph, stop_on_graph, 
                              start_on_read, align_length, 
                              root_to_stop );    }    }    }

          // Determine longest length of path (walkleafplus.plen) that we allow.

          int maxgap;
          if ( to_walk.nonempty( ) ) 
               maxgap = int( floor( p.Gap( ) + dmult * p.Dev( ) ) );
          else maxgap = max_walk;
          int leftsize = 0;
          for ( int i = 0; i < p.LeftSize( ); i++ )
               leftsize += L[ p.Left(i) ];
          int maxplen = maxgap + leftsize;

          // We keep a tree T as a digraphE.

          digraphE<walkleafplus> T;
          T.AddVertices(1);

          // Load p.Left onto the tree.
     
          int g = 0;
          for ( int i = 0; i < p.LeftSize( ); i++ )
          {    g += L[ p.Left(i) ];
               T.AddVertices(1);
               Bool open = ( i == p.LeftSize( ) - 1 && g <= maxplen );
               T.AddEdge( i, i+1, 
                    walkleafplus( p.Left(i), i, i+1, i-1, g, open ) );    }

          // Iteratively grow the tree.

          int lf = 0, opens = 1;
          while(1)
          {
               // Find an active leaf x.
     
               if ( max_nodes > 0 && T.N( ) > max_nodes )
               {    if ( to_walk.nonempty( ) ) fail[pi] = True;
                    goto finale;    }
               walkleafplus x;
               int l = 0;
               for ( lf++; lf < T.N( ); lf++ )
               {    const walkleafplus& x = T.EdgeObjectByIndexTo( lf, 0 );
                    if ( x.open ) break;    }
               if ( lf == T.N( ) ) break;
               l = lf;
               T.EdgeObjectByIndexToMutable( l, 0 ).open = False;
               x = T.EdgeObjectByIndexToMutable( l, 0 );
               int currente = T.EdgeObjectIndexByIndexTo( l, 0 );
               --opens;

               // Find the right read placements that stop at the current vertex
               // (x.w) and can't go further.

               static vec<int> stopped;
               stopped.clear( );
               for ( int i = 0; i < places_by_graph_stop[x.w].isize( ); i++ )
               {    const placed_read& p = places[ places_by_graph_stop[x.w][i] ];
                    if ( p.ReadId( ) % 2 == 0 ) continue;
                    const pp_read& r = ppp[ p.PairId( ) ].Right( );
                    if ( p.StopOnRead( ) == r.isize( ) ) stopped.push_back(i);    }

               // Find the right read placements that stop at the current vertex 
               // (x.w) and are such that the read goes further.

               static vec<int> nexts;
               nexts.clear( );
               for ( int i = 0; i < right_reachback[x.w].isize( ); i++ )
               {    const placed_read& p = places[ right_reachback[x.w][i] ];
                    int pid = p.ReadId( ) / 2;
                    const pp_read& r = ppp[pid].Right( );
                    if ( p.StopOnRead( ) == r.isize( ) ) continue;
                    int next = r[ p.StopOnRead( ) ];
                    nexts.push_back(next);    }
               UniqueSort(nexts);

               // Add nodes to graph.

               for ( int i = 0; i < nexts.isize( ); i++ )
               {    int vlast = l;
                    int plen = x.plen + L[ nexts[i] ];
                    int v = T.N( );
                    if ( verbosity >= 1 && v % 100000 == 0 )
                    {    cout << "v = " << v << ", opens = " << opens
                              << ", time used = " << TimeSince(clock) << endl;    }
                    T.AddVertices(1);
                    Bool open = ( plen <= maxplen );
                    T.AddEdge( vlast, v, 
                         walkleafplus( nexts[i], vlast, v, l, plen, open ) );
                    places_by_graph_stop.resize( places_by_graph_stop.size( ) + 1 );
                    right_reachback.resize( places_by_graph_stop.size( ) + 1 );
                    if (open) ++opens;    
                    vlast = v;
                    max_opens_seen = Max( max_opens_seen, opens );
                    if ( max_opens > 0 && opens > max_opens )
                    {    if ( to_walk.nonempty( ) ) fail[pi] = True;
                         goto finale;    }    }

               // Extend the alignments of the reads that stopped at the current
               // vertex.

               static vec<int> indices_to_remove;
               indices_to_remove.clear( );
               for ( int i = 0; i < places_by_graph_stop[x.w].isize( ); i++ )
               {    int iplace = places_by_graph_stop[x.w][i];
                    placed_read& p = places[iplace];
                    int read_id = p.ReadId( );
                    int pid = read_id/2;
                    const pp_read& r = ( read_id % 2 == 0 ? 
                         ppp[pid].Left( ) : ppp[pid].Right( ) );
                    if ( p.StopOnRead( ) == r.isize( ) ) continue;
                    indices_to_remove.push_back(i);
                    int next = r[ p.StopOnRead( ) ];
                    int inext = BinPosition( nexts, next );
                    if ( inext < 0 )
                    {    p.Kill( );
                         vec<int>& read_places = places_by_read_id[read_id];
                         for ( int j = 0; j < read_places.isize( ); j++ )
                         {    if ( read_places[j] == iplace )
                              {    for ( int k = j + 1; k < read_places.isize( );
                                        k++ )
                                   {    read_places[k-1] = read_places[k];    }
                                   read_places.pop_back( );
                                   break;    }    }    }
                    else
                    {    int v = T.N( ) - nexts.isize( ) + inext;
                         p.SetStopOnGraph(v);
                         p.SetAlignLength( p.AlignLength( ) + 1 );    
                         places_by_graph_stop[v].push_back(iplace);    
                         p.SetRootToStop( p.RootToStop( ) + L[next] );    
                         if ( p.Right( ) 
                              && BinMember( right_reachback[x.w], iplace ) )
                         {    right_reachback[v].push_back(iplace);    }    }    }
               EraseTheseIndices( places_by_graph_stop[x.w], indices_to_remove );

               // Find new read placements that start at the newly added vertices.
               // Left reads are directly added, right reads are treated as 
               // candidates.

               static vec< pair<int,int> > rcand;
               rcand.clear( );
               for ( int i = 0; i < nexts.isize( ); i++ )
               {    int u = nexts[i];
                    int plen = x.plen + L[u];
                    int v = T.N( ) - nexts.isize( ) + i;

                    // First place left reads.

                    for ( int j = 0; j < starts[u].isize( ); j++ )
                    {    int read_id = starts[u][j];
                         if ( read_id % 2 == 1 ) continue;
                         places_by_read_id[read_id].push_back( places.size( ) );
                         places_by_graph_stop[v].push_back( places.size( ) );
                         places.push( read_id, l, v, 0, 1, plen );    }

                    // Find candidates for right reads to be placed.

                    for ( int j = 0; j < starts[u].isize( ); j++ )
                    {    int read_id = starts[u][j];
                         if ( read_id % 2 == 0 ) continue;

                         // In cases where the gap could be negative, go ahead and
                         // place the read, if appropriate.

                         const pp_pair& p = ppp[read_id/2];
                         if ( overlaps[read_id/2].nonempty( ) )
                         {
                              // Figure out if the left read is placed all the way
                              // to the end.

                              Bool good_left = False;
                              for ( int k = 0; k < places_by_graph_stop[v].isize( );
                                   k++ )
                              {    int iplace = places_by_graph_stop[v][k];
                                   const placed_read& pv = places[iplace];
                                   if ( pv.ReadId( ) != read_id - 1 ) continue;
                                   int offset = pv.StopOnRead( ) - 1;
                                   if ( BinMember( overlaps[read_id/2], offset ) )
                                   {    good_left = True;
                                        break;    }    }
                              if (good_left)
                              {    places_by_read_id[read_id].push_back( 
                                        places.size( ) );
                                   places_by_graph_stop[v].push_back( 
                                        places.size( ) );
                                   for ( int m = 0; m < right_reachback[l].isize( );
                                        m++ )
                                   {    int mplace = right_reachback[l][m];
                                        const placed_read& pm = places[mplace];
                                        if ( pm.StopOnGraph( ) != v ) continue;
                                        const pp_pair& px = ppp[ pm.ReadId( )/2 ];
                                        if ( p.RightSize( ) - 2
                                             + pm.StopOnRead( ) < px.RightSize( ) )
                                        {    continue;    }
                                        Bool mismatch = False;
                                        for ( int a = 1; a < p.RightSize( ); a++ )
                                        {    int b = a - 1 + pm.StopOnRead( );
                                             if ( b >= px.RightSize( ) ) break;
                                             if ( p.Right(a) != px.Right(b) )
                                             {    mismatch = True;
                                                  break;    }    }
                                        if (mismatch) continue;
                                        right_reachback[v].push_back( 
                                             places.size( ) );    }
                                   places.push( read_id, l, v, 0, 1, plen );
                                   continue;    }    }

                         // Find candidates.

                         rcand.push_back( make_pair( read_id, v ) );    }    }

               // See which right read candidates have matching left reads
               // placed appropriately on the graph.

               int dist = 0;
               int current_edge = currente;
               static vec<int> rcand1;
               rcand1.clear( );
               for ( int j = 0; j < rcand.isize( ); j++ )
                    rcand1.push_back( rcand[j].first );
               int max_gap = 0;
               for ( int j = 0; j < rcand1.isize( ); j++ )
                    max_gap = Max( max_gap, hi_d[ rcand1[j]/2 ] );
               while(1)
               {    if ( rcand.empty( ) ) break;
                    const walkleafplus& e = T.EdgeObject(current_edge);
                    int w = e.w;
                    for ( int m = 0; m < places_by_graph_stop[w].isize( ); m++ )
                    {    const placed_read& left 
                              = places[ places_by_graph_stop[w][m] ];
                         int left_id = left.ReadId( );
                         if ( left_id % 2 != 0 ) continue;
                         int pid = left_id/2;
                         if ( left.StopOnRead( ) != ppp[pid].LeftSize( ) ) continue;
                         const pp_pair& p = ppp[pid];
                         int gap = x.plen - left.RootToStop( );
                         if ( gap < lo_d[pid] || gap > hi_d[pid] ) continue;
                         int j = BinPosition( rcand1, left_id + 1 );
                         if ( j < 0 ) continue;
                         int right_id = rcand[j].first;
                         int v = rcand[j].second;
                         places_by_read_id[right_id].push_back( places.size() );
                         places_by_graph_stop[v].push_back( places.size( ) );
                         for ( int m = 0; m < right_reachback[l].isize( ); m++ )
                         {    int mplace = right_reachback[l][m];
                              const placed_read& pm = places[mplace];
                              if ( pm.StopOnGraph( ) != v ) continue;
                              const pp_pair& px = ppp[ pm.ReadId( )/2 ];
                              if ( p.RightSize( ) - 2
                                   + pm.StopOnRead( ) < px.RightSize( ) )
                              {    continue;    }
                              Bool mismatch = False;
                              for ( int a = 1; a < p.RightSize( ); a++ )
                              {    int b = a - 1 + pm.StopOnRead( );
                                   if ( b >= px.RightSize( ) ) break;
                                   if ( p.Right(a) != px.Right(b) )
                                   {    mismatch = True;
                                        break;    }    }
                              if (mismatch) continue;
                              right_reachback[v].push_back( places.size( ) );    }
                         int plen = x.plen + L[ p.Right(0) ];
                         places.push( right_id, l, v, 0, 1, plen );
                         rcand.Erase( j, j+1 ), rcand1.Erase( j, j+1 );
                         if ( rcand.empty( ) ) break;    }
                    dist += L[e.e];
                    if ( dist > max_gap || T.To(e.v).empty( ) ) break;
                    current_edge = T.EdgeObjectIndexByIndexTo( e.v, 0 );    }    }

          // Get the closures.

          finale:
          if ( to_walk.nonempty( ) && fail[pi] && !create_closures_if_fail ) 
          {    cout << "\nmax_opens exceeded\n";
               cout << "v = " << T.N( ) << ", opens = " << max_opens_seen << "\n";
               continue;    }
          vec<pp_read> paths;
          vec<int> plens;
          for ( int l = 0; l < T.N( ); l++ )
          {    if ( T.From(l).empty( ) )
               {    if ( to_walk.empty( ) )
                    {    const walkleafplus& x = T.EdgeObjectByIndexTo( l, 0 );
                         if ( x.open ) continue;    }
                    static pp_read q;
                    q.clear( );
                    walkleafplus y = T.EdgeObjectByIndexTo( l, 0 );
                    int plen = y.plen;
                    int m = l;
                    while(1)
                    {    q.push_back(y.e);
                         m = y.v;
                         if ( m == 0 ) break;
                         y = T.EdgeObjectByIndexTo( m, 0 );    }
                    q.ReverseMe( );
                    paths.push_back(q);
                    plens.push_back(plen);    }    }
          if ( verbosity == 2 ) cout << paths.size( ) << " paths found\n";
          if ( verbosity >= 4 )
          {    cout << paths.size( ) << " paths found:\n";
               for ( int i = 0; i < paths.isize( ); i++ )
                    cout << "[" << i << "] " << paths[i] << "\n";    }
          int nclosures = 0;
          cout << "\nfound " << paths.size( ) << " paths\n"; // XXXXXXXXXXXXXXXXXXXX
          for ( int i = 0; i < paths.isize( ); i++ )
          {    const pp_read& P = paths[i];
               if ( to_walk.nonempty( ) )
               {    int plen = plens[i];
                    pp_pair q( P, p.Right( ), p.Gap( ) - plen + leftsize, p.Dev( ) );
                    static vec<pp_closure> qclosures;
                    GetClosures( q, L, dmult, qclosures );
                    closures[pi].append(qclosures);    }
               else
               {    
                    cout << "\npath " << i << "\n"; // XXXXXXXXXXXXXXXXXXXXXXXXXXXXX
                    cout << P << "\n"; // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

                    // Find read placements on P.

                    vec< vec<int> > Pstart( 2* ppp.size( ) );
                    for ( int j = 0; j < P.isize( ); j++ )
                    {    for ( int k = 0; k < starts[ P[j] ].isize( ); k++ )
                         {    int id = starts[ P[j] ][k];
                              const pp_read& r = ( id % 2 == 0 ? ppp[id/2].Left( )
                                   : ppp[id/2].Right( ) );
                              if ( P.Contains( r, j ) ) 
                                   Pstart[id].push_back(j);    }    }

                    cout << "\nCoverage of [0," << P.size( ) << "):\n"; // XXXXXXXXX

                    // Find pair placements.  Mark intervals covered by known
                    // closures.

                    vec<ho_interval> knowns;
                    int max_stop = 0;
                    for ( int j = 0; j < ppp.isize( ); j++ )
                    {    for ( int k1 = 0; k1 < Pstart[ 2*j ].isize( ); k1++ )
                         {    int start1 = Pstart[ 2*j ][k1];
                              for ( int k2 = 0; 
                                   k2 < Pstart[ 2*j + 1 ].isize( ); k2++ )
                              {    int start2 = Pstart[ 2*j + 1 ][k2];
                                   int gap = 0;
                                   int stop1 = start1 + ppp[j].LeftSize( );
                                   int stop2 = start2 + ppp[j].RightSize( );
                                   if ( stop1 <= start2 )
                                   {    for ( int u = stop1; u < start2; u++ )
                                             gap += L[ P[u] ];    }
                                   else
                                   {    for ( int u = start2; u < stop1; u++ )
                                             gap -= L[ P[u] ];    }
                                   if ( Abs( gap - ppp[j].Gap( ) ) 
                                        > dmult * ppp[j].Dev( ) )
                                   {    continue;    }
                                   int start = Min( start1, start2 );
                                   int stop = Max( stop1, stop2 );
                                   max_stop = Max( stop, max_stop );
                                   pp_closure C;
                                   C.SetToRangeOf( P, start, stop );
                                   cout << "[" << start << "," << stop << ")";
                                   Bool known = False;
                                   for ( int u = 0; 
                                        u < known_closures[j].isize( ); u++ )
                                   {    if ( known_closures[j][u] == C )
                                        {    known = True;
                                             knowns.push( start, stop );
                                             break;    }    }
                                   if (known) cout << " *";
                                   cout << "            [" << start1 << "," << stop1
                                        << ") --> " << "[" << start2 << ","
                                        << stop2 << ")\n";
                                   // closures[j].push_back(C);   
                                   // ++nclosures;    
                                                      }    }    }    

                    // Mark intervals covered by known chunks.

                    for ( int j = 0; j < P.isize( ); j++ )
                    {    for ( int m = 0; m < chunk_starts[ P[j] ].isize( ); m++ )
                         {    int chid = chunk_starts[ P[j] ][m];
                              if ( P.Contains( chunks[chid], j ) )
                              {    knowns.push( j, 
                                        j + chunks[chid].isize( ) );    }    }    }
                    UniqueSort(knowns);
                    cout << "known coverage:\n";
                    for ( int j = 0; j < knowns.isize( ); j++ )
                    {    pp_read r;
                         r.SetToRangeOf( P, knowns[j].Start( ),
                              knowns[j].Stop( ) );
                         cout << knowns[j] << " " << r << "\n";    }

                    // Determine if P is completely covered by known stuff, with
                    // 100-kmer overlaps.

                    if ( knowns.nonempty( ) && knowns[0].Start( ) == 0 )
                    {    vec<ho_interval> knownsb;
                         int N = 0;
                         for ( int j = 0; j < P.isize( ); j++ )
                              N += L[ P[j] ];
                         for ( int i = 0; i < knowns.isize( ); i++ )
                         {    ho_interval h = knowns[i];
                              int n0 = 0, n1 = 0;
                              for ( int j = 0; j < h.Start( ); j++ )
                                   n0 += L[ P[j] ];
                              for ( int j = h.Start( ); j < h.Stop( ); j++ )
                                   n1 += L[ P[j] ];
                              knownsb.push( n0, n0 + n1 );    }
                         int high = knownsb[0].Stop( );
                         for ( int i = 1; i < knownsb.isize( ); i++ )
                         {    if ( knownsb[i].Start( ) > high - 100 ) break;
                              high = Max( high, knownsb[i].Stop( ) );    }
                         if ( high == N ) continue;    }

                    // Find the part of the left end that is covered by known
                    // closures and chunks, requiring an overlap of 100 kmers, then 
                    // hack this off, leaving a 100-kmer tail.

                    const int min_over = 100;
                    vec<ho_interval> knownsb, un;
                    int N = 0;
                    for ( int j = 0; j < P.isize( ); j++ )
                         N += L[ P[j] ];
                    PRINT(N); // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
                    for ( int i = 0; i < knowns.isize( ); i++ )
                    {    ho_interval h = knowns[i];
                         int n0 = 0, n1 = 0;
                         for ( int j = 0; j < h.Start( ); j++ )
                              n0 += L[ P[j] ];
                         for ( int j = h.Start( ); j < h.Stop( ); j++ )
                              n1 += L[ P[j] ];
                         if ( n1 <= min_over ) continue;
                         knownsb.push( n0, n0 + n1 + 1 - min_over );    }
                    Uncovered( N, knownsb, un );
                    int M = 0, f, first = un[0].Start( ) - 1;
                    for ( f = 0; f < P.isize( ); f++ )
                    {    if ( M >= first ) break;
                         M += L[ P[f] ];    }
                    PRINT3( M, f, first ); // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
                    pp_read Pt;
                    Pt.SetToRangeOf( P, f-1, P.size( ) );
                    cout << "\nleft truncation of path:\n" << Pt << "\n";    

                    // Chunk truncated path.  Idea is that chunks should have size
                    // >= 200 kmers, and should overlap by >= 100 kmers. 

                    vec<ho_interval> new_chunks;
                    for ( int i = 0; i < Pt.isize( ); i++ )
                    {    int j, sum = L[ Pt[i] ], sum1 = 0;
                         for ( j = i+1; j < Pt.isize( ); j++ )
                         {    sum += L[ Pt[j] ];
                              sum1 += L[ Pt[j] ];
                              if ( sum >= 200 && sum1 >= 100 ) break;    }
                         if ( j == Pt.isize( ) )
                         {    if ( new_chunks.empty( ) )
                                   new_chunks.push( 0, Pt.size( ) );
                              else new_chunks.back( ).SetStop( Pt.size( ) );    }
                         else
                         {    int k, over = 0;
                              for ( k = i - 1; k >= 0; k-- )
                              {    over += L[ Pt[k] ];
                                   if ( over >= 100 ) break;    }
                              if ( k < 0 ) k = 0;
                              new_chunks.push( k, j + 1 );    }
                         i = j;    }
                    for ( int i = 0; i < new_chunks.isize( ); i++ )
                    {    pp_read ch;
                         ch.SetToRangeOf( Pt, new_chunks[i].Start( ),
                              new_chunks[i].Stop( ) );
                         if ( ch.Length(L) < 200 ) continue;
                         chunk_starts[ ch[0] ].push_back( chunks.size( ) );
                         chunks.push_back(ch);
                         cout << "new chunk: " << new_chunks[i] 
                              << " " << ch << "\n";    }

                    }    }

          if ( to_walk.nonempty( ) ) 
          {    nclosures = closures[pi].size( );
               Sort( closures[pi] );    }
          if ( verbosity >= 1 ) 
          {    cout << "v = " << T.N( ) << ", opens = " << max_opens_seen
                    << ", found " << nclosures << " closures\n";    }    
          if ( to_walk.nonempty( ) && verbosity >= 3 )
          {    for ( int i = 0; i < closures[pi].isize( ); i++ )
                    cout << "[" << i << "] " << closures[pi][i] << "\n";   
               cout << "\n";    }
          if ( verbosity >= 2 )
          {    cout << "PairedWalkRight time used = " << TimeSince(this_clock) 
                    << "\n";    }    }

     // Done.

     if ( verbosity >= 1 )
     {    cout << "PairedWalkRight total time used = " << TimeSince(clock) 
               << "\n";    }    
     for ( int i = 0; i < closures.isize( ); i++ )
          UniqueSort( closures[i] );
     if ( to_walk.empty( ) )
     {    cout << "total closures found = " << SizeSum(closures) << "\n";    }    }
