/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// Experimental tools for generating and manipulating a graph whose vertices are 
// pp_pairs p.  We declare that there is an edge from p1 to p2 if at least one of
// the following conditions is satisfied:
//
// 1. The two pairs overlap properly on both sides, consistent with their 
// separations, and p2 extends p1 to the right, on at least one side.
// 2. The pair p1 is a "closed pair", i.e. its left and right sides agree and its
// gap is negative with deviation zero, p2L overlaps p1 by at least 100 kmers, 
// extending up to the right end of p1, and either p2L extends beyond the end of p1 
// on the right, or else the right end of p2R would extend beyond the end of p1,
// even if p2R were moved to the left by dmult standard deviations.
// 3. Same as #2, but roles of p1 and p2 switched and directionality reversed.
//
// ***** NO NO NO.  Only #1 is used now. *****
//
// In cases where there are multiple overlaps between p1 and p2, only one edge is
// generated.
//
// Everything here is in flux.

#ifndef PAIR_GRAPH_H
#define PAIR_GRAPH_H

#include "CoreTools.h"
#include "Feudal.h"
#include "graph/Digraph.h"
#include "paths/PairedPair.h"
#include "paths/simulation/Placement.h"

void MakeGraphFromPairs( const vec<pp_pair>& ppp, const vec<int>& L,
     const double dmult, digraph& PG );

void OutputPairsGraph( ostream& out, digraph& PG );

// OutputTruthLabeledPairsGraph.  Use truth data to mark and print the pairs graph,
// according to the following rules:
// if a vertex (i.e. a pair) is alignable to the internal neighborhood, label black;
// if a vertex is not alignable to the internal neighborhood but is alignable to the
// full neighborhood, label it magenta;
// if a vertex is not alignable to the full neighborhood but is alignable to the
// genome, label it yellow;
// if a vertex is not alignable to the genome at all, label it red.
// In addition, for each edge, form a new pair by merging, and color the edge using
// the same rules.  If an edge has multiple instantiations as a pair, use the
// precedence black > magenta > yellow > red.

void OutputTruthLabeledPairsGraph( ostream& out, digraph& PG, 
     const vec<pp_pair>& ppp, const vec<int>& L, const double dmult,
     const HyperKmerPath& h, const KmerBaseBroker& kbb, const vecbasevector& genome, 
     const int NHOOD_RADIUS, const int NHOOD_RADIUS_INTERNAL,
     const serfvec<placement>& locsv, const vec<int>& genome_path_lengths );

class pair_placement {

     public:

     pair_placement( ) { }
     pair_placement( int pair_id, int gid, int start1, int stop1, int start2,
          int stop2, Bool rc ) : pair_id(pair_id), gid(gid), start1(start1), 
          stop1(stop1), start2(start2), stop2(stop2), rc(rc) { }

     int pair_id;
     int gid;           // genome contig
     int start1, stop1;
     int start2, stop2;
     Bool rc;

     int Length( ) const { return stop2 - start1; }

     friend Bool operator<( const pair_placement& p1, const pair_placement& p2 )
     {    if ( p1.gid < p2.gid ) return True;
          if ( p1.gid > p2.gid ) return False;
          return p1.start1 < p2.start1;    }

     friend ostream& operator<<( ostream& out, const pair_placement& p )
     {    return out << "[" << p.pair_id << "] " << p.gid << "." << p.start1
               << "-" << p.stop2 << " " << ( p.rc ? "rc" : "fw" ) 
               << " [" << p.Length( ) << "]";    }

};

void PrintPairLocs( const Bool print, const String& title, const String& sub_dir, 
     const vec<pp_pair>& ppp, const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     const vecbasevector& genome, const int NHOOD_RADIUS, 
     const int NHOOD_RADIUS_INTERNAL, const serfvec<placement>& locsv,
     const vec<Bool>& remove = vec<Bool>( ), vec<Bool>* ppp_valid = 0 );

int GetAndIncrementPairlocsStage( );

#endif
