/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Alignment.h"
#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "math/Functions.h"
#include "math/HoInterval.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/KmerPathOnHyper.h"
#include "paths/PairGraph.h"
#include "paths/PairedPair.h"
#include "paths/simulation/Placement.h"

class pair_join {

     public:

     pair_join( ) { }
     pair_join( int j1, int j2, const pp_pair& p ) : j1(j1), j2(j2), p(p) { }

     int j1, j2;         // indices of the pairs
     pp_pair p;          // the joined pair

};

String GraphColor( int c )
{    ForceAssertLe( c, 3 );
     if ( c == 0 ) return "black";
     else if ( c == 1 ) return "magenta";
     else if ( c == 2 ) return "yellow";
     else return "red";    }

void ValidatePairs( const vec<pp_pair>& ppp, const HyperKmerPath& h, 
     const KmerBaseBroker& kbb, const vecbasevector& genome, const int NHOOD_RADIUS, 
     const int NHOOD_RADIUS_INTERNAL, const serfvec<placement>& locsv, 
     const vec<int>& genome_path_lengths, vec<int>& color )
{    color.resize_and_set( ppp.size( ), 3 );
     for ( int pass = 1; pass <= 3; pass++ )
     {    for ( int t = 0; t < locsv.size( ); t++ )
          {    vec<int> ppp_index;
               vec<pp_pair> ppp_part;
               for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( color[i] >= pass )
                    {    ppp_index.push_back(i);
                         ppp_part.push_back( ppp[i] );    }    }
               static vecbasevector b;
               if ( pass == 1 || pass == 2 )
               {    const placement& p = locsv[t];
                    int g = p.GenomeId( );
                    int radius
                         = ( pass == 1 ? NHOOD_RADIUS_INTERNAL : NHOOD_RADIUS );
                    int start = Max( 0, p.pos( ) - radius );
                    int stop = Min( genome_path_lengths[g], p.Pos( ) + radius );
                    b.resize(1);
                    b[0].SetToSubOf( genome[g], start, stop - start );    }
               else b = genome;
               vec<Bool> ppp_part_valid;
               static vec<Bool> remove;
               PrintPairLocs( False, "", "", ppp_part, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove, &ppp_part_valid );
               for ( int i = 0; i < ppp_part_valid.isize( ); i++ )
               {    if ( ppp_part_valid[i] )
                         color[ ppp_index[i] ] = pass - 1;    }    }    }    }

static int pairlocs_stage(0);

int GetAndIncrementPairlocsStage( )
{    ++pairlocs_stage;
     return pairlocs_stage - 1;    }

void PrintPairLocs( const Bool print, const String& title, const String& sub_dir,
     const vec<pp_pair>& ppp, const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     const vecbasevector& genome, const int NHOOD_RADIUS, 
     const int NHOOD_RADIUS_INTERNAL, const serfvec<placement>& locsv,
     const vec<Bool>& remove, vec<Bool>* pppvalid )
{    if (print)
     {    Mkdir777( sub_dir + "/pairlocs" );
          if ( pairlocs_stage == 0 ) 
               System( "/bin/rm -f " + sub_dir + "/pairlocs/*" );    }
     String outfile = sub_dir + "/pairlocs/" + ToString(pairlocs_stage);

     // Copy or link easy graph.

     if (print)
     {    String graph_outfile 
               = sub_dir + "/pairlocs/" + ToString(pairlocs_stage) + ".graph";
          String last_graph_outfile 
               = sub_dir + "/pairlocs/" + ToString(pairlocs_stage-1) + ".graph";
          String current_graph = sub_dir + "/current_easy_graph";
          Bool linked = False;
          if ( pairlocs_stage > 0 )
          {    int cmp 
                    = System( "cmp -s " + current_graph + " " + last_graph_outfile );
               if ( cmp == 0 )
               {    if ( IsSymbolicLink(last_graph_outfile) )
                    {    System( "cp -d " + last_graph_outfile 
                              + " " + graph_outfile );    }
                    else Symlink( last_graph_outfile, graph_outfile );
                    linked = True;    }    }
          if ( !linked ) Cp2( current_graph, graph_outfile );    }

     String report = "STAGE " + ToString(pairlocs_stage) + ", " + title + "\n\n";
     Ofstream( out, ( print ? outfile : "/dev/null" ) );
     if (print)
     {    cout << "\n" << report;
          flush(cout);
          out << report;
          pairlocs_stage++;    }
     vec<pair_placement> pair_places;
     vec<int> L( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          L[i] = h.EdgeLength(i);
     static vecbasevector last_genome;
     static HyperBasevector last_hyperbasevector;
     static vec<pp_pair> last_pairs;
     static vec<pair_placement> last_places;
     static vec<int> last_color;
     static vec<Bool> last_remove;
     HyperBasevector hb( h, kbb );
     vec<int> color( ppp.size( ), 3 );

     // Recover cached results if possible.

     vec<Bool> precomputed( ppp.size( ), False );
     if ( genome == last_genome && hb == last_hyperbasevector )
     {    vec<int> id( last_pairs.size( ), vec<int>::IDENTITY );
          SortSync( last_pairs, id );
          vec< vec<int> > last_places_index( last_pairs.size( ) );
          for ( int j = 0; j < last_places.isize( ); j++ )
          {    if ( last_places[j].pair_id < 0 ) continue;
               last_places_index[ last_places[j].pair_id ].push_back(j);    }
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( remove.nonempty( ) && remove[j] ) continue;
               int x = BinPosition( last_pairs, ppp[j] );
               if ( x < 0 ) continue;
               int m = id[x];
               if ( last_remove.nonempty( ) && last_remove[m] ) continue;
               precomputed[j] = True;
               for ( int u = 0; u < last_places_index[m].isize( ); u++ )
               {    pair_placement p = last_places[ last_places_index[m][u] ];
                    p.pair_id = j;
                    pair_places.push_back(p);
                    color[j] = last_color[m];    }    }    }
     last_genome = genome;
     last_hyperbasevector = hb;
     last_pairs = ppp;
     last_remove = remove;

     // Go through three passes.

     for ( int pass = 1; pass <= 3; pass++ )
     {    for ( int t = 0; t < locsv.size( ); t++ )
          {    vec<int> ppp_index;
               vec<pp_pair> ppp_part;
               for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( remove.nonempty( ) && remove[i] ) continue;
                    if ( !precomputed[i] && color[i] >= pass - 1 )
                    {    ppp_index.push_back(i);
                         ppp_part.push_back( ppp[i] );    }    }
               static vecbasevector b;
               int g = -1, start = -1;
               if ( pass == 1 || pass == 2 )
               {    const placement& p = locsv[t];
                    g = p.GenomeId( );
                    int radius
                         = ( pass == 1 ? NHOOD_RADIUS_INTERNAL : NHOOD_RADIUS );
                    start = Max( 0, p.pos( ) - radius );
                    int stop = Min( int( genome[g].size( ) + 1 - h.K( ) ), 
                         p.Pos( ) + radius );
                    b.resize(1);
                    b[0].SetToSubOf( genome[g], start, stop - start );    }
               else b = genome;
               vec<Bool> ppp_part_valid( ppp_part.size( ), False );
               vec< vec<int> > edges;
               vec< vec<alignment_plus> > ealigns;
               vecbasevector edgesb;
               for ( int i = 0; i < ppp_part.isize( ); i++ )
                    edges.push_back( ppp_part[i].Left( ), ppp_part[i].Right( ) );
               UniqueSort(edges);
               ealigns.resize( edges.size( ) );
               for ( int i = 0; i < edges.isize( ); i++ )
               {    static KmerPath p;
                    p.Clear( );
                    for ( int j = 0; j < edges[i].isize( ); j++ )
                         p.Append( h.EdgeObject( edges[i][j] ) );
                    edgesb.push_back_reserve( kbb.Seq(p) );    }
               edgesb.Append(b);
               PerfectAligner pal( h.K( ), PerfectAligner::findProperOnly );
               vec<alignment_plus> aligns;
               pal.Align( edgesb, aligns, edges.size( ) );
               for ( int i = 0; i < aligns.isize( ); i++ )
               {    const alignment_plus& ap = aligns[i];
                    int id1 = ap.Id1( ), id2 = ap.Id2( );
                    if ( ap.a.pos1( ) > 0 || ap.a.Pos1( ) < edgesb[id1].isize( ) ) 
                         continue;
                    ealigns[id1].push_back(ap);    }    
               for ( int i = 0; i < ppp_part.isize( ); i++ )
               {    const pp_pair& p = ppp_part[i];
                    int i1 = BinPosition( edges, (vec<int>) p.Left( ) );
                    int i2 = BinPosition( edges, (vec<int>) p.Right( ) );
                    for ( int j1 = 0; j1 < ealigns[i1].isize( ); j1++ )
                    {    const alignment_plus& ap1 = ealigns[i1][j1];
                         for ( int j2 = 0; j2 < ealigns[i2].isize( ); j2++ )
                         {    const alignment_plus& ap2 = ealigns[i2][j2];
                              if ( ap1.Id2( ) != ap2.Id2( ) ) continue;
                              if ( ap1.Rc2( ) != ap2.Rc2( ) ) continue;
                              int sep;
                              if ( !ap1.Rc2( ) ) sep = ap2.a.pos2( ) - ap1.a.Pos2( );
                              else sep = ap1.a.pos2( ) - ap2.a.Pos2( );
                              double psep = ppp_part[i].Gap( ) - h.K( ) + 1;
                              if ( Abs( double(sep) - psep ) 
                                   > 4.0 * ppp_part[i].Dev( ) )
                              {    continue;    }
                              ppp_part_valid[i] = True;
                              int pair_id = ppp_index[i];
                              Bool rc = ap1.Rc2( );
                              int gid = ( pass <= 2 ? g : ap1.Id2( ) );
                              int Start1, Stop1, Start2, Stop2;
                              if ( !rc ) 
                              {    Start1 = start + ap1.pos2( );
                                   Stop1 = start + ap1.Pos2( );
                                   Start2 = start + ap2.pos2( );
                                   Stop2 = start + ap2.Pos2( );    }
                              else
                              {    Start1 = start + ap2.pos2( );
                                   Stop1 = start + ap2.Pos2( );
                                   Start2 = start + ap1.pos2( );
                                   Stop2 = start + ap1.Pos2( );    }
                              pair_placement pp( pair_id, gid, Start1, Stop1, 
                                   Start2, Stop2, rc );
                              pair_places.push_back(pp);    }    }    }
               for ( int i = 0; i < ppp_part_valid.isize( ); i++ )
               {    if ( ppp_part_valid[i] )
                         color[ ppp_index[i] ] = pass - 1;    }    }    }
     vec< vec<ho_interval> > covered( genome.size( ) ), covered0( genome.size( ) );
     int K = h.K( );
     for ( int i = 0; i < pair_places.isize( ); i++ )
     {    pair_placement p = pair_places[i];
          p.start1 += K/2;
          p.start2 += K/2;
          p.stop1 -= K/2;
          p.stop2 -= K/2;
          covered[ p.gid ].push_back( ho_interval( p.start1, p.stop2 ) );
          covered0[ p.gid ].push_back( ho_interval( p.start1, p.stop1 ) );
          covered0[ p.gid ].push_back( ho_interval( p.start2, p.stop2 ) );    }
     for ( int g = 0; g < genome.size( ); g++ )
     {    vec<ho_interval> uncovered;
          int N = genome[g].size( );
          if ( covered[g].empty( ) ) continue;
          Uncovered( N, covered[g], uncovered );
          covered0[g].append(uncovered);    }
     for ( int g = 0; g < genome.size( ); g++ )
     {    vec<ho_interval> uncovered, uncovered0;
          int N = genome[g].size( );
          if ( covered[g].empty( ) ) continue;
          Uncovered( N, covered[g], uncovered );
          Uncovered( N, covered0[g], uncovered0 );
          for ( int j = 0; j < uncovered.isize( ); j++ )
          {    const ho_interval& u = uncovered[j];
               if ( u.Start( ) == 0 || u.Stop( ) == N ) continue;
               pair_places.push_back( pair_placement( 
                    -1, g, u.Start( ), u.Stop( ), u.Start( ), u.Stop( ), 
                    False ) );    }
          for ( int j = 0; j < uncovered0.isize( ); j++ )
          {    const ho_interval& u = uncovered0[j];
               if ( u.Start( ) == 0 || u.Stop( ) == N ) continue;
               pair_places.push_back( pair_placement( 
                    -2, g, u.Start( ), u.Stop( ), u.Start( ), u.Stop( ), 
                    False ) );    }    }
     if ( pppvalid != 0 )
     {    pppvalid->resize( ppp.size( ), True );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove.nonempty( ) && remove[i] ) continue;
               if ( color[i] == 3 ) (*pppvalid)[i] = False;    }    }
     if (print)
     {    out << "Pairs:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove.nonempty( ) && remove[i] ) continue;
               out << "\n[" << i << "] ";
               ppp[i].Print( out, L );
               if ( color[i] == 1 ) out << " [NEAR EDGE OF NHOOD]";
               if ( color[i] == 2 ) out << " [OTHER PART OF GENOME]";
               if ( color[i] == 3 ) out << " [INVALID]";
               out << "\n";    }
          out << "\nPair locations:\n\n";
          Sort(pair_places);
          for ( int i = 0; i < pair_places.isize( ); i++ )
          {    const pair_placement& p = pair_places[i];
               if ( p.pair_id >= 0 && remove.nonempty( ) && remove[p.pair_id] ) 
                    continue;
               if ( p.pair_id >= 0 ) 
               {    out << p;
                    const pp_pair& pp = ppp[p.pair_id];
                    if ( IsClosed( pp, L ) ) out << " [CLOSED]";
                    out << "\n";    }
               else if ( p.pair_id == -1 )
               {    out << "GAP " << p.gid << "." << p.start1 << "-" << p.stop2
                         << " [" << p.Length( ) << "]\n";    }
               else
               {    out << "gap " << p.gid << "." << p.start1 << "-" << p.stop2
                         << " [" << p.Length( ) << "]\n";    }    }    }
     
     // Cache results.

     last_places = pair_places;
     last_color = color;    }

class read_extension {

     public:

     read_extension( ) { }
     read_extension( int pair_id, int edge_id, Bool sloppy )
          : pair_id(pair_id), edge_id(edge_id), sloppy(sloppy) { }

     int pair_id;
     int edge_id;
     Bool sloppy;

     friend Bool operator==( const read_extension& e1, const read_extension& e2 )
     {    return e1.pair_id == e2.pair_id && e1.edge_id == e2.edge_id
               && e1.sloppy == e2.sloppy;    }

     friend Bool operator<( const read_extension& e1, const read_extension& e2 )
     {    if ( e1.pair_id < e2.pair_id ) return True;
          if ( e1.pair_id > e2.pair_id ) return False;
          if ( e1.edge_id < e2.edge_id ) return True;
          if ( e1.edge_id > e2.edge_id ) return False;
          if ( e1.sloppy < e2.sloppy ) return True;
          return False;    }

};

void ShowExtensions( const vec<pp_pair>& ppp0, const vec<int>& L, const double dmult,
     const HyperKmerPath& h, const KmerBaseBroker& kbb, const vecbasevector& genome, 
     const int NHOOD_RADIUS, const int NHOOD_RADIUS_INTERNAL, 
     const serfvec<placement>& locsv, const vec<int>& genome_path_lengths )
{    vec<pp_pair> ppp(ppp0);
     vec< vec<int> > left_ext_total( ppp.size( ) ), right_ext_total( ppp.size( ) );
     cout << "\nEXTENSIONS:\n";
     for ( int pass = 1; pass <= 50; pass++ )
     {    cout << "\npass = " << pass << "\n\n";
          int N = ppp.size( );
          vec<read_extension> left_ext, right_ext;
          for ( int i = 0; i < N; i++ )
          {    for ( int j = 0; j < N; j++ )
               {    if ( i == j ) continue;
                    const pp_pair &p1 = ppp[i], &p2 = ppp[j];
                    static vec<int> o1, o2;
                    GetOverlaps( p1.Left( ), p2.Left( ), o1 );
                    if ( o1.empty( ) ) continue;
                    GetOverlaps( p1.Right( ), p2.Right( ), o2 );
                    if ( o2.empty( ) ) continue;
                    for ( int i1 = 0; i1 < o1.isize( ); i1++ )
                    {    for ( int i2 = 0; i2 < o2.isize( ); i2++ )
                         {    if ( o1[i1] + p2.LeftSize( ) <= p1.LeftSize( )
                                   && o2[i2] + p2.RightSize( ) <= p1.RightSize( ) )
                              {    continue;    }
                              double g1 = p1.Gap( ), g2 = p2.Gap( );
                              for ( int u = p1.LeftSize( ) - o1[i1]; 
                                   u < p2.LeftSize( ); u++ )
                              {    g1 -= L[ p2.Left(u) ];    }
                              for ( int u = 0; u < -o2[i2]; u++ )
                                   g1 -= L[ p2.Right(u) ];
                              for ( int u = o1[i1] + p2.LeftSize( ); 
                                   u < p1.LeftSize( ); u++ )
                              {    g2 -= L[ p1.Left(u) ];    }
                              for ( int u = 0; u < o2[i2]; u++ )
                                   g2 -= L[ p1.Right(u) ];
                              double d1 = p1.Dev( ), d2 = p2.Dev( );
                              double g, d;
                              if ( !CombineMeasurements( 
                                   g1, g2, d1, d2, dmult, g, d ) ) 
                              {    continue;    }
                              if ( o1[i1] + p2.LeftSize( ) > p1.LeftSize( ) )
                              {    left_ext.push_back( read_extension( i, 
                                        p2.Left( p1.LeftSize( ) - o1[i1] ),
                                        !o1.solo( ) ) );    }
                              if ( o2[i2] + p2.RightSize( ) > p1.RightSize( ) )
                              {    right_ext.push_back( read_extension( i, 
                                        p2.Right( p1.RightSize( ) - o2[i2] ),
                                        !o2.solo( ) ) );
                                              }    }    }    }    }
          UniqueSort(left_ext), UniqueSort(right_ext);
          for ( int i = 0; i < left_ext.isize( ); i++ )
          {    int j;
               for ( j = i + 1; j < left_ext.isize( ); j++ )
                    if ( left_ext[j].pair_id != left_ext[i].pair_id ) break;
               if ( left_ext[i].edge_id == left_ext[j-1].edge_id
                    && !left_ext[i].sloppy )
               {    int id = left_ext[i].pair_id, n = left_ext[i].edge_id;
                    cout << "[" << id << "] extended on left by "
                         << BaseAlpha(n) << "\n";    
                    ppp[id].LeftMutable( ).push_back(n);
                    ppp[id].SetGap( ppp[id].Gap( ) - L[n] );
                    left_ext_total[id].push_back(n);    }
               i = j - 1;    }
          for ( int i = 0; i < right_ext.isize( ); i++ )
          {    int j;
               for ( j = i + 1; j < right_ext.isize( ); j++ )
                    if ( right_ext[j].pair_id != right_ext[i].pair_id ) break;
               if ( right_ext[i].edge_id == right_ext[j-1].edge_id
                    && !right_ext[i].sloppy )
               {    int id = right_ext[i].pair_id, n = right_ext[i].edge_id;
                    cout << "[" << id << "] extended on right by "
                         << BaseAlpha(n) << "\n";    
                    ppp[id].RightMutable( ).push_back(n);
                    right_ext_total[id].push_back(n);    }
               i = j - 1;    }    }
     vec<int> color0, color;
     ValidatePairs( ppp0, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
          locsv, genome_path_lengths, color0 );
     ValidatePairs( ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
          locsv, genome_path_lengths, color );
     cout << "\nTOTAL EXTENSIONS:\n";
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( left_ext_total[i].empty( ) && right_ext_total[i].empty( ) ) continue;
          cout << "\n";
          if ( left_ext_total[i].nonempty( ) )
          {    cout << "[" << i << "] extended on left by ";
               for ( int j = 0; j < left_ext_total[i].isize( ); j++ )
               {    if ( j > 0 ) cout << ".";
                    cout << BaseAlpha( left_ext_total[i][j] );    }
               cout << "\n";    }
          if ( right_ext_total[i].nonempty( ) )
          {    cout << "[" << i << "] extended on right by ";
               for ( int j = 0; j < right_ext_total[i].isize( ); j++ )
               {    if ( j > 0 ) cout << ".";
                    cout << BaseAlpha( right_ext_total[i][j] );    }
               cout << "\n";    }    
          if ( color0[i] != color[i] )
          {    cout << GraphColor( color0[i] ) << " --> " << GraphColor( color[i] )
                    << "\n";    }    }    }

void MakePairJoinData( const vec<pp_pair>& ppp, const vec<int>& L,
     const double dmult, vec<pair_join>& joins )
{    joins.clear( );
     int N = ppp.size( );
     for ( int i = 0; i < N; i++ )
     {    for ( int j = 0; j < N; j++ )
          {    if ( i == j ) continue;
               const pp_pair &p1 = ppp[i], &p2 = ppp[j];
               static vec<int> o1, o2;
               GetOverlaps( p1.Left( ), p2.Left( ), o1 );
               if ( o1.empty( ) ) continue;
               GetOverlaps( p1.Right( ), p2.Right( ), o2 );
               if ( o2.empty( ) ) continue;
               for ( int i1 = 0; i1 < o1.isize( ); i1++ )
               {    for ( int i2 = 0; i2 < o2.isize( ); i2++ )
                    {    if ( o1[i1] + p2.LeftSize( ) <= p1.LeftSize( )
                              && o2[i2] + p2.RightSize( ) <= p1.RightSize( ) )
                         {    continue;    }
                         double g1 = p1.Gap( ), g2 = p2.Gap( );
                         for ( int u = p1.LeftSize( ) - o1[i1]; 
                              u < p2.LeftSize( ); u++ )
                         {    g1 -= L[ p2.Left(u) ];    }
                         for ( int u = 0; u < -o2[i2]; u++ )
                              g1 -= L[ p2.Right(u) ];
                         for ( int u = o1[i1] + p2.LeftSize( ); 
                              u < p1.LeftSize( ); u++ )
                         {    g2 -= L[ p1.Left(u) ];    }
                         for ( int u = 0; u < o2[i2]; u++ )
                              g2 -= L[ p1.Right(u) ];
                         double d1 = p1.Dev( ), d2 = p2.Dev( );
                         double g, d;
                         if ( !CombineMeasurements( g1, g2, d1, d2, dmult, g, d ) ) 
                              continue;
                         static pp_read y1, y2;
                         JoinReads( p1.Left( ), p2.Left( ), o1[i1], y1 );
                         JoinReads( p1.Right( ), p2.Right( ), o2[i2], y2 );
                         joins.push_back( pair_join( i, j,
                              pp_pair( y1, y2, g, d ) ) );    }    }    }    }
     /*
     for ( int i = 0; i < N; i++ )
     {    const pp_pair &p1 = ppp[i];
          if ( p1.Dev( ) != 0 || p1.Left( ) != p1.Right( ) ) continue;
          for ( int j = 0; j < N; j++ )
          {    if ( i == j ) continue;
               const pp_pair &p2 = ppp[j];
               static vec<int> o;
               GetOverlaps( p1.Right( ), p2.Left( ), o );
               double g2 = p2.Gap( ), d2 = p2.Dev( );
               for ( int u = 0; u < o.isize( ); u++ )
               {    if ( o[u] + p2.LeftSize( ) < p1.RightSize( ) ) continue;
                    if ( OverlapLength( p1.Right( ), p2.Left( ), o[u], L ) < 100 )
                         continue;
                    if ( o[u] + p2.LeftSize( ) == p1.RightSize( ) )
                    {    double right2 = g2 - dmult * d2;
                         for ( int r = 0; r < p2.RightSize( ); r++ )
                              right2 += L[ p2.Right(r) ];
                         if ( right2 <= 0 ) continue;    }
                    pp_read left;
                    JoinReads( p1.Right( ), p2.Left( ), o[u], left );
                    joins.push_back( pair_join( i, j, 
                         pp_pair( left, p2.Right( ), g2, d2 ) ) );    }    }    }
     for ( int j = 0; j < N; j++ )
     {    const pp_pair& p2 = ppp[j];
          if ( p2.Dev( ) != 0 || p2.Left( ) != p2.Right( ) ) continue;
          for ( int i = 0; i < N; i++ )
          {    if ( i == j ) continue;
               const pp_pair &p1 = ppp[i];
               static vec<int> o;
               GetOverlaps( p2.Left( ), p1.Right( ), o );
               double g1 = p1.Gap( ), d1 = p1.Dev( );
               for ( int u = 0; u < o.isize( ); u++ )
               {    if ( o[u] > 0 ) continue;
                    if ( OverlapLength( p2.Left( ), p1.Right( ), o[u], L ) < 100 )
                         continue;
                    if ( o[u] == 0 )
                    {    double left2 = -g1 + dmult * d1;
                         for ( int r = 0; r < p1.LeftSize( ); r++ )
                              left2 -= L[ p1.Left(r) ];
                         if ( left2 >= 0 ) continue;    }
                    pp_read right;
                    JoinReads( p2.Left( ), p1.Right( ), o[u], right );
                    joins.push_back( pair_join( j, i, pp_pair( 
                         p1.Left( ), right, g1, d1 ) ) );    }    }    }    
     */
     }

void MakeGraphFromPairs( const vec<pp_pair>& ppp, const vec<int>& L,
     const double dmult, digraph& PG )
{    int N = ppp.size( );
     vec< vec<vrtx_t> > from(N), to(N);
     vec<pair_join> joins;
     MakePairJoinData( ppp, L, dmult, joins );
     for ( int i = 0; i < joins.isize( ); i++ )
     {    from[ joins[i].j1 ].push_back( joins[i].j2 );
          to[ joins[i].j2 ].push_back( joins[i].j1 );    }
     for ( int v = 0; v < N; v++ )
     {    UniqueSort( from[v] );
          UniqueSort( to[v] );    }
     PG.Initialize( from, to );    }

void OutputPairsGraph( ostream& out, digraph& PG )
{    PG.DOT(out);    }

void OutputTruthLabeledPairsGraph( ostream& out, digraph& PG, 
     const vec<pp_pair>& ppp, const vec<int>& L, const double dmult,
     const HyperKmerPath& h, const KmerBaseBroker& kbb, const vecbasevector& genome, 
     const int NHOOD_RADIUS, const int NHOOD_RADIUS_INTERNAL,
     const serfvec<placement>& locsv, const vec<int>& genome_path_lengths )
{    vec<pair_join> joins;
     MakePairJoinData( ppp, L, dmult, joins );
     ShowExtensions( ppp, L, dmult, h, kbb, genome, NHOOD_RADIUS, 
          NHOOD_RADIUS_INTERNAL, locsv, genome_path_lengths );
     vec<pp_pair> pppe(ppp);
     for ( int i = 0; i < joins.isize( ); i++ )
          pppe.push_back( joins[i].p );
     vec<int> color;
     ValidatePairs( pppe, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
          locsv, genome_path_lengths, color );
     vec<String> vertex_colors( PG.N( ) );
     vec< vec<String> > edge_colors( PG.N( ) );
     for ( int v = 0; v < PG.N( ); v++ )
          edge_colors[v].resize( PG.From(v).size( ) );
     for ( int i = 0; i < ppp.isize( ); i++ )
          vertex_colors[i] = GraphColor( color[i] );
     vec< vec<int> > edge_colorsi( PG.N( ) );
     for ( int v = 0; v < PG.N( ); v++ )
          edge_colorsi[v].resize( PG.From(v).size( ), 3 );
     for ( int i = ppp.isize( ); i < pppe.isize( ); i++ )
     {    int j1 = joins[ i - ppp.isize( ) ].j1;
          int j2 = joins[ i - ppp.isize( ) ].j2;
          int p2 = BinPosition( PG.From(j1), j2 );
          edge_colorsi[j1][p2] = Min( edge_colorsi[j1][p2], color[i] );    }
     for ( int v = 0; v < PG.N( ); v++ )
     {    for ( int j = 0; j < PG.From(v).isize( ); j++ )
               edge_colors[v][j] = GraphColor( edge_colorsi[v][j] );    }
     PG.DOT( out, vertex_colors, edge_colors );    }
